﻿<#  
  +----------------------------------------------------------------------+
  | Program      : Create-NoUACTaskShortcut.ps1                          |
  | Programmer   : RetiredGeek (WSL) aka: ComputerMentor                 |
  | Created      : 24 Nov 2014                                           |
  | Updated      : 27 Nov 2014                                           |
  | Description  : Prompts the user with a dialog for the data required  |
  |                to create a Shortcut and Scheduled task for a program |
  |                that normally requires Elevated Privileges prompt.    |
  | Requirements : Windows 8.1 - PowerShell 4.0                          |
  | Last Updated :                                                       |
  | Current Vers : 1.3                                                   |
  |   11/26/2014 : Added check for Windows & PowerShell minimum reqs.    |
  |   11/27/2014 : Added suggestions by Cliff.H on WSL, Thanks Cliff     |
  +----------------------------------------------------------------------+
#>

Function Create-NoUACTask {
<#+-----------------------------------------------------------------+
  | Function     :    Create-NoUACTask                              |
  | Description  : Create Task to get around UAC Prompt             |
  | Programmer   : RetiredGeek (WSL) aka: ComputerMentor            |
  | Created      : 21 Nov 2014                                      |
  | Last Updated :                                                  |
  | Current Vers : 1.0                                              |
  +-----------------------------------------------------------------+
#>
param (
        [Parameter(Mandatory=$true)]
          [String]$TaskName,
        [Parameter(Mandatory=$true)]
          [String]$TaskCommand,
        [Parameter(Mandatory=$false)]
          [String]$TaskArg,
        [Parameter(Mandatory=$false)]
          [String]$TaskDesc="No Description Supplied"
      )

$TaskAction = New-ScheduledTaskAction -Execute `"$TaskCommand`" `
                                      -Argument `"$TaskArg`"

Register-ScheduledTask -Action $TaskAction  `
                       -TaskName "$TaskName" `
                       -User "$env:USERDOMAIN\$env:username" `
                       -RunLevel Highest `
                       -Description "$TaskDesc"

}   #End Function Create-NoUACTask

Function Create-NoUACShortCut {
<#
  +-------------------------------------------------------------------------+
  | Programmer   : RetiredGeek (WSL) aka: ComputerMentor                    |
  | Created      : 24 Nov 2014                                              |
  | Last Updated : 28 Nov 2014                                              |
  | Current Vers : 1.3                                                      |
  |                                                                         |
  |Where:                                                                   |
  |  $TaskName             - is the Name of the Task/Shortcut to create     |
  |  $TaskCommand          - is the Command Executed by the Task            |
  |  $DestinationPath      - is the location to store the created shortcut  |
  | ----- Use only ONE of the Following ----                                |
  |  $ShortcutIcon         - is the path & number of the icon to use.       |
  |                             ex: "C:\MyIcon.ico,0"  - optional           |
  |  $UseTaskCommandIcon   - will select the first icon in the TaskCommand  |
  |                          file.                                          |
  +-------------------------------------------------------------------------+
#>
param ( [Parameter(Mandatory=$true)]
          [String]$TaskName,
        [Parameter(Mandatory=$false)]
          [String]$TaskCommand,
        [Parameter(Mandatory=$false)]
          [String]$DestinationPath="$env:UserProfile\Desktop", 
        [Parameter(Mandatory=$false)]
          [String]$ShortcutIcon="NO",
          [switch]$UseTaskCommandIcon
      )

$WshShell = New-Object -comObject WScript.Shell
$Shortcut = $WshShell.CreateShortcut("$DestinationPath\$TaskName.lnk")
$Shortcut.TargetPath = "C:\Windows\System32\schtasks.exe" 
$Shortcut.Arguments = "`/run `/TN " + "`"" + $TaskName +"`""
$Shortcut.WindowStyle = 7

If ($ShortcutIcon -ne 'NO') {
  $Shortcut.IconLocation = $ShortcutIcon
}
Else {
    If ($UseTaskCommandIcon.IsPresent) {
      $Shortcut.IconLocation = $TaskCommand
    }
}
$Shortcut.Save()

}    #End Function Create-NoUACShortCut

# -------------------   Main ------------------------
Add-Type -AssemblyName System.Windows.Forms  #PS V3+
Add-Type -AssemblyName System.Drawing

$ErrorActionPreference = 'Stop'

$MsgBox = [Windows.Forms.MessageBox]
$Buttons = [Windows.Forms.MessageBoxButtons]
$MBIcons = [Windows.Forms.MessageBoxIcon]

<#  **** Check Operating Requirements   ***
    **** Windows 8.1                    ***
    **** PowerShell 4.0 as Admin        ***
#>
$CurOS = Get-CimInstance Win32_OperatingSystem
$FullVersion = $CurOS.Version
$OSVerItems = $CurOs.Version.split('.')
$OSVer = $OSVerItems[0]*10 + $OSVerItems[1]
$CurPSVer = $psversiontable.psversion.Major
If (-NOT ([Security.Principal.WindowsPrincipal] `
          [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole(`
          [Security.Principal.WindowsBuiltInRole] "Administrator")) 
          {$PSPriv = "User"} Else {$PSPriv = "Administrator"} 

If ($OSVer -lt 63 -or $CurPSVer -lt 4 -or $PSPriv -eq "User") {
#*** Now strings incorporate ALL spaces thus no identing!
  $Message = `
@("This program requires Windows 8.1 {6.3.9600} and
and PowerShell 4.0 as Administrator to run!

You are running:
Windows:`t`t$FullVersion
PowerShell:`t$CurPSVer
Running as:`t$($PSPriv)")

  [void]$MsgBox::Show("$Message","System Exit", `
                 $Buttons::OK, $MBIcons::Stop)
  Exit
}
# *** End Check Operating Requirements ***

# Font styles are: Regular, Bold, Italic, Underline, Strikeout
$Font = New-Object System.Drawing.Font("Tahoma",12,`
        [System.Drawing.FontStyle]::Regular)
$LBFont = New-Object System.Drawing.Font("Courier New",11,`
          [System.Drawing.FontStyle]::Regular)

#Form Object Location/Width/Height sizes
$LNSpacing = 35
$TNLoc = 10
$TCLoc = $TNLoc+$LNSpacing
$OALoc = $TCLoc+$LNSpacing
$TALoc = $OALoc+$LNSpacing
$TDLoc = $TALoc+$LNSpacing
$DPLoc = $TDLoc+$LNSpacing
$ICLoc = $DPLoc+$LNSpacing
$UTLoc = $ICLoc+$LNSpacing
$BtnLoc = 340
$BtnWidth = 85
$LblWidth = 210
$TBWidth = 590

$OKButton = New-Object System.Windows.Forms.Button
$OKButton.Location = New-Object System.Drawing.Size(10,$BtnLoc)
$OKButton.Size = New-Object System.Drawing.Size($BtnWidth,30)
$OKButton.Text = "OK"
$OKButton.backcolor = "Green"
$OKButton.Font = $Font
$OKButton.DialogResult = [System.Windows.Forms.DialogResult]::OK

$CancelButton = New-Object System.Windows.Forms.Button
$CancelButton.Location = New-Object System.Drawing.Size(105,$BtnLoc)
$CancelButton.Size = New-Object System.Drawing.Size($BtnWidth,30)
$CancelButton.Text = "Cancel"
$CancelButton.BackColor = "Red"
$CancelButton.Font = $Font
$CancelButton.DialogResult = [System.Windows.Forms.DialogResult]::Cancel

$objForm = New-Object System.Windows.Forms.Form 
$objForm.Text = "Avoid UAC ShortCut/SchTask Creation:"
$objForm.Size = New-Object System.Drawing.Size(850,420) 
$objForm.StartPosition = "CenterScreen"
$objForm.AcceptButton = $OKButton
$objForm.CancelButton = $CancelButton

$objLabelTN = New-Object System.Windows.Forms.Label
$objLabelTN.Location = New-Object System.Drawing.Size(10,$TNLoc) 
$objLabelTN.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelTN.Text = "Task/Shortcut Name:"
$objLabelTN.Font = $font

$objLabelCmd = New-Object System.Windows.Forms.Label
$objLabelCmd.Location = New-Object System.Drawing.Size(10,$TCLoc) 
$objLabelCmd.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelCmd.Text = "Task/Shortcut Cmd:"
$objLabelCmd.Font = $font

$objLabelOptArgs = New-Object System.Windows.Forms.Label
$objLabelOptArgs.Location = New-Object System.Drawing.Size(220,$OALoc) 
$objLabelOptArgs.Size = New-Object System.Drawing.Size(300,25) 
$objLabelOptArgs.Text = "----- Optional Arguments -----"
$objLabelOptArgs.Font = $font

$objLabelArg = New-Object System.Windows.Forms.Label
$objLabelArg.Location = New-Object System.Drawing.Size(10,$TALoc) 
$objLabelArg.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelArg.Text = "Task/Shortcut Args:"
$objLabelArg.Font = $font

$objLabelTD = New-Object System.Windows.Forms.Label
$objLabelTD.Location = New-Object System.Drawing.Size(10,$TDLoc) 
$objLabelTD.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelTD.Text = "Task Description:"
$objLabelTD.Font = $font

$objLabelDP = New-Object System.Windows.Forms.Label
$objLabelDP.Location = New-Object System.Drawing.Size(10,$DPLoc)
$objLabelDP.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelDP.Text = "Shortcut Location:"
$objLabelDP.Font = $font

$objLabelIC = New-Object System.Windows.Forms.Label
$objLabelIC.Location = New-Object System.Drawing.Size(10,$ICLoc) 
$objLabelIC.Size = New-Object System.Drawing.Size($LblWidth,25) 
$objLabelIC.Text = "Icon Location:"
$objLabelIC.Font = $font

$objLabelUT = New-Object System.Windows.Forms.Label
$objLabelUT.Location = New-Object System.Drawing.Size(10,$UTLoc) 
$objLabelUT.Size = New-Object System.Drawing.Size(($LblWidth+40),25) 
$objLabelUT.Text = "Use Task Command Icon:"
$objLabelUT.Font = $font

$objTBTN = New-Object System.Windows.Forms.TextBox 
$objTBTN.Location = New-Object System.Drawing.Size(220,$TNLoc) 
$objTBTN.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBTN.Font = $lbfont

$objTBCmd = New-Object System.Windows.Forms.TextBox 
$objTBCmd.Location = New-Object System.Drawing.Size(220,$TCLoc) 
$objTBCmd.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBCmd.Font = $lbfont

$objTBArg = New-Object System.Windows.Forms.TextBox 
$objTBArg.Location = New-Object System.Drawing.Size(220,$TALoc) 
$objTBArg.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBArg.Font = $lbfont

$objTBTD = New-Object System.Windows.Forms.TextBox 
$objTBTD.Location = New-Object System.Drawing.Size(220,$TDLoc) 
$objTBTD.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBTD.Font = $lbfont

$objTBDP = New-Object System.Windows.Forms.TextBox 
$objTBDP.Location = New-Object System.Drawing.Size(220,$DPLoc) 
$objTBDP.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBDP.Font = $lbfont

$objTBIC = New-Object System.Windows.Forms.TextBox 
$objTBIC.Location = New-Object System.Drawing.Size(220,$ICLoc) 
$objTBIC.Size = New-Object System.Drawing.Size($TBWidth,25) 
$objTBIC.Font = $lbfont

$objCBUseTC = New-Object System.Windows.Forms.CheckBox 
$objCBUseTC.Location = New-Object System.Drawing.Size(260,$UTLoc) 
$objCBUseTC.Size = New-Object System.Drawing.Size(25,25) 
$objCBUseTC.Font = $lbfont

$objForm.Controls.AddRange( `
     @($OKButton,$CancelButton, `
       $objLabelTN,$objLabelCmd,$objLabelArg,$objLabelOptArgs, `
              $objLabelTD,$objLabelDP,$objLabelIC,$objLabelUT, `
       $objTBTN,$objTBCmd,$objTBArg,$objTBTD,$objTBDP,$objTBIC,$objCBUseTC))
$objForm.Topmost = $True

$dialogResult = $objForm.ShowDialog()

if ($dialogResult -eq [System.Windows.Forms.DialogResult]::OK)
{
  
  $TaskName = $objTBTN.Text
  $TaskCommand = $objTBCmd.Text
  $TaskArg =$objTBArg.Text
  $TaskDesc = $objTBTD.Text
  $DestinationPath = $ObjTBDP.Text
  $ShortcutIcon = $objTBIC.Text
  $UseTaskCommandIcon = $objcbusetc.checked

  $FoundError = $false

  If ($TaskName -eq "") {
    $Message = "The Task/Shortcut Name parameter value is BLANK:"
    $MsgBox::Show("$Message","ERROR: Invalid Task/Shortcut Name", `
                   $Buttons::OK, $MBIcons::Stop)
    $FoundError = $true
  }

  If ($TaskCommand -eq "") {
    $Message = "The Task Command parameter value is BLANK:"
    $MsgBox::Show("$Message","ERROR: Invalid Task Command", `
                   $Buttons::OK, $MBIcons::Stop)
    $FoundError = $true
  }
  Else {
      If (!(Test-Path -Path $TaskCommand)) {

        $Message = "The TaskCommand parameter value is not valid:"
        $MsgBox::Show("$Message `n $TaskCommand","ERROR: Invalid Task Command", `
                       $Buttons::OK, $MBIcons::Stop)
        $FoundError = $true
      }       # End If (!(Test-Path -Path $TaskCommand))
  }

  If ($DestinationPath -ne "") {
    If (!(Test-Path -Path $DestinationPath)) {
      $Message = "The Shortcut Location value is not valid:"
      $MsgBox::Show("$Message `n $DestinationPath","ERROR: Invalid Shortcut Location", `
                     $Buttons::OK, $MBIcons::Stop)
      $FoundError = $true
    }       # End If (!(Test-Path -Path $TaskCommand))
  }
  ELse {
    $DestinationPath = "$env:UserProfile\Desktop"
  }
  If ($FoundError) { EXIT }

#------ Error Checks PASSED Create Task & Shortcut ---

   If ($TaskDesc -eq "") {
     Create-NoUACTask -TaskCommand "$TaskCommand" `
                      -TaskArg $TaskArg `
                      -TaskName $TaskName 
   }
   Else {
      Create-NoUACTask -TaskCommand "$TaskCommand" `
                      -TaskArg $TaskArg `
                      -TaskName $TaskName `
                      -TaskDesc "$TaskDesc"
   }

   If ($UseTaskCommandIcon) {
     Create-NoUACShortCut -TaskName $TaskName `
                          -DestinationPath $DestinationPath `
                          -TaskCommand "$TaskCommand" `
                          -UseTaskCommandIcon 
   }
   Else {  
       If ($ShortcutIcon -ne "") {
         Create-NoUACShortCut -TaskName $TaskName `
                              -DestinationPath $DestinationPath `
                              -ShortcutIcon $ShortcutIcon
       }                   
       Else {
           Create-NoUACShortCut -TaskName $TaskName `
                           -DestinationPath $DestinationPath 
       }
   }
}   #End if ($dialogResult -eq [System.Windows.Forms.DialogResult]::OK)
Else
{
    $Message = "User selected CANCEL or clicked the close box."
    $MsgBox::Show("$Message","User Exit", `
                   $Buttons::OK, $MBIcons::Stop)

}

$objForm.dispose()

