﻿<#
.SYNOPSIS
   Produce information about the computer in a tabbed dialog.
   Similar to Piriform's Speccy or BelArc Advisor

.DESCRIPTION
   CMsPCInfo.ps1 provides information on your hardware; windows;
   video; storage; network; printers; drivers; programs; and
   selected registry/service settings. 

.PARAMETER ComputerName
    A single Computer or an array of computer names.
    The default is localhost ($env:COMPUTERNAME).

.OUTPUTS
    Optional output to File: Written to users default Documents Directory
    With button to write to a text file.


.NOTES
  Computer Mentors System Information Dialog V14.00
  Code adapted from:
           Powershell PC Info Script V1.0b
              Coded By:Trenton Ivey(kno)
                     hackyeah.com
  Incorporated Coding changes recommed by Cafed00d @ WSL

  Updated: 
     09/03/15 Single GenerateTab Function with
              proper argument passing syntax!
     09/10/15 Added new information & fixed format
              of some output
     09/14/15 Fixed Monitor reporting problem
     10/23/15 Added check for Administrator Priv.
     10/25/15 Expanded Storage Information
     10/28/15 Added physical drive information
     10/29/15 Get-Drives & Combine-Object Functs
     11/21/15 Added Win10 Telemetry Service
     12/15/15 Added Write to file option button
     12/19/15 Added Comment based help
     12/21/15 Reorged code to movee data collection from Main to
              the GenerateForm relative tab.
     12/30/15 Expanded the .Net information.
     01/21/16 Fixed UpTime for more than 24 hrs.
     02/08/16 Solved problem with Table display field width via
              a Function to calculate longest value in field.
     02/26/16 Added Partition information re Alignment and TRIM Status.
     03/05/16 Added status of OS Compacted.
     03/09/16 Sort Partitions in Disk Info and Indicate Boot Status.
     03/11/16 Added test for Min Win version to run. (8.1)
              Last Full Battery Tab info only shows on Battery Devices.
     03/27/16 Added Environment Variables tab and reordered tabs and
              moved tab code into Functions for ease of editing.

.EXAMPLE
    PS C:\>CMsPCInfo.ps1

#>

Function GenerateForm {

#---------------------- Form Objects     ----------------------------
$CMsForm        = New-Object -TypeName System.Windows.Forms.Form
$statusBar1     = New-Object -TypeName System.Windows.Forms.StatusBar
$lblMainTitle   = New-Object -TypeName System.Windows.Forms.Label
$btnExit        = New-Object -TypeName System.Windows.Forms.Button
$btnWriteFile   = New-Object -TypeName System.Windows.Forms.Button
$MainTabCtrl = New-Object -TypeName System.Windows.Forms.TabControl

#---------------------- End Form Objects -----------------------------

#----------------------------------------------------------------------
#                     Event Handler Script Blocks
#----------------------------------------------------------------------

$btnWriteFile_OnClick=
{

   $InfoFileName = 
     "$([environment]::getfolderpath("mydocuments"))\CMsPCInfo.txt"

   "Computer Mentors System Information - Version $($PGMVersion)`r`n`r`n" +
   "---     Hardware      ---    `r`n" + $HWInfo         +
   "---       Video       ---    `r`n" + $VideoInfo      +
   "---      Storage      ---`r`n`r`n" + $StorageInfo    +
   "---      Network      ---    `r`n" + $NetworkInfo    +
   "---     Printers      ---    `r`n" + $PrinterInfo    +
   "---      Windows      ---    `r`n" + $WindowsInfo    +
   "---    Environment    ---    `r`n" + $EnvironInfo    +
   "---      Drivers      ---`r`n`r`n" + $DriversInfo    +
   "--- Installed Program ---    `r`n" + $InstPgmsInfo   +
   "--- Registry/Services ---    `r`n" + $ServicesInfo    `
     >  "$InfoFileName"

  If (Test-Path variable:BatteryInfo) {
   "--- Battery Status    ---    `r`n" + $BatteryInfo `
     >> "$InfoFileName"
  }

   $StatusBar1.text = "Data written to: $($InfoFileName)"
   $btnWriteFile.Visible = $False

} #End $btnWriteFile_OnClick

$btnExit_OnClick=
{
 $CMsForm.Close()
}

$OnLoadForm_StateCorrection=
{
  #  Correct the initial state of the form to prevent the
  #  .Net maximized form issue
$CMsForm.WindowState = $IntFrmWinState
}

#--------------------- End Event Handler Script Blocks ----------------------

#------------------------------- Main Form ----------------------------------

#--- Form Drawing References ---

$Form_Width       = 1050
$Form_Height      = 600
$MainTitleHeight  =  35
$StatusBar_Height =  25
$ExitBtn_Height   =  30
$tabWidth         = $Form_Width-20
$tabHeight        = $Form_Height - $StatusBar_Height + 
                                   $MainTitleHeight  + 20
$tabSize          = New-Object -TypeName System.Drawing.Size(
                                              $tabWidth,$tabHeight)
$tabLocation      = New-Object -TypeName System.Drawing.point(4,22)
$tboxWidth        = $tabWidth-20
$tboxHeight       = 430
$tboxFont         = New-Object -TypeName System.Drawing.Font(
                                             "Courier New",20,1,0)
$tboxSize         = New-Object -TypeName System.Drawing.Size(
                                              $tboxWidth,$tboxHeight)
$tboxLocation     = New-Object -TypeName System.Drawing.Point(10,10)
$tboxForeColor    = [Drawing.Color]::Yellow
$tboxBackColor    = [Drawing.Color]::Blue

$System_Windows_Forms_Padding = 
                    New-Object -TypeName System.Windows.Forms.Padding
$System_Windows_Forms_Padding.All = 0

$CMsForm.ControlBox    = $False
$CMsForm.ClientSize    = New-Object -TypeName `
                      System.Drawing.Size($($Form_Width),$($Form_Height))
$CMsForm.Name          = "CMsForm"
$CMsForm.BackColor     = [Drawing.Color]::DarkGray
$CMsForm.ForeColor     = [Drawing.Color]::White
$CMsForm.StartPosition =
                  [System.Windows.Forms.FormStartPosition]::CenterScreen
$CMsForm.Text          =
          "Computer Mentor's Computer Information Version $PGMVersion " +
          "Running with $(Get-AdminStatus) privledges"
$CMsForm.FormBorderStyle = 
                  [System.Windows.Forms.FormBorderStyle]::Fixed3D

# --- Status Bar ---

$statusBar1.Name     = "statusBar1"
$statusBar1.Location = New-Object -TypeName `
                System.Drawing.Point($($Form_Height-$StatusBar_Height),0)
$statusBar1.Size     = New-Object -TypeName `
              System.Drawing.Size($($Form_Width-40),$($StatusBar_Height))
$statusBar1.Font     = New-Object -TypeName `
                System.Drawing.Font("Verdana",14,1,0)
$statusBar1.TabIndex = 8
$statusBar1.Text     = "Ready"

$CMsForm.Controls.Add($statusBar1)

#------------- Main Title -------------

$lblMainTitle.Name      = "lblMainTitle"
$lblMainTitle.Location  = New-Object -TypeName System.Drawing.Point(20,15)
$lblMainTitle.Size      = New-Object -TypeName `
                                 System.Drawing.Size($($Form_Width-40),35)
$lblMainTitle.TextAlign = 'MiddleCenter'
$lblMainTitle.TabIndex  = 7
$lblMainTitle.Text      = "Computer Information"
$lblMainTitle.Font      = New-Object -TypeName `
                                 System.Drawing.Font("Verdana",26,1,0)
$lblMainTitle.BackColor = [Drawing.Color]::DarkGray
$lblMainTitle.ForeColor = [Drawing.Color]::Blue

$CMsForm.Controls.Add($lblMainTitle)

#------------  WriteFile Button ----------------

$btnWriteFile.Name      = "btnWriteFile"
$btnWriteFile.Location  = New-Object -TypeName `
      System.Drawing.Point(10,$($Form_Height - $StatusBar_Height - 
	                            $ExitBtn_Height))
$btnWriteFile.Size      = New-Object -TypeName System.Drawing.Size(185,35)
$btnWriteFile.TabIndex  = 6
$btnWriteFile.Text      = "Write to File"
$btnWriteFile.BackColor = [Drawing.Color]::Green
$btnWriteFile.ForeColor = [Drawing.Color]::White
$btnWriteFile.Font      = $tboxFont

$btnWriteFile.add_Click($btnWriteFile_OnClick)
$CMsForm.Controls.Add($btnWriteFile)

#------------  Exit Button ----------------

$btnExit.Name      = "btnExit"
$btnExit.Location  = New-Object -TypeName `
          System.Drawing.Point($($Form_Width-80),$($Form_Height - 
		                       $StatusBar_Height - $ExitBtn_Height))
$btnExit.Size      = New-Object -TypeName System.Drawing.Size(75,35)
$btnExit.TabIndex  = 6
$btnExit.Text      = "Exit"
$btnExit.BackColor = [Drawing.Color]::Red
$btnExit.ForeColor = [Drawing.Color]::White
$btnExit.Font      = $tboxFont

$btnExit.add_Click($btnExit_OnClick)
$CMsForm.Controls.Add($btnExit)

#----------- Main Tab Control ---------------------

$MainTabCtrl.Name      = "MainTabControl"
$MainTabCtrl.Location  = New-Object -TypeName System.Drawing.Point(10,60)
$MainTabCtrl.Size      = New-Object -TypeName `
                         System.Drawing.Size($($Form_Width-20), 
						        $($Form_Height - $StatusBar_Height - 90))
$MainTabCtrl.ShowToolTips  = $True
$MainTabCtrl.TabIndex      = 4
$MainTabCtrl.Font          =  New-Object -TypeName `
                              System.Drawing.Font("Tahoma",20,2,0)

$CMsForm.Controls.Add($MainTabCtrl)

  $ComputerSysObj =
      get-WMIObject -computer $compname -Class Win32_ComputerSystem


$HWInfo       = HardwareTab
$VideoInfo    = VideoTab
$StorageInfo  = StorageTab
$NetworkInfo  = NetworkTab
$PrinterInfo  = PrinterTab
$WindowsInfo  = WindowsTab
$EnvironInfo  = EnvironmentTab
$DriversInfo  = DriversTab
$InstPgmsInfo = ProgramsTab
$ServicesInfo = RegistryServicesTab
$BatteryInfo  = BatteryTab


#--------------------------   End Form Code  ----------------------------

  #Save the initial state of the form
  $IntFrmWinState = $CMsForm.WindowState
  #Init the OnLoad event to correct the initial state of the form
  $CMsForm.add_Load($OnLoadForm_StateCorrection)

  #Show the Form
  $CMsForm.ShowDialog() | Out-Null

} #---------------------- End Function GenerateForm ---------------------

Function HardwareTab {

  $enclosureNames = (
    "unknown",  # 0
    "Other" ,
    "Unknown" ,
    "Desktop" ,
    "Low Profile Desktop" ,
    "Pizza Box" ,  #5
    "Mini Tower" ,
    "Tower" ,
    "Portable" ,
    "Laptop" ,
    "Notebook" , #10
    "Hand Held" ,
    "Docking Station" ,
    "All-in-One" ,
    "Sub Notebook" ,
    "Space Saving" ,  #15
    "Lunch Box" ,
    "Main System Chassis",
    "Expansion Chassis",
    "Sub-Chassis",
    "Bus Expansion Chassis", #20
    "Peripheral Chassis" ,
    "Storage Chassis" ,
    "Rack Mount Chassis" ,
    "Sealed-Case PC" #24
  )

  Add-Row -ItemName 'Manufacturer' -ItemValue $ComputerSysObj.Manufacturer
  Add-Row -ItemName 'Model'        -ItemValue $ComputerSysObj.Model
  Add-Row -ItemName 'System Type'  -ItemValue $ComputerSysObj.SystemType

  $enclosure = get-wmiobject -computer $compname `
                             -Class Win32_SystemEnclosure
  $ChassisNo = $enclosure.ChassisTypes[0]
  $ChassisName =  if ($ChassisNo -ge $enclosureNames.length) {
                         "Currently Unassigned"
                  }
                  else {
                         $enclosureNames[$ChassisNo]
                  }

  Add-Row  -ItemName 'Enclosure' -ItemValue $ChassisName

  Add-Row  -ItemName ''

  $BIOSObject = get-WMIObject -computer $compname -Class Win32_BIOS
  Add-Row -ItemName 'BIOS Serial No'  -ItemValue $BIOSObject.SerialNumber
  Add-Row -ItemName '     Name'       -ItemValue $BIOSObject.Name
  Add-Row -ItemName '     Version'    -ItemValue `
                                         $BIOSObject.SMBIOSBIOSVersion
  Add-Row -ItemName '     Date'       -ItemValue `
                                   $BIOSObject.ReleaseDate.Substring(0,8)
  Add-Row -ItemName '     Manufacturer' -ItemValue `
                                             $BIOSObject.Manufacturer

  $BV = Get-WmiObject -Class Win32_BIOS -Namespace "root\CIMV2" |
        Select-Object -Property BIOSVersion

  If ($BV.BIOSVersion.count -eq 3) {
    $Separator = "-"
    $BIOSManuf = $BV.BIOSVersion[2].split($Separator)  #BIOS Manufacturer
  Add-Row -ItemName '     Creator'      -ItemValue $BIOSManuf[0]
  }

  Add-Row -ItemName ''

  If ($AdminPriv) {
    $ErrorActionPreference = 'SilentlyContinue'
    $UEFISB = Confirm-SecureBootUEFI
    $ErrorActionPreference = 'Continue'

    If (Test-Path variable:UEFISB ){
      If ($UEFISB) {$UEFIStatus="On"} Else {$UEFIStatus="OFF"}
      Add-Row -ItemName 'Secure Boot UEFI'  -ItemValue $UEFIStatus
    }
    Else {
         Add-Row -ItemName 'Secure Boot UEFI' -ItemValue "Not Supported"
    }
  }   #End If ($AdminPriv)

  ELse {
         Add-Row -ItemName 'Secure Boot UEFI' `
                 -ItemValue "Requires Admin Access to verify!"
  }

  Add-Row -ItemName ''

  $CPU_Object = get-WMIObject -computer $compname -Class Win32_Processor

  Add-Row -ItemName 'Processor Name'   -ItemValue $CPU_Object.Name
  Add-Row -ItemName '  Info'           -ItemValue $CPU_Object.Caption
  Add-Row -ItemName '  Maker'          -ItemValue `
                                            $CPU_Object.Manufacturer
  Add-Row -ItemName '  ID'             -ItemValue $CPU_Object.ProcessorID
  Add-Row -ItemName '  Physical Cores' -ItemValue `
                                            $CPU_Object.NumberofCores
  Add-Row -ItemName '  Logical  Cores' `
          -ItemValue $CPU_Object.NumberofLogicalProcessors
  Add-Row -ItemName '  Address Width'  -ItemValue `
                                            $CPU_Object.AddressWidth

  $HyperThreadingStatus = if ($CPU_Object.NumberOfCores -le
                              $CPU_Object.NumberofLogicalProcessors)
                              {'Enabled'} Else {'Disabled'}
  Add-Row -ItemName '  HyperThreading' -ItemValue $HyperThreadingStatus

  $VMFirmEnabled = if ($CPU_Object.VirtualizationFirmwareEnabled)
                        {'Enabled'} Else {'DisAbled'}
  Add-Row -ItemName '  VM Firmware'    -ItemValue "$VMFirmEnabled"

  $HWWidth = Get-MaxLength -TestObj $CITable.Item -MinLen 20

  $fmtBattery = @{Expression={$_.Item};Label="Item  ";Width=$HWWidth},
            @{Expression={$_.Value};Label="Value";Width=78-$HWWidth}

  $HWInfo = $CITable | Format-Table -Property $fmtBattery | Out-String

#Memory Info

  $MemoryInfo   = get-WMIObject -Computer $compname `
                                -Class Win32_PhysicalMemory |
                  Sort-Object  -Property {$_.DeviceLocator}

  $ManufLen = Get-MaxLength -TestObj $MemoryInfo.Manufacturer -MinLen 12

  $fmtMEM = @{Expression={$_.DeviceLocator.Trim('DIMM')};
                             Label="MB`nSlot";align='center';Width=4},
            @{Expression={$_.BankLabel.Trim('BANK ')};
                             Label="`nBank";Align='center';Width=4},
            @{Expression={$_.InterleavePosition};
                             Label="Bank`nPos";align='center';Width=4},
            @{Expression={$_.Speed};Label="`nSpeed";Width=5},
            @{Expression={$_.DataWidth};Label="Data`nWidth";Width=5},
            @{Expression={'{0:#.00}' -f ($_.Capacity/1gb)};
                           Label="Size`n/ GB";Width=5;align='right'},
            @{Expression={$_.Manufacturer};Label="`nManufacturer";
                                           Width=$ManufLen},
            @{Expression={$_.SerialNumber};Label="`nSerial No.";Width=10}

  $MemoryInfo = $MemoryInfo | Format-Table -Property $fmtMEM | Out-String

  $MemTitle     =  "Memory Information:" | Out-String

  $tabHardware  = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxHardware = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabHardware) ([ref]$tboxHardware) ('Hardware') `
              ('Computer, BIOS/UEFI, Processor, Memory')

  $tboxHardware.Text = $HWInfo + $MemTitle + $MemoryInfo

  Return ,$tboxHardware.Text

} # ------------------   End HardwareTab  -------------------------------

Function VideoTab {

  $Video = Get-WmiObject -Class Win32_VideoController `
                         -ComputerName $compname

  $VideoTabStartCnt = $CITable.Rows.Count
  Add-Row -ItemName 'Video Name' -ItemValue $Video.Name
  Add-Row -ItemName '  RAM (Mb)' -ItemValue ($Video.AdapterRAM/1mb)
  Add-Row -ItemName '  Mode Description' `
          -ItemValue  $Video.VideoModeDescription
  Add-Row -ItemName '  Refresh Rate' `
          -ItemValue $Video.CurrentRefreshRate
  Add-Row -ItemName '  Installed Drivers' `
          -ItemValue $Video.InstalledDisplayDrivers
  Add-Row -ItemName '  Driver Date' `
          -ItemValue ($Video.ConvertToDateTime($Video.DriverDate))
  Add-Row -ItemName '  Driver Version' -ItemValue $Video.DriverVersion

  $Mons = get-wmiobject -Query `
            'select * from Win32_PnPEntity where service="monitor"' |
            Select-Object -Property Name, Manufacturer, HardwareID

  $Mons | Add-Member -Name 'SerialNo' -MemberType 'NoteProperty' `
                   -Value $null

ForEach ($xx in $Mons) {

  $ErrorActionPreference = 'SilentlyContinue'

   $Separator = "{","\","}"
   $xx.HardwareID = $xx.HardwareId.split($Separator)[1]

   $keytype=[Microsoft.Win32.RegistryHive]::LocalMachine
   $reg=[Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey($keytype,"")
   $regKey= $reg.OpenSubKey("SYSTEM\\CurrentControlSet\\Enum\DISPLAY" )

   $regKey = $reg.OpenSubKey(
           "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\$($xx.HardwareId)")
   $DID = $regkey.GetSubKeyNames()

   ForEach($DID_KEY_NAME in $DID) {

         $regKey= $reg.OpenSubKey(
          "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\$($xx.HardwareId)" +
          "\\$($DID_KEY_NAME)\\Device Parameters")
         $EDID = $regKey.GetValue("EDID")

         ForEach($int in $EDID){

           $EDID_String = $EDID_String+([char]$int)

         }

         #Serial Number
         $checkstring =
            [char]0x00 + [char]0x00 + [char]0x00 + [char]0xFF + [char]0x00
         $matchfound =  $EDID_String -match "$checkstring(\S+)"
         if($matchfound){
           $xx.SerialNo = [String]$matches[1]}
         else {
         $xx.SerialNo = '-'
         #$monrow.Serial = '-'
         }

         $EDID_String = ''

   }   # End - foreach($DID_KEY_NAME in $DID)


}   #  End ---- ForEach $xx ----

 $ErrorActionPreference = 'Continue'

  $MNLen = Get-MaxLength -TestObj $Mons.Name         -MinLen 12
  $MFLen = Get-MaxLength -TestObj $Mons.Manufacturer -MinLen 12

  $monfmt  = @{Expression={$_.Name};Label="Monitor Name";Width=$MNLen},
             @{Expression={$_.Manufacturer};Label="Manufacturer";
                                            Width=$MFLen},
             @{Expression={$_.SerialNo};Label="Serial Number";Width=15},
             @{Expression={$_.HardWareID};Label="Registry ID";Width=15}

  $VidWidth = (Get-MaxLength -TestObj `
          $CITable.Item[$VideoTabStartCNt..$($CITable.Item.Count-1)]) + 3

  $fmtVid = @{Expression={$_.Item};Label="Item  ";Width=$VidWidth},
            @{Expression={$_.Value};Label="Value";Width=78-$VidWidth}

  $VideoHW    = 
           $citable.rows[$VideoTabStartCnt..$($CITable.Rows.Count - 1)] |
                  Format-Table -Property $fmtVid -Wrap | Out-String

  $MonTitle   = "Monitor Information:" | Out-String

  $MonInfo = $mons | Format-Table -Property $monfmt | Out-String

  $tabVideo       = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxVideo      = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabVideo) ([ref]$tboxVideo) ('Video') `
              ('Video Adapters, Resolutions, Monitors')

  $tboxVideo.Text = $VideoHW + $MonTitle + $MonInfo

  Return ,$tboxVideo.Text

} #----------------------- End VideoTab ---------------------------------

Function StorageTab {

 #Physical Drive Info

  $PhyDiskInfo = Get-Disk  |
          Where-Object {$_.size -gt 0 } | Sort-Object -Property DiskNumber
 
  $SSD = 0
  ForEach ($x in $phydiskinfo) { if ($x.model -match 'SSD') { $SSD+=1} }

  If ($SSD -gt 0) {

      $TrimVal = $(fsutil behavior query disabledeletenotify).Split('=')
                  
#--- PS IIF construct! ---
      $TrimStatus = (&{If([int]$TrimVal[1] -eq 0) {"Trim is Enabled"}
                     Else {"Trim is Disabled....Fix THIS!"}})
      $TrimStatus = 
@"
[ ---> $TrimStatus <--- ]

"@ | Out-String

  }  #End If ($SSD...

  $DNLen = Get-MaxLength -TestObj $PhyDiskInfo.Model        -MinLen 4
  $SNLen = Get-MaxLength -TestObj $PhyDiskInfo.SerialNumber -MinLen 13

  $fmtPhyDisk1 = 
     @{Expression={$_.DiskNumber};Label="Drive`n No.";Width=5;
                                  Align='Center'},
     @{Expression={$_.Model};Label="`nName";Width=$DNLen},
     @{Expression={$_.Model -Match 'SSD'};Label="`nSSD";
                                          Width=5;Align='left'},
     @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
                      Label="Disk Size`n / GB";Width=9;align='right'},
     @{Expression={$_.NumberOfPartitions};Label="Parti`ntions";Width=5},
     @{Expression={$_.PartitionStyle};Label="GPT`nMBR";Width=3},
     @{Expression={$_.IsBoot};Label="`nBoot";Width=5;Align="Left"},
     @{Expression={$_.BusType};Label="Data`nBus";Width=4}

  $fmtPhyDisk2 = 
     @{Expression={$_.DiskNumber};Label="Drive`n No.";Width=5;
                                  Align='Center'},
     @{Expression={$_.Model};Label="`nName";Width=$DNLen},
     @{Expression={$_.SerialNumber.Trim()};Label="`nSerial Number";
                                           Width=$SNLen;Align='left'},
     @{Expression={$_.HealthStatus};Label="Status";Width=7}

  $PhyDiskTitle = "Physical Disk Information:" | Out-String
  
  $PhyDiskInfo1 = $PhyDiskInfo | Format-Table -Property $fmtPhyDisk1 | 
                                 Out-String
  $PhyDiskInfo2 = $PhyDiskInfo | Format-Table -Property $fmtPhyDisk2 | 
                                 Out-String

 #Logical Drive Info

  $DiskTypeHash = @{
   2="Removable"
   3="Fixed"
   4="Network"
   5="Compact"}

  $fmtDisk = @{Expression={$_.Name};Label="Drive`nLetter";Width=6},
             @{Expression={$_.VolumeName};Label="Volume`nName";Width=10},
             @{Expression={$_.filesystem};Label="File`nSystem";Width=6},
             @{Expression={$DiskTypeHash.item([int]$_.DriveType)};
                                         Label="Drive`nType";Width=9},
             @{Expression={$_.compressed};Label="`nCompressed";Width=10;
                                          Align='left'},
             @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
               Label="Disk Size`n / GB";Width=9;align='right'},
             @{Expression={ '{0:#,000.00}' -f ($_.FreeSpace/1gb)};
               Label="Free Space`n / GB";Width=10;align='right'},
             @{Expression={$_.DiskIndex};Label="Drv`nNo";Width=4},
             @{Expression={$_.Bootable};Label="`nBoot";
                                        Width=5;Align='left'}

  $DiskInfoTitle = "Logical Disk Information:" | Out-String

  $DiskInfo = Get-Drives | Sort-Object  -Property DiskIndex, Name |
                           Format-Table -Property $fmtDisk | Out-String

  If ($AdminPriv) {
    $fmtPart = @{Expression={$_.Index};Label="Part`nNo";
                                       Width=4;Align='right'},
               @{Expression={$_.Alignment};Label="Part`nAlignment";
                                           Width=9;Align='left'},
               @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
                                           Label="Partition`nSize/GB";
                                           width=9;align='right'},
               @{Expression={(&{IF ($_.Bootable) {"Yes"} 
                                 else {""}})};Label="Bootable";
                                          Width=8; Align='Center'}

  $Partitions = `
            Get-WmiObject Win32_DiskPartition -ComputerName $CompName |
            Sort-Object DiskIndex, Index

  $Partitions | add-member -NotePropertyName Alignment `
                           -NotePropertyValue ""

  ForEach ($part in $Partitions) {
      $Part.Alignment = `
         (&{IF ($Part.StartingOffset % 4096 -eq 0) {"Aligned"} 
                                              else {"UnAligned"}}) 
  }

  $Partitions = Format-Table -InputObject $Partitions -Property $fmtPart `
                             -GroupBy Diskindex |Out-String
  }
  Else {
   $Partitions = @"

   This information is Available ONLY when RunAs Administrator

"@ | Out-String
  }

  $PartInfoTitle = "Disk Partition Information:" | Out-String

#CD/DVD Info

  $CDTitle = "CD/DVD Information:" | Out-String
  $CD    = Get-WmiObject -Class Win32_CDROMDrive -ComputerName $compname

  If ($CD -ne $null) {

    $fmtCD     = @{Expression={$_.Drive};Label="Drive Letter";Width=6},
                 @{Expression={$_.Name};Label="`nName";Width=25},
                 @{Expression={$_.CapabilityDescriptions};
                   Label="`nCapabilities";Width=40}

    $CDDVDINFO = $CD |  Format-Table -Wrap -Property $fmtCD | Out-String
  }
  Else{
    $CDDVDINFO = "None-Avaliable" | Out-String
  }

  #Mapped Drives Info

  $MDriveTitle = "Mapped Drive Information:" | Out-String

  $MappedDrives = get-WMIObject -class Win32_MappedLogicalDisk `
                                -computer $compname |
       Where-Object {$_.size -ge 0 -and $_.ProviderName -ne $null }

  If ($MappedDrives -eq $Null) {

    If ($AdminPriv) {
      $MDriveInfo = @"

    --No Mapped Drives--

    If drives were mapped using standard user privileges, even from an
    administrator account, they will not show up if this script is run
    as Administrator!

"@

    } #End If ($AdminPriv)

    Else {
          $MDriveinfo = @"

          --No Mapped Drives--

"@
    }

    $MDriveInfo = $MDriveInfo | Out-String

  } #End If ($MappedDrives...)

  Else {
        $fmtMDR = 
           @{Expression={$_.Name};Label="Drive`nLetter";Width=6},
           @{Expression={$_.Compressed};Label="Compressed";Width=5},
           @{Expression={$_.ProviderName};Label="Provider";Width=50},
           @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
                    Label="Disk Size`n / GB";Width=9;align='right';}

        $MDriveInfo = $MappedDrives| Format-Table -Property $fmtMDR |
                                     Out-String

  } #End Else ($MappedDrives...)

#Shares Info

  $fmtShare   = @{Expression={$_.Name};Label="`nShare Name";Width=12},
                @{Expression={$_.PSComputerName};Label="`nComputer Name";
                                                 Width=15},
                @{Expression={$_.MaximumAllowed};Label="Max`nUsers";
                                                 Width=5;align='Left'},
                @{Expression={$_.Path};Label="`nPath";Width=35}

  $ShareTitle = "Share Information:" | Out-String

  $ShareInfo  = Get-WmiObject -Class Win32_Share `
                              -ComputerName $compname |
                Where-Object {$_.Type -eq 0} | 
                Sort-Object  -Property Name |
                Format-Table -Property $fmtShare | Out-String

  $tabStorage       = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxStorage      = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabStorage) ([ref]$tboxStorage) ('Storage') `
              ('Drives, CD/DVD/BlueRay, Mapped Drives, Shares')

  $tboxStorage.Text = $PhyDiskTitle  + $PhyDiskInfo1 + $PhyDiskInfo2 +
                      $PartInfoTitle + $Partitions   + $TrimStatus   +
                      $DiskInfoTitle + $DiskInfo     +
                      $CDtitle       + $CDDVDINFO    +
                      $ShareTitle    + $ShareInfo    +
                      $MDriveTitle   + $MDriveInfo

  Return ,$tboxStorage.Text

} #-------------------- End Function StorageTab -------------------------

Function NetworkTab {

  $NetInfo = Get-NetAdapter -Name * 

  $NameLen = Get-MaxLength -TestObj $NetInfo.Name -MinLen 10
  $IFDLen = 80 - $NameLen - 20

  $fmtNet = @{Expression={$_.Name};Label="Connection`nName";
                                   Width=$NameLen},
            @{Expression={$_.Status};Label="`nStatus";Width=12},
            @{Expression={ '{0:#,000}' -f ($_.Speed/1000000)};
                                     Label="Speed`n / MB";
                                     Width=5;Align='right'},
            @{Expression={$_.InterfaceDescription};
                                   Label="`nAdapter Name";Width=$NameLen}

  $NetInfo = $NetInfo | Format-Table -Property $fmtNet -wrap | Out-String

  $fmtNetIP = 
      @{Expression={$_.InterfaceAlias};Label='Interface';Width=10},
      @{Expression={$_.AddressFamily}; Label='IP Width'; Width= 5},
      @{Expression={$_.IPAddress};   Label='IP Address'; Width=30}

  $IPInfo =  Get-NetIPAddress -ErrorAction SilentlyContinue `
                              -InterfaceAlias "Ethernet","Wi-Fi"  |
             Sort-Object  -Property InterfaceAlias, AddressFamily |
             Format-Table -Property $fmtNetIP | Out-String

  $tabNetwork  = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxNetwork = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabNetwork) ([ref]$tboxNetwork) ('Network') `
              ('Status of Network Adapters (NICs)')

  $tboxNetwork.Text = "$NetInfo $IPInfo"

  Return ,$tboxNetwork.Text

} #------------------------- End Network Tab ----------------------------

Function PrinterTab {

  $fmtPrt = @{Expression={$_.DeviceID};  Label="Printer";Width=34},
            @{Expression={$_.DriverName};Label="Driver"; Width=24},
            @{Expression={$_.PortName};  Label="Port";   Width=20}

  $tabPrinters       = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxPrinters      = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabPrinters) ([ref]$tboxPrinters) ("Printers")

  $tboxPrinters.Text =
           Get-WMIObject  -Computer $compname -Class Win32_Printer |
           Select-Object  -Property DeviceID,DriverName, PortName |
           Sort-Object    -Property DeviceId |
           Format-Table   -Property $fmtPrt -Wrap | Out-String

  Return ,$tboxPrinters.Text

} #------------------------- End Printers Tab ---------------------------

Function WindowsTab {

  Get-CimClass -PropertyName version -QualifierName dynamic | Out-Null
  $CurOS          = Get-CimInstance -Class Win32_OperatingSystem
  $OSKey          = Get-WindowsProductKey $ComputerSysObj.name
  $OS_Object      = get-WMIObject -Class Win32_OperatingSystem `
                                  -computer $compname
  $LocalDateTime  = 
        $OS_Object.ConvertToDateTime($OS_Object.LocalDateTime)
  $LastBootUpTime = 
        $OS_Object.ConvertToDateTime($OS_Object.LastBootUpTime)
  $SysRoot        = Get-ChildItem  -Path Env:SystemRoot
  $UserProf       = Get-ChildItem -Path Env:UserProfile

  $PreUpTime      = @([String]($LocalDateTime - $LastBootUpTime)).split('.')
  If ($PreUptime.count -gt 2) {
    $UpTime = "$($PreUptime[0]) Day(s) $($PreUptime[1])"
  }
  Else {
    $UpTime = $PreUpTime[0]
  }

  $OSCompacted = Compact.exe /CompactOS:Query
  $OSCompacted = 
     (($OSCompacted -like "*not in the Compact state*").count -eq 0)

  $OSTabStartCNt = $CITable.Rows.Count
  Add-Row -ItemName 'OS Name'         -ItemValue $CurOS.Caption
  Add-Row -ItemName '   Version'      -ItemValue $CurOS.Version
  Add-Row -ItemName '   Bit-Width'    -ItemValue $CurOS.OSArchitecture
  Add-Row -ItemName '   Key'          -ItemValue $OSKey
  Add-Row -ItemName '   Registered Owner'   `
          -ItemValue $CurOs.RegisteredUser

  If ($CurOS.Organization -ne "") {
    Add-Row -ItemName 'Registered Org'
            -ItemValue = $CurOS.Organization
  }
  Else {

    $ErrorActionPreference = "SilentlyContinue"
    $RO = Get-ItemProperty `
       -Path "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\" `
       -Name Organization
    $ErrorActionPreference = "Continue"

    If ($RO -ne "") {
      Add-Row -ItemName  '   Registered Org' `
              -ItemValue $RO.Organization
    } #End If ($RO...
  }   #End Else

  Add-Row -ItemName '   System Root'  -ItemValue $SysRoot.Value
  Add-Row -ItemName '   Install Date' `
          -ItemValue $CurOS.InstallDate -f "mm/dd/yy"
  Add-Row -ItemName '   Date/Time'    -ItemValue $LocalDateTime
  Add-Row -ItemName '   Last Boot'    -ItemValue $LastBootUpTime
  Add-Row -ItemName '   Up Time'      -ItemValue $UpTime
  Add-Row -ItemName '   OS Compacted' -ItemValue $OSCompacted
  Add-Row -ItemName ' '
  Add-Row -ItemName 'Computer Name'   -ItemValue $ComputerSysObj.Name
  Add-Row -ItemName 'Current User ID' -ItemValue $ComputerSysObj.Username
  Add-Row -ItemName 'User Profile'    -ItemValue $UserProf.Value
  Add-Row -ItemName 'Domain Name'     -ItemValue $ComputerSysObj.Domain
  Add-Row -ItemName ' '

  Add-Row -ItemName 'Security Software'
  $FWKey = "System\CurrentControlSet\Services\SharedAccess"
  $FWKey = "$FWKey\Parameters\FirewallPolicy\StandardProfile"
  $FWallStatus =
     [Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey("LocalMachine",
     $env:COMPUTERNAME).OpenSubKey("$FWKey").GetValue("EnableFirewall")

  If ($FWallStatus -eq 1) {
    $FWallStatus = "On"
  }
  Else {
    $FWallStatus = "OFF"
  }

  Add-Row -ItemName "  Firewall"   -ItemValue $FWallStatus

  $AV = Get-AntiVirus
  Add-Row -ItemName '  AV Program' -ItemValue "$($AV.displayName)"
  Add-Row -ItemName '  AV State'   -ItemValue $($AV.productState)
  Add-Row -ItemName ' '

  #--- Windows Update Information ---

  $WU = Get-WindowsUpdateConfig
  Add-Row -ItemName 'Windows Update'
  Add-Row -ItemName '  Notification Level' -ItemValue $WU.NotificationLevel
  Add-Row -ItemName '  Update Days'        -ItemValue $WU.UpdateDays
  Add-Row -ItemName '  Update Hour'        -ItemValue $WU.UpdateHour
  Add-Row -ItemName '  Recommended Updates' `
          -ItemValue $WU.'Recommended updates'
  Add-Row -ItemName ' '

  #--- PowerShell Information ---

  Add-Row -ItemName  "PowerShell Information"
  Add-Row -ItemName  "  PS Version"  `
          -ItemValue "$($PSVersionTable.PSVersion)"
  Add-Row -ItemName  "  Runtime Lang. Version"  `
          -ItemValue ("$($PSVersionTable.clrversion.Major)." + 
                      "$($Psversiontable.clrversion.minor)")
  Add-Row -ItemName  "  WS-Man Stack  Version"  `
          -ItemValue "$($PSVersionTable.WsManStackVersion)"
  Add-Row -ItemName  "  PS Remoting   Version"  -ItemValue `
                 ("$($PSVersionTable.PSRemotingProtocolVersion.Major)." +
                  "$($PSVersionTable.PSRemotingProtocolVersion.Minor)")
  
  $BasePath = 'HKCU:\SOFTWARE\Microsoft\PowerShell\1'
  If (Test-Path -Path `
    "$BasePath\ShellIds\Microsoft.PowerShell") {
  $PSEP = Get-ItemProperty -path `
     "$BasePath\ShellIds\Microsoft.PowerShell" |
        Select ExecutionPolicy
  Add-Row -ItemName  "  Execution Policy"  `
          -ItemValue "$($PSEP.ExecutionPolicy)"
  }

     #--- .Net Information ---

  $DotNetTitle = "Installed .Net Versions:" | Out-String

  $DotNet =
   Get-ChildItem `
     'HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP' -recurse |
   Get-ItemProperty -name Version,Release -EA 0 |
   Where { $_.PSChildName -match '^(?!S)\p{L}'} | Sort-Object Version |
   Select PSChildName, Version, Release

  $NameLen = Get-MaxLength -TestObj $DotNet.PSChildName -MinLen 9
  $VerLen  = Get-MaxLength -TestObj $DotNet.Version     -MinLen 11
  $RNLen   = Get-MaxLength -TestObj $DotNet.Release     -MinLen 11

  $fmtDotNet = 
    @{Expression={$_.PSChildName};Label=".Net Type";Width=$NameLen},
    @{Expression={$_.Version};Label="Version No:";Width=$VerLen},
    @{Expression={$_.Release};Label="Release No:";Width=$RNLen}

  $DOtNet = $DotNet | Format-Table $fmtDotNet | Out-String

  $WinWidth = (Get-MaxLength -TestObj `
              $CITable.Item[$OSTabStartCNt..$($CITable.Item.Count-1)]) + 3

  $fmtWin = @{Expression={$_.Item};Label="Item  ";Width=$WinWidth},
            @{Expression={$_.Value};Label="Value";Width=78-$WinWidth}

  $WinInfo = $citable.rows[$OSTabStartCnt..$($CITable.Rows.Count - 1)] |
               Format-Table -Property $fmtWin | Out-String

  #--- User Account Information ---

  $uai    = Get-UserAccounts

  $UIDLen = Get-MaxLength -TestObj $uai.Name   -MinLen 6
  $WGLen  = Get-MaxLength -TestObj $uai.Domain -MinLen 9

  $fmtAcct = @{Expression={$_.Name};Label="`nUserID";Width=$UIDLen},
             @{Expression={$_.Domain};
                Label="Domain/`nWorkGroup";Width=$WGLen},
             @{Expression={If ($_.Disabled) {'Disabled'} ELse {'Enabled'}};
                Label="`nStatus";align='left';Width=8},
             @{Expression={If ($_.PasswordRequired) {'   Yes'}
                                               ELse {'   No'}};
                Label="|----P A`nRequired";align='left';Width=8},
             @{Expression={If ($_.PasswordExpires) {'  Yes'} ELse {'  No'}};
                Label="S S W O`nExpires";align='left';Width=7},
             @{Expression={If ($_.PasswordChangeable) {'   Yes'}
                           ELse {'   No'}};
                Label="R D S---|`nChangable";align='left';Width=9}

  $UATitle = "User Account Info:" | Out-String
  $uai     = $uai | Format-Table -Property $fmtAcct | Out-String

  $tabWindows       = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxWindows      = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabWindows) ([ref]$tboxWindows) ('Windows') `
              ('Windows, Security S/W, PowerShell, .Net, User Accts.')

  $tboxWindows.Text = $WinInfo + $DotNetTitle + $DotNet + $UATitle + $uai

  Return ,$tboxWindows.Text

} #------------------------ End Windows Tab -----------------------------

Function EnvironmentTab {

  $WinEnvVars = Get-ChildItem Env:
  $EnvLen   = Get-MaxLength -TestObj $WinEnvVars.Name -MinLen 10

  $fmtENV = @{Expression={$_.Name};Label="Env Variable";Width=$EnvLen},
            @{Expression={$_.Value};Label="Value";Width=78-$EnvLen}
  $EnvInfo = $WinEnvVars | Format-Table $fmtENV -Wrap | Out-String

  $tabEnviron       = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxEnviron      = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabEnviron) ([ref]$tboxEnviron) ('Environment') `
              ('Windows Environment Variables.')

  $tboxEnviron.Text = $EnvInfo

  Return ,$tboxEnviron.Text

} #----------------------- End EnvironmentTab ---------------------------

Function DriversTab {

  $DriverInfo  = Get-WmiObject -Computer $compname `
                               -Class Win32_PNPSignedDriver |
                 Where-Object  -Property DriverVersion -notlike `
                               -Value "*.?.????*.*" |
                 Where-Object  -Property DeviceName -ne -Value $Null |
                 Sort-Object   -Property DeviceName -Unique

  $DVLen = Get-MaxLength -TestObj $DriverInfo.DriverVersion -MinLen 14
  $DDLen = 78 - $DVLen

  $fmtDRVR = 
    @{Expression={$_.DeviceName};Label="Driver Description";Width=$DDLen},
    @{Expression={$_.DriverVersion};Label="Version Number" ;Width=$DVLen}

  $DrvTitle    = "Non-Windows Unique Drivers and Version Numbers:" |
                   Out-String

  $DriverInfo = $DriverInfo | Format-Table -Property $fmtDRVR | Out-String

  $tabDrivers  = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxDrivers = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabDrivers) ([ref]$tboxDrivers) ("Drivers")

  $tboxDrivers.Text = $DrvTitle + $DriverInfo

  Return ,$tboxDrivers.text

} #------------------------- End Drivers Tab  ---------------------------

Function ProgramsTab {

  $fmtPgms = 
      @{Expression={$_.DisplayName};Label="Program Name";Width=40},
      @{Expression={$_.InstallDate};Label="Installed";Width=9},
      @{Expression={$_.InstallLocation};Label="Install Path";Width=31}

if (!([Diagnostics.Process]::GetCurrentProcess().Path -match
       '\\syswow64\\'))
{
  $unistallPath = "\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\"
  $unistallWow6432Path =
    "\SOFTWARE\Wow6432Node\Microsoft\Windows\CurrentVersion\Uninstall\"

 $PgmsInfo =
  @(
    if (Test-Path   -Path "HKLM:$unistallWow6432Path") {
      Get-ChildItem -Path "HKLM:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKLM:$unistallPath") {
      Get-ChildItem -Path "HKLM:$unistallPath"  }
    if (Test-Path   -Path "HKCU:$unistallWow6432Path") {
      Get-ChildItem -Path "HKCU:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKCU:$unistallPath") {
      Get-ChildItem -Path "HKCU:$unistallPath"  }
   ) |
   Where-Object {
     $_.GetValue('DisplayName') -and
    ($_.GetValue('UninstallString') -or
     $_.GetValue('NoRemove')) -and
    !$_.GetValue('SystemComponent') -and
    !$_.GetValue('ReleaseType') -and
    !$_.GetValue('ParentKeyName')
  } | Get-ItemProperty 
 

#Add Member to Object
  $PgmsInfo| Add-Member -NotePropertyName BasePath -NotePropertyValue ""

  $ErrorActionPreference = "SilentlyContinue"

  $PgmsInfo | ForEach-Object {
     $Parts = $_.InstallLocation.Trim('"').Split('\')
     $_.BasePath = "$($Parts[0])\$($Parts[1])"
     $Temp = "\"
     For ($i = 2 ; $i -lt $Parts.Count ; $i++) {
        $Temp = $Temp + $Parts[$i] + "\"
     }
     $_.InstallLocation = $Temp.Replace('\\','\')
     $_.InstallDate = $_.InstallDate.Replace('/','')
  }

  $ErrorActionPreference = "Continue"
   
  $PgmsInfo = $PgmsInfo |  Sort-Object -Property BasePath, DisplayName |
              Format-Table -Property $fmtPgms -Wrap -GroupBy BasePath |
              Out-String
} #End If (!([Diagnostics.Process]::GetCurrentProcess().Path...
else
{
  $PgmsInfo =
    "You are running 32-bit Powershell on 64-bit system.`n" +
    "Please run 64-bit Powershell instead." | Out-String
}

  $tabInstPgms    = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxInstPgms   = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabInstPgms) ([ref]$tboxInstPgms) ("Installed Programs")

  $tboxInstPgms.Text = $PgmsInfo

  Return ,$tboxInstPgms.Text

} #------------------------   End Programs Tab    -----------------------

Function RegistryServicesTab {

  $ServicesTabStartCnt = $CITable.Rows.Count

  $Prop = Get-ItemProperty `
   -Path "HKLM:\SYSTEM\CurrentControlSet\Control\Session Manager\Power" `
   -Name "HiberbootEnabled"

  Add-Row -ItemName "Registry: HiberbootEnabled" `
          -ItemValue ([Bool]$($Prop.HiberbootEnabled.Equals(1)))

  $Prop = Get-ItemProperty `
          -Path "HKLM:SYSTEM\CurrentControlSet\Control\Power" `
          -Name "HibernateEnabled"

  Add-Row -ItemName "Registry: HibernateEnabled" `
          -ItemValue ([Bool]$($Prop.HibernateEnabled.Equals(1)))

  Get-Service -Name HomeGroup*,USBDLM*,Macrium*,Remote*,MsKey*,
                    VSS,Wsearch,DiagTrack |
     ForEach-Object  {
        Add-Row -ItemName "Service:  $($_.DisplayName)" `
                -ItemValue $($_.Status)
     }

  Add-Row -ItemName ''
  Add-Row -ItemName '[  Security Items  ]'
  $UAC = Get-ItemProperty -Name "EnableLUA" -Path `
     "HKLM:\SOFTWARE\Microsoft\Windows\CurrentVersion\Policies\System" 
  Add-Row -ItemName 'Registry: UAC Enabled' `
          -ItemValue ($UAC.EnableLUA.Equals(1))

  Get-Service -Name MBAM*,EMET*,WinDefend*,MpsSvc,wuauserv,BDESVC,fhsvc |
     ForEach-Object  {
        Add-Row -ItemName "Service:  $($_.DisplayName)" `
                -ItemValue $($_.Status)
     }

  $fmtServices    = @{Expression={$_.Item};Label="Item";Width=60},
                    @{Expression={$_.Value};Label="Value";Width=20}

  $ServicesInfo   = `
       $CITable.Rows[$ServicesTabStartCnt..$($CITable.Rows.Count - 1)] |
                  Select-Object -Property Item,Value   |
                  Format-Table  -Property $fmtServices | Out-String

  $tabServices    = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxServices   = New-Object -TypeName System.Windows.Forms.Textbox

  GenerateTab ([ref]$tabServices) ([ref]$tboxServices) `
              ('Registry/Services') ('Registry/Services...Hibernate,' +
               ' HyperBoot, and Homegroup, MalwareBytes')

  $tboxServices.Text = $ServicesInfo

  Return ,$tboxServices.text

} #---------------------------- End Services Tab ------------------------

Function BatteryTab {

  $Battery = $True

  Try {
        $colBatStatus = gwmi -Class BatteryStatus `
                             -Namespace root\wmi `
                             -ErrorAction  "Stop"
  }
  Catch [Exception] 
  {
    $Battery = $False
  }

If ($Battery) {

  $Availability_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Running/Full Power',
  'Warning',
  'In Test',
  'Not Applicable',
  'Power Off',
  'Off Line',
  'Off Duty',
  'Degraded',
  'Not Installed',
  'Install Error',
  'Power Save - Unknown',
  'Power Save - Low Power Mode',
  'Power Save - Standby',
  'Power Cycle',
  'Power Save - Warning',
  'Paused',
  'Not Ready',
  'Not Configured',
  'Quiesced'
  )

  $BatteryStatus_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Fully Charged',
  'Low',
  'Critical',
  'Charging',
  'Charging and High',
  'Charging and Low',
  'Charging and Critical',
  'Undefined',
  'Partially Charged'
  )

  $Chemestry_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Lead Acid',
  'Nickel Cadmium',
  'Nickel Metal Hydride',
  'Lithium-ion',
  'Zinc Air',
  'Lithium Polymer'
  ) 
  
  $StatusInfo_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Enabled',
  'Disabled',
  'Not Applicable'
  ) 

  $colItems = get-wmiobject -ComputerName $CompName `
                            -Class "Win32_Battery" `
                            -Namespace "root\CIMV2"

  $BatteryTabStartCnt = $CITable.Rows.Count

  If ($colBatStatus.PowerOnline) {
    Add-Row -ItemName 'On Mains' -ItemValue $colbatstatus.poweronline
      If ($colItems.TimeToFullCharge -ne $Null) {
        Add-Row -ItemName 'Time To Full Charge' `
                -ItemValue $colItems.TimeToFullCharge 
      }
      Add-Row -ItemName 'Charging' -ItemValue $colBatStatus.Charging
  }

  Else {
      Add-Row -ItemName  'Availability' -ItemValue `
                   $Availability_ReturnValue[$colItems.Availability]
      If ($colItems.BatteryRechargeTime -ne $null) {
        Add-Row -ItemName 'Battery Recharge Time'  `
                -ItemValue $colItems.BatteryRechargeTime]
      }
     If ($colItems.ConfigManagerErrorCode -ne $Null) {
        Add-Row -ItemName  'Configuration Manager Error Code' `
                -ItemValue $colItems.ConfigManagerErrorCode 
      }
      If ($colItems.ConfigManagerUserConfig -ne $Null) {
        Add-Row -ItemName 'Configuration Manager User Configuration' `
                -ItemValue $colItems.ConfigManagerUserConfig 
      }
      Add-Row -ItemName 'Description' $colItems.Description 
      If ($colItems.DesignCapacity -ne $Null) {
      Add-Row -ItemName 'Design Capacity' `
              -ItemValue $colItems.DesignCapacity
      }
      Add-Row -ItemName  'Design Voltage' `
              -ItemValue $($colItems.DesignVoltage/1000 )
      Add-Row -ItemName  'Device ID' `
              -ItemValue $colItems.DeviceID 

      If ($colItems.ErrorCleared -ne $Null) {
        Add-Row -ItemName  'Error Cleared' `
                -ItemValue $colItems.ErrorCleared 
      }
      If ($colItems.ErrorDescription -ne $Null) {
        Add-Row -ItemName  'Error Description' `
                -ItemValue $colItems.ErrorDescription 
      }

      Add-Row -ItemName  'Estimated Charge Remaining' `
              -ItemValue  "$($colItems.EstimatedChargeRemaining)%" 

      $TS = New-TimeSpan -Minutes $($colItems.EstimatedRunTime)
      Add-Row -ItemName 'Estimated Run Time'  -ItemValue $ts 

      If ($colItems.ExpectedBatteryLife -ne $Null) {
        Add-Row -ItemName  'Expected Battery Life' `
                -ItemValue $colItems.ExpectedBatteryLife 
        Add-Row -ItemName  'Expected Life' `
                -ItemValue $colItems.ExpectedLife 
      }
      If ($colItems.FullChargeCapacity -ne $Null) {
        Add-Row -ItemName  'Full Charge Capacity' `
                -ItemValue $colItems.FullChargeCapacity
      } 
      If ($colItems.InstallDate -ne $Null) {
        Add-Row -ItemName  'Installation Date' `
                -ItemValue $colItems.InstallDate 
      }
      If ($colItems.LastErrorCode -ne $Null) {
        Add-Row -ItemName  'Last Error Code' `
                -ItemValue $colItems.LastErrorCode 
      }
      If ($colItems.MaxRechargeTime -ne $Null) {
        Add-Row -ItemName  'Maximum Recharge Time' `
                -ItemValue $colItems.MaxRechargeTime 
      }
      Add-Row -ItemName 'Name' -ItemValue $colItems.Name 
      If ($colItems.PNPDeviceID -ne $Null) {
        Add-Row -ItemName 'PNP Device ID' `
                -ItemValue $colItems.PNPDeviceID
      }
      If ($colItems.SmartBatteryVersion -ne $Null) {
        Add-Row -ItemName  'Smart Battery Version' `
                -ItemValue $colItems.SmartBatteryVersion 
      }
      Add-Row -ItemName 'Status' $colItems.Status 
      If ($colItems.StatusInfo -ne $Null) {
        Add-Row -ItemName 'Status Information' `
                -ItemValue  $StatusInfo_ReturnValue[$colItems.StatusInfo]
      }
      If ($colItems.TimeOnBattery -ne $Null) {
        $TS = New-TimeSpan -Seconds $colItems.TimeOnBattery
        Add-Row -ItemName 'Time On Battery' -ItemValue $TS  
      }
  } #End Else..If ($colBatStatus.PowerOnline)

  $BatteryChg = Test-BatteryHealth
  If ($BatteryChg -eq -1) { $BatteryChg = "Undetermined"}
  Add-Row -ItemName  'Battery Last Full Charge' `
          -ItemValue "$BatteryChg% of Original Capacity"  

  Add-Row -ItemName  'Current Battery Charge mWhr' `
          -ItemValue $colBatStatus.RemainingCapacity
  Add-Row -ItemName  'Battery Status' `
          -ItemValue $BatteryStatus_ReturnValue[$colItems.BatteryStatus]
  Add-Row -ItemName  'Caption' `
          -ItemValue $colItems.Caption 
  Add-Row -ItemName  'Chemistry' `
          -ItemValue $Chemestry_ReturnValue[$colItems.Chemistry] 
  Add-Row -ItemName  'Power Management Capabilities' `
          -ItemValue $colItems.PowerManagementCapabilities 
  Add-Row -ItemName  'Power Management Supported' `
          -ItemValue $colItems.PowerManagementSupported 

  $ItemLen = Get-MaxLength -TestObj `
          $CITable.Item[$BatteryTabStartCnt..$($CITable.Rows.Count -1)]
  $ItemLen += 2

  $fmtBattery = @{Expression={$_.Item};Label="Item  ";Width=$ItemLen},
                @{Expression={$_.Value};Label="Value";Width=80-$ItemLen}

  $BatteryInfo = 
    $CITable.Rows[$BatteryTabStartCnt..$($CITable.Rows.Count -1)] |
    Format-Table -Property $fmtBattery | Out-String

  $tabBattery  = New-Object -TypeName System.Windows.Forms.TabPage
  $tboxBattery = New-Object -TypeName System.Windows.Forms.TextBox

  GenerateTab ([ref]$tabBattery) ([ref]$tboxBattery) ('Battery') `
              ('Protable Devices Battery Information')

  $tboxBattery.Text = $BatteryInfo

  Return ,$tboxBattery.text   
    
  } #End If ($Battery)

} #---------------------------- End Battery Tab  ------------------------

Function GenerateTab {

 param (
         [Parameter(Mandatory=$True)]
          [System.Windows.Forms.TabPage] $tabObjectName,
         [Parameter(Mandatory=$True)]
          [System.Windows.Forms.Textbox] $tboxObjectName,
         [Parameter(Mandatory=$True)]
          [string] $tabName,
         [Parameter(Mandatory=$False)]
          [string] $TipText
        )

  $tabObjectName.Name          = "$tabName"
  $tabObjectName.Location      = $tabLocation
  $tabObjectName.Padding       = $System_Windows_Forms_Padding
  $tabObjectName.Size          = $tabSize
  $tabObjectName.TabIndex      = 0
  $tabObjectName.Text          = $TabName
  If ($TipText -ne $null) {
    $tabObjectName.ToolTipText = $TipText
  }

  $tabObjectName.DataBindings.DefaultDataSourceUpdateMode = 0
  $tabObjectName.UseVisualStyleBackColor = $True

  $tboxObjectName.Location   = $tboxLocation
  $tboxObjectName.Size       = $tboxSize
  $tboxObjectName.MultiLine  = $True
  $tboxObjectName.ForeColor  = $tboxForeColor
  $tboxObjectName.BackColor  = $tboxBackColor
  $tboxObjectName.ScrollBars = 'Both'
  $tboxObjectName.Font       = $tboxFont
  $tboxObjectName.ReadOnly   = $True

  $MainTabCtrl.Controls.Add($tabObjectName)
  $tabObjectName.Controls.add($tboxObjectName)

}   #---------------------- End GenerateTab -----------------------------

Function Add-Row {

  param (
         [string]$ItemName,
         [string]$ItemValue
        )

  $CITRow       = $CITable.NewRow()  #Create Row Variable
  $CITRow.Item  = $ItemName          #Assign items to row variable
  $CITRow.Value = $ItemValue
  $CITable.Rows.Add($CITRow)         #Add Row to Table using Row Variable

}  #------------------ End Function Add-Row -----------------------------

Function Combine-Object {

  param(
	 [Parameter(Mandatory=$True)] $object1,
	 [Parameter(Mandatory=$True)] $object2
	)

#	 Courtesy of: http://powershell.com/cs/media/p/7924.aspx

	$propertylistObj1 = @($object1 | Get-Member -ea Stop `
                                                -memberType *Property |
                                     Select-Object -ExpandProperty Name)
	$propertylistObj2 = @($object2 | Get-Member -memberType *Property |
                                     Select-Object -ExpandProperty Name |
                                     Where-Object { $_ -notlike '__*'})

	$propertylistObj2 | ForEach-Object {
		if ($propertyListObj1 -contains $_) {
			$name = '_{0}' -f $_
		}
        else {
			$name = $_
		}

	  $object1 = $object1 | Add-Member -NotePropertyName $name `
                                       -NotePropertyValue ($object2.$_) `
                                       -PassThru
	}

	Return ,$object1

} #----------------- End Function Combine-Object ------------------------

Function Get-AdminStatus {

    If (-NOT ([Security.Principal.WindowsPrincipal] `
          [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole(`
          [Security.Principal.WindowsBuiltInRole] "Administrator"))
    {"User"}
    Else
    {"Administrator"}

} # -----------  End Function Get-AdminStatus -----------------

Function Get-AntiVirus{

<#+--------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddawy |
  +--------------------------------------------------+
#>
  [CmdletBinding()]
  param (
    [parameter(ValueFromPipeline=$true,
      ValueFromPipelineByPropertyName=$true)]
     [string]$computername="$env:COMPUTERNAME"
  )

  PROCESS{
   $os = Get-WmiObject -Class Win32_OperatingSystem `
                       -ComputerName $computername

   if ([int]$os.BuildNumber -ge 6001 ) {
     $av = Get-WmiObject -Namespace 'ROOT\SecurityCenter2' `
                         -Class AntiVirusProduct `
                         -ComputerName $computername
   }
   else {
         $av = Get-WmiObject -Namespace 'ROOT\SecurityCenter' `
            -Class AntiVirusProduct -ComputerName $computername
   }

   Return ,$av

  } #End Process

} #--------------- End Function Get-AntiVirus  ----------------------

Function Get-Drives {

#	 Courtesy of: http://powershell.com/cs/media/p/7924.aspx

  Get-WmiObject -computer $compname -Class Win32_DiskPartition |
	ForEach-Object {
		$partition = $_
		$logicaldisk = $partition.psbase.GetRelated('Win32_LogicalDisk')
		if ($logicaldisk -ne $null) {
			Combine-Object $logicaldisk $partition
		}
	} |  Select-Object -Property Name, VolumeName, FileSystem, DriveType,
                                 Size, Compressed, FreeSpace,  DiskIndex, 
                                 Index, Bootable

  #Note: This returns the Select-Object output to the caller!
  #      The Select-Object could be dropped but unneeded data would be
  #      returned to caller...inefficient!

} #--------------    End Function Get-Drives  ----------------------

Function Get-MaxLength {
<#
.SYNOPSIS
   Finds the length of the longest item in collection.

.DESCRIPTION
   Use this Function to get the length of the longest item in a
   collection for use in format strings or other places where needed

.PARAMETER TestObj
    The qualified object to be tested. See example!

.Parameter MinLen
    The minimum length of the item (if using for formatting) which
    should be the Label (title) length. Note if the object item
    being tested does not have a Length property you MUST specify
    the label length!



.OUTPUTS
    Returns a numerical value

.NOTES


.EXAMPLE
   $NameLen = Get-MaxLength -TestObj $DotNet.PSChildName
   $VerLen  = Get-MaxLength -TestObj $DotNet.Version
   $RNLen   = Get-MaxLength -TestObj $DotNet.Release -MinLen 11

     #--- .Net Information ---

  $fmtDotNet = 
    @{Expression={$_.PSChildName};Label=".Net Type";Width=$NameLen},
    @{Expression={$_.Version};Label="Version No:";Width=$VerLen},
    @{Expression={$_.Release};Label="Release No:";Width=$RNLen}

  $Dotnet | Format-Table $fmtDotNet
#>

  Param(
    [Parameter(Mandatory=$True)]
     [object] $TestObj,
    [Parameter(Mandatory=$False)]
     [int] $MinLen = 0
  )

   $ErrorActionPreference = "SilentlyContinue"

   foreach ($x in $TestObj) {
     If ($x.Trim().length -gt $MinLen) {
       $MinLen = $x.Trim().length
     }
   }

   $ErrorActionPreference = "Continue"

   Return ,$MinLen

}  #-----------  End Function Get-MaxLength  ------------------------

Function Get-UserAccounts{

<#+--------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddawy |
  +--------------------------------------------------+
#>
  [CmdletBinding()]
  param (
   [parameter(ValueFromPipeline=$true,
     ValueFromPipelineByPropertyName=$true)]
    [string]$computername="$env:COMPUTERNAME"
  )

  PROCESS{
   Get-WmiObject -Class Win32_UserAccount -ComputerName $computername |
     Select-Object -Property `
            AccountType, Description,  Disabled, Domain, FullName,
            InstallDate, LocalAccount, Lockout,  Name, SID, SIDType, 
            PasswordChangeable, PasswordExpires, PasswordRequired
  } #End Process

}  #------------------  End Function Get-UserAccounts  ------------------

Function Get-WindowsProductKey {

  param (
          [Parameter(Mandatory=$True)]
            [string]$computer
        )

 $Reg = [WMIClass] ("\\" + $computer + "\root\default:StdRegProv")
 $values = [byte[]]($reg.getbinaryvalue(2147483650,
           "SOFTWARE\Microsoft\Windows NT\CurrentVersion",
           "DigitalProductId").uvalue)
 $lookup = [char[]]("B","C","D","F","G","H","J","K","M","P","Q","R",
                    "T","V","W","X","Y","2","3","4","6","7","8","9")
 $keyStartIndex      = [int]52;
 $keyEndIndex        = [int]($keyStartIndex + 15);
 $decodeLength       = [int]29
 $decodeStringLength = [int]15
 $decodedChars       = New-Object char[] $decodeLength
 $hexPid             = New-Object System.Collections.ArrayList

 for ($i = $keyStartIndex; $i -le $keyEndIndex; $i++) {
   [void]$hexPid.Add($values[$i])
 }

 for ( $i = $decodeLength - 1; $i -ge 0; $i--) {

   if (($i + 1) % 6 -eq 0) {
     $decodedChars[$i] = '-'
   }
   else
   {
     $digitMapIndex = [int]0
     for ($j = $decodeStringLength - 1; $j -ge 0; $j--)
     {
      $byteValue = [int](($digitMapIndex * [int]256) -bor 
                          [byte]$hexPid[$j])
      $hexPid[$j] = [byte] ([math]::Floor($byteValue / 24))
      $digitMapIndex = $byteValue % 24
      $decodedChars[$i] = $lookup[$digitMapIndex]
     }
   }
 }   #End For ($i...)

 $WindowsKey = ''
 $decodedChars | ForEach-Object { $WindowsKey+=$_}

 Return ,$WindowsKey

}  #----------  End Function Get-WindowsProductKey ----------------------

Function Get-WindowsUpdateConfig {

<#+---------------------------------------------------------+
  | Thanks to James O'Niel for this one!                    |
  | http://blogs.technet.com/b/jamesone/archive/2009/01/27/ |
  |        managing-windows-update-with-powershell.aspx     |
  +---------------------------------------------------------+
#>

  $AUNotificationLevels= @{
    0="Not configured";
    1="Disabled" ;
    2="Notify before download";
    3="Notify before installation";
    4="Scheduled installation"}

  $AUDays=@{
    0="Every Day";
    1="Every Sunday";
    2="Every Monday";
    3="Every Tuesday";
    4="Every Wednesday";
    5="Every Thursday";
    6="Every Friday";
    7="EverySaturday"}

  $AUSettings = (New-Object -com "Microsoft.Update.AutoUpdate").Settings

  $AUObj = New-Object -TypeName System.Object

  Add-Member -inputObject $AuObj `
             -MemberType NoteProperty `
             -Name "NotificationLevel" `
             -Value $AUNotificationLevels[$AUSettings.NotificationLevel]

  Add-Member -inputObject $AuObj `
             -MemberType NoteProperty `
             -Name "UpdateDays" `
             -Value $AUDays[$AUSettings.ScheduledInstallationDay]

  Add-Member -inputObject $AuObj `
             -MemberType NoteProperty `
             -Name "UpdateHour" `
             -Value $AUSettings.ScheduledInstallationTime

  Add-Member -inputObject $AuObj `
             -MemberType NoteProperty `
             -Name "Recommended updates" `
             -Value $(If ($AUSettings.IncludeRecommendedUpdates)
                      {"Included"}  else {"Excluded"})

  Return ,$AuObj

}  #---------- End Function Get-WindowsUpdateConfig ---------------------

Function Test-BatteryHealth { 
    $fullchargecapacity = (Get-WmiObject `
                               -Class "BatteryFullChargedCapacity" `
                               -Namespace "ROOT\WMI").FullChargedCapacity 
    $designcapacity = (Get-WmiObject `
                               -Class "BatteryStaticData" `
                               -Namespace "ROOT\WMI").DesignedCapacity 
 
    if ($fullchargecapacity -eq $designcapacity) { 
        $Batteryhealth=-1
    } 
    Else {
          $batteryhealth = ($fullchargecapacity / $designcapacity) * 100 
          if ($batteryhealth -gt 100) {$batteryhealth = 100} 
    }
    return [decimal]::round($batteryhealth)  

} #----------------  End Test-BatteryHealth -----------------------------

Function ValidateWinVer {

  Param (
         [Parameter (Mandatory=$True)]
           [Int] $MajorVer,
         [Parameter (Mandatory=$False)]
           [Int] $MinorVer = 0
  )

  $CurOS = Get-CimInstance Win32_OperatingSystem
  $VersionItems = $CurOS.Version.split('.')

  If (([int]$VersionItems[0] -lt $MajorVer) -or `
      (([int]$VersionItems[0] -eq $MajorVer) -and 
       ([int]$VersionItems[1] -lt $MinorVer))) {
    $Message = "You are running Windows " + 
      "$($VersionItems[0]).$($VersionItems[1])`n`nThis " +
      "program requires at least Windows $MajorVer.$MinorVer to run."

     [Windows.Forms.MessageBox]::Show($Message,"Program Terminated:", `
     [Windows.Forms.MessageBoxButtons]::OK , `
     [Windows.Forms.MessageBoxIcon]::Information)

    Exit
  } #End If

} #------------------- End ValidateWinVer -------------------------------

#-------------------------------- Add a Helper --------------------------

$showWindowAsync = Add-Type –memberDefinition @”
[DllImport("user32.dll")]
public static extern bool ShowWindowAsync(IntPtr hWnd, int nCmdShow);
“@ -name “Win32ShowWindowAsync” -namespace Win32Functions –passThru

Function Show-PowerShell() {
     [void]$showWindowAsync::ShowWindowAsync(
                          (Get-Process –id $pid).MainWindowHandle, 10)
}  #End Function Show-PowerShell

Function Hide-PowerShell() {
    [void]$showWindowAsync::ShowWindowAsync(
                          (Get-Process –id $pid).MainWindowHandle, 2)
}  #End Function Hide-PowerShell

<#
  +------------------------------------------------------------+
  |                        Main Program                        |
  +------------------------------------------------------------+
#>

$PGMVersion = 14.00   #--- Remember to Update! ----
$compname   = $env:COMPUTERNAME

Add-Type -AssemblyName  System.Windows.Forms
Add-Type -AssemblyName  System.Drawing

Hide-Powershell       #--- Hide the CMD Window ---
Clear-Host
ValidateWinVer -MajorVer 6 -MinorVer 3

# Created Computer Information Table (CIT)
  $CITable = New-Object -TypeName `
                         System.Data.DataTable "Computer Information"
#Create Columns for table
  $CITItem  = New-Object -TypeName System.Data.DataColumn Item,([string])
  $CITValue = New-Object -TypeName `
                          System.Data.DataColumn Value,([string])
#Add Columns to table
  $CITable.columns.add($CITItem)
  $CITable.columns.add($CITValue)

If ( (Get-AdminStatus) -eq "User") {
  $AdminPriv = $False
}
Else {
  $AdminPriv = $True
}

GenerateForm

#  ------------------ End Main Program ----------------------------