﻿<#
.SYNOPSIS
   Produce information about the computer in a tabbed dialog.
   Similar to Piriform's Speccy or BelArc Advisor

.DESCRIPTION
   CMsPCInfo.ps1 provides information on your hardware; windows;
   video; storage; network; printers; drivers; programs; and
   selected registry/service settings. 

.PARAMETER ComputerName
    A single Computer or an array of computer names.
    The default is localhost ($env:COMPUTERNAME).

.PARAMETER Quiet
    Switch if present output & errors written to files.

.PARAMETER ClearLog
    Switch if present error log is cleared BEFORE processing
    the computer list.

.OUTPUTS
    Optional output to File: Written to users default Documents Directory
    With button to write to a text file.

.NOTES
  Computer Mentors System Information Dialog V18.10
  Programmed by    : RetiredGeek @ Windows Secrets Lounge (WSL)
  Code adapted from:
           Powershell PC Info Script V1.0b
              Coded By:Trenton Ivey(kno)
                     hackyeah.com
  Incorporated Coding changes recommed by Cafed00d @ WSL

  Updated: 
     09/03/15 Single GenerateTab Function with
              proper argument passing syntax!
     09/10/15 Added new information & fixed format
              of some output
     09/14/15 Fixed Monitor reporting problem
     10/23/15 Added check for Administrator Priv.
     10/25/15 Expanded Storage Information
     10/28/15 Added physical drive information
     10/29/15 Get-Drives & Combine-Object Functs
     11/21/15 Added Win10 Telemetry Service
     12/15/15 Added Write to file option button
     12/19/15 Added Comment based help
     12/21/15 Reorged code to movee data collection from Main to
              the GenerateForm relative tab.
     12/30/15 Expanded the .Net information.
     01/21/16 Fixed UpTime for more than 24 hrs.
     02/08/16 Solved problem with Table display field width via
              a Function to calculate longest value in field.
     02/26/16 Added Partition information re Alignment and TRIM Status.
     03/05/16 Added status of OS Compacted.
     03/09/16 Sort Partitions in Disk Info and Indicate Boot Status.
     03/11/16 Added test for Min Win version to run. (8.1)
              Last Full Battery Tab info only shows on Battery Devices.
     03/27/16 Added Environment Variables tab and reordered tabs and
              moved tab code into Functions for ease of editing.
     05/03/16 Cleaned up Error message to show actual Windows version
              vs NT version! Also removed Exit from function and 
              Returned True if passed and False if failed, check
              in main program for clarity of execution.
     05/05/16 Added tests for individual items to allow to run on
              Windows 7+!
     05/12/16 Added Firmware Detection code (UEFI/BIOS). Fixed
              logic errors on OS Version checking.
     05/23/16 Added Test-SSD function to make SSD detection 
              more robust. Thanks Grant Carthew Via Git-Hub
     05/24/16 Added function for BalloonTips an message to show
              program is running after PS window is hidden.
     06/06/16 Addes support to running commands against remote
              computers using PSRemoting.
     06/10/16 Added check to insure $CompName argument is avaialble on
              the network and to automatically add it to the trusted
              server list! Fixed some mis-reported items on remotes.
     06/18/16 Cleaned up code to run properly on local machine! (V17.95)
     06/19/16 Added Multi-Machine support. (V18.02)
     06/20/16 Added Quiet & ClearLog switchs (V18.04)

.Link
  http://windowssecrets.com/forums/showthread.php/173483

.EXAMPLE
    PS C:\>CMsPCInfo.ps1

    Returns data for the local computer.

.EXAMPLE
    PS C:\>CMSPCInfo.ps1 -Computers "DellXPS8700"

.EXAMPLE
    PS C:\>CMSPCInfo.ps1 -Computers @("DellXPS8700","DellXPS14z",...)

    Returns data for all reachable computers in the list that are
    configured for remote access.

.EXAMPLE
    PS C:\>CMSPCInfo.ps1 -Computers @("DellXPS8700","DellXPS14z",...) -Quiet

    Produces no screen output other than balloon-tips and local
    machine startup messages. Data is written to computer specific
    files named for the computer and placed in the user's 
    Documents file. Error data is time stamped and written to the
    CMsPCInfo-Errors.log also placed in the users Documents file.

.EXAMPLE
    PS C:\>CMSPCInfo.ps1 -Computers @("DellXPS8700","DellXPS14z",...) -Quiet -ClearLog

    Same as previous example but clears the Error Log before starting.

#>


#NOTE: TWO blank lines above required for comment based help!

Param (
  [Parameter(Mandatory=$False)]
    [String[]] $Computers = "$Env:COMPUTERNAME",
  [Parameter(Mandatory=$False)]
    [Switch] $Quiet,
  [Parameter(Mandatory=$False)]
    [Switch] $ClearLog
)
Function GenerateForm {

#---------------------- Form Objects ----------------------------
$CMsForm      = New-Object -TypeName System.Windows.Forms.Form
$statusBar1   = 
    New-Object -TypeName System.Windows.Forms.StatusBar
$lblMainTitle = New-Object -TypeName System.Windows.Forms.Label
$btnExit      = New-Object -TypeName System.Windows.Forms.Button
$btnWriteFile = New-Object -TypeName System.Windows.Forms.Button
$MainTabCtrl  = 
    New-Object -TypeName System.Windows.Forms.TabControl

#---------------------- End Form Objects ------------------------

#----------------------------------------------------------------
#                     Event Handler Script Blocks
#----------------------------------------------------------------

$btnWriteFile_OnClick = {
 
   Write-ComputerInfoFile
  
   $StatusBar1.text = "Data written to: $($InfoFileName)"
   $btnWriteFile.Visible = $False
}

$btnExit_OnClick=
{
 $CMsForm.Close()
}

$OnLoadForm_StateCorrection=
{
  #  Correct the initial state of the form to prevent the
  #  .Net maximized form issue
$CMsForm.WindowState = $IntFrmWinState
}

#---------------- End Event Handler Script Blocks ---------------

#------------------------- Main Form ----------------------------

#--- Form Drawing References ---

$Form_Width       = 1050
$Form_Height      = 600
$MainTitleHeight  =  35
$StatusBar_Height =  25
$ExitBtn_Height   =  30
$tabWidth         = $Form_Width-20
$tabHeight        = $Form_Height - $StatusBar_Height + 
                                   $MainTitleHeight  + 20
$tboxWidth        = $tabWidth-20
$tboxHeight       = 430
$tabSize = New-Object -TypeName System.Drawing.Size(
                                $tabWidth,$tabHeight)
$tabLocation = 
      New-Object -TypeName System.Drawing.point(4,22)
$tboxFont = New-Object -TypeName System.Drawing.Font(
                                 "Courier New",20,1,0)
$tboxSize = New-Object -TypeName System.Drawing.Size(
                               $tboxWidth,$tboxHeight)
$tboxLocation     = 
      New-Object -TypeName System.Drawing.Point(10,10)
$tboxForeColor    = [Drawing.Color]::Yellow
$tboxBackColor    = [Drawing.Color]::Blue

$System_Windows_Forms_Padding = 
                New-Object -TypeName System.Windows.Forms.Padding
$System_Windows_Forms_Padding.All = 0

$CMsForm.ControlBox = $False
$CMsForm.ClientSize = New-Object -TypeName System.Drawing.Size(
                                $($Form_Width),$($Form_Height))
$CMsForm.Name       = "CMsForm"
$CMsForm.BackColor  = [Drawing.Color]::DarkGray
$CMsForm.ForeColor  = [Drawing.Color]::White
$CMsForm.StartPosition =
        [System.Windows.Forms.FormStartPosition]::CenterScreen
$CMsForm.Text = "Computer Mentor's Computer Information " +
                "Version $PGMVersion Running with " +
                "$(Get-AdminStatus) privledges" 
$CMsForm.FormBorderStyle = 
               [System.Windows.Forms.FormBorderStyle]::Fixed3D

# --- Status Bar ---

$statusBar1.Name     = "statusBar1"
$statusBar1.Location = 
     New-Object -TypeName System.Drawing.Point(
                         $($Form_Height-$StatusBar_Height),0)
$statusBar1.Size  = New-Object -TypeName System.Drawing.Size(
                       $($Form_Width-40),$($StatusBar_Height))
$statusBar1.Font     = 
     New-Object -TypeName System.Drawing.Font("Verdana",14,1,0)
$statusBar1.TabIndex = 8
$statusBar1.Text     = "Ready"

$CMsForm.Controls.Add($statusBar1)

#------------- Main Title -------------

$lblMainTitle.Name      = "lblMainTitle"
$lblMainTitle.Location  = 
              New-Object -TypeName System.Drawing.Point(20,15)
$lblMainTitle.Size      = 
      New-Object -TypeName System.Drawing.Size($(
                               $Form_Width-40),35)
$lblMainTitle.TextAlign = 'MiddleCenter'
$lblMainTitle.TabIndex  = 7
$lblMainTitle.Text      = "Computer Information for: $CompName"
$lblMainTitle.Font      = 
     New-Object -TypeName System.Drawing.Font("Verdana",26,1,0)
$lblMainTitle.BackColor = [Drawing.Color]::DarkGray
$lblMainTitle.ForeColor = [Drawing.Color]::Blue

$CMsForm.Controls.Add($lblMainTitle)

#------------  WriteFile Button ----------------

$btnWriteFile.Name      = "btnWriteFile"
$btnWriteFile.Location  = 
  New-Object -TypeName System.Drawing.Point(10,$($Form_Height -
                          $StatusBar_Height - $ExitBtn_Height))
$btnWriteFile.Size      = 
               New-Object -TypeName System.Drawing.Size(185,35)
$btnWriteFile.TabIndex  = 6
$btnWriteFile.Text      = "Write to File"
$btnWriteFile.BackColor = [Drawing.Color]::Green
$btnWriteFile.ForeColor = [Drawing.Color]::White
$btnWriteFile.Font      = $tboxFont

$btnWriteFile.add_Click($btnWriteFile_OnClick)
$CMsForm.Controls.Add($btnWriteFile)

#------------  Exit Button ----------------

$btnExit.Name      = "btnExit"
$btnExit.Location  = 
  New-Object -TypeName System.Drawing.Point($(
                   $Form_Width-80),$($Form_Height - 
		           $StatusBar_Height - $ExitBtn_Height))
$btnExit.Size = New-Object -TypeName System.Drawing.Size(75,35)
$btnExit.TabIndex  = 6
$btnExit.Text      = "Exit"
$btnExit.BackColor = [Drawing.Color]::Red
$btnExit.ForeColor = [Drawing.Color]::White
$btnExit.Font      = $tboxFont

$btnExit.add_Click($btnExit_OnClick)
$CMsForm.Controls.Add($btnExit)

#----------- Main Tab Control ---------------------

$MainTabCtrl.Name      = "MainTabControl"
$MainTabCtrl.Location  = 
                 New-Object -TypeName System.Drawing.Point(10,60)
$MainTabCtrl.Size      = 
      New-Object -TypeName System.Drawing.Size($($Form_Width-20), 
					    $($Form_Height - $StatusBar_Height - 90))
$MainTabCtrl.ShowToolTips  = $True
$MainTabCtrl.TabIndex      = 4
$MainTabCtrl.Font          =  
        New-Object -TypeName System.Drawing.Font("Tahoma",20,2,0)

$CMsForm.Controls.Add($MainTabCtrl)

$CompSysObj =
    Get-WMIObject -computer $CompName -Class Win32_ComputerSystem


$HWInfo       = HardwareTab
$VideoInfo    = VideoTab
$StorageInfo  = StorageTab
$NetworkInfo  = NetworkTab
$PrinterInfo  = PrinterTab
$WindowsInfo  = WindowsTab
$EnvironInfo  = EnvironmentTab
$DriversInfo  = DriversTab
$InstPgmsInfo = ProgramsTab
$ServicesInfo = RegistryServicesTab
$BatteryInfo  = BatteryTab


#--------------------   End Form Code  ------------------------

  #Save the initial state of the form
  $IntFrmWinState = $CMsForm.WindowState
  #Init the OnLoad event to correct initial state of the form
  $CMsForm.add_Load($OnLoadForm_StateCorrection)

  #Show the Form or Write File
  If ($IsQuiet) {
      Write-ComputerInfoFile
  }
  Else {
    $CMsForm.ShowDialog() | Out-Null
  }
} #------------------ End Function GenerateForm ---------------

Function HardwareTab {

  $BTArgs = @{Text    = "Investigating Computer Setup"
            Title   = "I'm Working on $CompName"
            Icon    = 'Info'
            Timeout =   [Int32]500}
  Show-BalloonTip  @BTArgs

  $enclosureNames = (
    "unknown",  # 0
    "Other" ,
    "Unknown" ,
    "Desktop" ,
    "Low Profile Desktop" ,
    "Pizza Box" ,  #5
    "Mini Tower" ,
    "Tower" ,
    "Portable" ,
    "Laptop" ,
    "Notebook" , #10
    "Hand Held" ,
    "Docking Station" ,
    "All-in-One" ,
    "Sub Notebook" ,
    "Space Saving" ,  #15
    "Lunch Box" ,
    "Main System Chassis",
    "Expansion Chassis",
    "Sub-Chassis",
    "Bus Expansion Chassis", #20
    "Peripheral Chassis" ,
    "Storage Chassis" ,
    "Rack Mount Chassis" ,
    "Sealed-Case PC" #24
  )

  Add-Row -ARIN 'Manufacturer' -ARIV $CompSysObj.Manufacturer
  Add-Row -ARIN 'Model'        -ARIV $CompSysObj.Model
  Add-Row -ARIN 'System Type'  -ARIV $CompSysObj.SystemType

  $WMIArgs = @{Class    = 'Win32_SystemEnclosure'
               Computer = $CompName}
  $enclosure = Get-WMIObject @WmiArgs

  $ChassisNo = $enclosure.ChassisTypes[0]
  $ChassisName =  if ($ChassisNo -ge $enclosureNames.length) {
                         "Currently Unassigned"
                  }
                  else {
                         $enclosureNames[$ChassisNo]
                  }

  Add-Row  -ARIN 'Enclosure' -ARIV $ChassisName

  Add-Row  -ARIN ''

  $WMIArgs = @{Class    = 'Win32_BIOS'
               Computer = $CompName}
  $BIOSObject =  Get-WMIObject @WMIArgs

  Add-Row -ARIN 'BIOS Serial No' -ARIV $BIOSObject.SerialNumber
  Add-Row -ARIN '     Name'      -ARIV $BIOSObject.Name
  $ARAgs = @{ARIN = '     Version'
             ARIV = $BIOSObject.SMBIOSBIOSVersion}
  Add-Row @ARAgs
  $ARAgs = @{ARIN = '     Date'
             ARIV = $BIOSObject.ReleaseDate.Substring(0,8)}
  Add-Row @ARAgs
  $ARAgs = @{ARIN = '     Manufacturer'
             ARIV = $BIOSObject.Manufacturer}
  Add-Row @ARAgs

  $WMIArgs = @{Class    = 'Win32_BIOS'
               Namespace = 'Root\CIMV2'
               Computer = $CompName}
  $BV = Get-WmiObject @WMIArgs |
        Select-Object -Property BIOSVersion

  If ($BV.BIOSVersion.count -eq 3) {
    $Separator = "-"
    $BIOSManuf = 
       $BV.BIOSVersion[2].split($Separator)  #BIOS Manufacturer
    Add-Row -ARIN '     Creator'  -ARIV $BIOSManuf[0]
  }

  If ($ServerOS -ge 8.0) {

    Add-Row -ARIN ''

#--- PS InLine Switch construct! ---
    $FT= (&{ 
    Switch (Get-BiosType -CompName "$CompName") {
          1       {"Legacy BIOS"}
          2       {"UEFI"}
          Default {"Unknown"}
    }})
    Add-Row -ARIN 'Firmware Type' -ARIV "$FT"

    If ($FT -eq 'UEFI') {
      If ($AdminPriv) {
        $ICArgs = @{ComputerName = "$CompName"
                    ScriptBlock = 
           {Confirm-SecureBootUEFI -ErrorAction "SilentlyContinue"}}
        $UEFISB = Invoke-Command @ICArgs
    
        If (Test-Path variable:UEFISB ){
          If ($UEFISB) {$UEFIStatus="On"} Else {$UEFIStatus="OFF"}
          Add-Row -ARIN 'Secure Boot UEFI'  -ARIV $UEFIStatus
        }
        Else {
             Add-Row -ARIN 'Secure Boot UEFI' -ARIV "Not Supported"
        }
      }   #End If ($AdminPriv)
    
      Else {
             $ARArgs = @{ARIN = 'Secure Boot UEFI' 
                         ARIV = "Requires Admin Access to verify!"}
             Add-Row @ARArgs
      } 

    } #End If ($FT -eq 'UEFI')
  } #End If ($ServerOS -ge 8.0...

  Add-Row -ARIN ''

  $CPU_Object = 
     Get-WMIObject -computer $CompName -Class Win32_Processor

  Add-Row -ARIN 'Processor Name'  -ARIV $CPU_Object.Name
  Add-Row -ARIN '  Info'          -ARIV $CPU_Object.Caption
  Add-Row -ARIN '  Maker'     -ARIV $CPU_Object.Manufacturer
  Add-Row -ARIN '  ID'            -ARIV $CPU_Object.ProcessorID
#------ Examples of Splatting ------------------
  $ARArgs = @{ARIN = '  Physical Cores' 
              ARIV = $CPU_Object.NumberofCores}
  Add-Row @ARArgs
  $ARArgs = @{ARIN = '  Logical  Cores'
              ARIV = $CPU_Object.NumberofLogicalProcessors}
  Add-Row @ARArgs
  $ARArgs = @{ARIN = '  Address Width'
              ARIV = $CPU_Object.AddressWidth}
  Add-Row @ARArgs
#------ End Splatting Example

  $HyperThreadingStatus = 
        if ($CPU_Object.NumberOfCores -le
            $CPU_Object.NumberofLogicalProcessors)
                      {'Enabled'} Else {'Disabled'}
  $ARArgs = @{ARIN = '  HyperThreading'
              ARIV = $HyperThreadingStatus}
  Add-Row @ARArgs 
 
  If ($ServerOs -ge 8.0) {
    $VMFirmEnabled = 
       If ($CPU_Object.VirtualizationFirmwareEnabled)
           {'Enabled'} Else {'DisAbled'}
    $ARArgs = @{ARIN = '  VM Firmware'
                ARIV = "$VMFirmEnabled"}
     Add-Row @ARArgs
  }

  $HWWidth = Get-MaxLength -TestObj $CITable.Item -MinLen 20

  $fmtBattery = 
    @{Expression={$_.Item};Label="Item  ";Width=$HWWidth},
    @{Expression={$_.Value};Label="Value";Width=78-$HWWidth}

  $HWInfo = 
     $CITable | Format-Table -Property $fmtBattery | Out-String

#Memory Info

  $WMIArgs = @{Computer = $CompName 
               Class = 'Win32_PhysicalMemory'}

  $MemoryInfo   = Get-WMIObject @WMIArgs  |
                  Sort-Object  -Property {$_.DeviceLocator}

  $ManufLen = 
     Get-MaxLength -TestObj $MemoryInfo.Manufacturer -MinLen 12

  $fmtMEM = 
    @{Expression={$_.DeviceLocator.Trim('DIMM')};
                   Label="MB`nSlot";align='center';Width=4},
    @{Expression={$_.BankLabel.Trim('BANK ')};
                   Label="`nBank";Align='center';Width=4},
    @{Expression={$_.InterleavePosition};
                   Label="Bank`nPos";align='center';Width=4},
    @{Expression={$_.Speed};Label="`nSpeed";Width=5},
    @{Expression={$_.DataWidth};Label="Data`nWidth";Width=5},
    @{Expression={'{0:#.00}' -f ($_.Capacity/1gb)};
                   Label="Size`n/ GB";Width=5;align='right'},
    @{Expression={$_.Manufacturer};Label="`nManufacturer";
                                           Width=$ManufLen},
    @{Expression={$_.SerialNumber};Label="`nSerial No.";
                                   Width=10}

  $MemoryInfo = $MemoryInfo | Format-Table -Property $fmtMEM |
                              Out-String

  $MemTitle     =  "Memory Information:" | Out-String

  $tboxHardware = 
              New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {
    $tabHardware  = 
                New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName  = [ref]$tabHardware
                tboxObjectName = [ref]$tboxHardware
                tabName        = 'Hardware'
                TipText = 
                       'Computer, BIOS/UEFI, Processor, Memory'}
    GenerateTab @GTArgs
  }

  $tboxHardware.Text = $HWInfo + $MemTitle + $MemoryInfo

  Return ,$tboxHardware.Text

} # ------------------   End HardwareTab  ---------------------

Function VideoTab {

  $Video = Get-WmiObject -Class Win32_VideoController `
                         -ComputerName $compname

  $VideoTabStartCnt = $CITable.Rows.Count
  Add-Row -ARIN 'Video Name' -ARIV $Video.Name
  Add-Row -ARIN '  RAM (Mb)' -ARIV ($Video.AdapterRAM/1mb)
  Add-Row -ARIN '  Mode Description' `
          -ARIV  $Video.VideoModeDescription
  Add-Row -ARIN '  Refresh Rate' `
          -ARIV $Video.CurrentRefreshRate
  Add-Row -ARIN '  Installed Drivers' `
          -ARIV $Video.InstalledDisplayDrivers
  Add-Row -ARIN '  Driver Date' `
          -ARIV ($Video.ConvertToDateTime($Video.DriverDate))
  Add-Row -ARIN '  Driver Version' -ARIV $Video.DriverVersion

  $WMIArgs = @{Computer = "$CompName"
               Query =
       'select * from Win32_PnPEntity where service="monitor"'}
  $Mons = Get-WmiObject @WmiArgs |
         Select-Object -Property Name, Manufacturer, HardwareID

  $AMArgs = @{Name       = 'SerialNo' 
              MemberType = 'NoteProperty' 
              Value      = $null}
  $Mons | Add-Member @AMArgs

ForEach ($xx in $Mons) {

  $ErrorActionPreference = 'SilentlyContinue'

   $Separator = "{","\","}"
   $xx.HardwareID = $xx.HardwareId.split($Separator)[1]

   $keytype=[Microsoft.Win32.RegistryHive]::LocalMachine
   $reg=[Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey(
                                              $keytype,"")
   $regKey = $reg.OpenSubKey(
       "SYSTEM\\CurrentControlSet\\Enum\DISPLAY" )
   $regKey = $reg.OpenSubKey(
       "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\$(
                                    $xx.HardwareId)")
   $DID = $regkey.GetSubKeyNames()

   ForEach($DID_KEY_NAME in $DID) {

         $regKey= $reg.OpenSubKey(
           "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\" +
           "$($xx.HardwareId)\\$($DID_KEY_NAME)" +
           "\\Device Parameters")
         $EDID = $regKey.GetValue("EDID")

         ForEach($int in $EDID){
           $EDID_String = $EDID_String+([char]$int)
         }

         #Serial Number
         $checkstring = [char]0x00 + [char]0x00 + [char]0x00 + 
                        [char]0xFF + [char]0x00
         $matchfound =  $EDID_String -match "$checkstring(\S+)"
         if($matchfound){
           $xx.SerialNo = [String]$matches[1]}
         else {
               $xx.SerialNo = '-'
         }

         $EDID_String = ''

   }   # End - foreach($DID_KEY_NAME in $DID)

}   #  End ---- ForEach $xx ----

  $ErrorActionPreference = 'Continue'

  $MNLen = Get-MaxLength -TestObj $Mons.Name         -MinLen 12
  $MFLen = Get-MaxLength -TestObj $Mons.Manufacturer -MinLen 12

  $monfmt  = @{Expression={$_.Name};
                 Label="Monitor Name";Width=$MNLen},
             @{Expression={$_.Manufacturer};
                 Label="Manufacturer";Width=$MFLen},
             @{Expression={$_.SerialNo};
                 Label="Serial Number";Width=15},
             @{Expression={$_.HardWareID};
                 Label="Registry ID";Width=15}
  $TOArg = @{TestObj = 
    $CITable.Item[$VideoTabStartCNt..$($CITable.Item.Count-1)]}
  $VidWidth = (Get-MaxLength @TOArg) + 3

  $fmtVid = @{Expression={$_.Item};
                Label="Item  ";Width=$VidWidth},
            @{Expression={$_.Value};
                Label="Value";Width=78-$VidWidth}

  $VideoHW    = 
     $citable.rows[$VideoTabStartCnt..$(
                           $CITable.Rows.Count - 1)] |
     Format-Table -Property $fmtVid -Wrap | Out-String

  $MonTitle   = "Monitor Information:" | Out-String

  $MonInfo = 
         $mons | Format-Table -Property $monfmt | Out-String

  $tboxVideo = 
       New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {
    $tabVideo  =
         New-Object -TypeName System.Windows.Forms.TabPage
  
    $GTArgs = @{tabObjectName = [ref]$tabVideo
                tboxObjectName = [ref]$tboxVideo
                tabName = 'Video'
                TipText = 
                    'Video Adapters, Resolutions, Monitors'}
    GenerateTab @GTArgs
    }

  $tboxVideo.Text = $VideoHW + $MonTitle + $MonInfo

  Return ,$tboxVideo.Text

} #----------------------- End Video Tab -----------------------

Function StorageTab {

  If ($ServerOS -ge 8.1) {

    #Physical Drive Info

   $SBCode = {Get-Disk  | Where-Object {$_.size -gt 0 } |  
              Sort-Object -Property DiskNumber}

    If ($LocalMachine)   {
      $PhyDiskInfo = & $SBCode
    }
    Else {
      $PDIArgs = @{ComputerName = "$CompName" 
                   ScriptBlock  = $SBCode}
      $PhyDiskInfo = Invoke-Command @PDIArgs
    }

  $SSD = $False

  $AMArgs = @{Type  = "NoteProperty"
              Name  = "SSD"
              Value = "No"}
  $PhyDiskInfo| Add-Member @AMArgs

  If ($LocalMachine) {

    ForEach ($x in $phydiskinfo) { 
    
           if (Test-SSD $x.number) { 
  		     $SSD  = $True
             $x.SSD = "Yes"
           } #End If (Test-SSD...
  		  
    } #End ForEach $x
  
      If ($SSD) {
        $TrimVal = 
          $(fsutil behavior query disabledeletenotify).Split('=')
      }

  } #End If ($CompName -eq ...

  Else {
    ForEach ($x in $phydiskinfo) { 
       $SSDArgs = @{Computer = $CompName
                    ScriptBlock = ${function:Test-SSD}
                    ArgumentList = $x.number
                   }

       if (Invoke-Command @SSDArgs) { 
		     $SSD  = $True
         $x.SSD = "Yes"
       } #End If (Test-SSD...
  		  
    } #End ForEach $x
  
      If ($SSD) {
        $TVArgs = @{Computer = $CompName
                    ScriptBlock = 
      {$(fsutil behavior query disabledeletenotify).Split('=')}
                   }
        $TrimVal = Invoke-Command @TVArgs
      }
  } #End Else
                  
#--- PS IIF construct! ---
      $TrimStatus = (&{If([int]$TrimVal[1] -eq 0) 
                         {"Trim is Enabled"}
                    Else {"Trim is Disabled....Fix THIS!"}})
      $TrimStatus = 
@"
[ ---> $TrimStatus <--- ]

"@ | Out-String

  $DNLen = 
    Get-MaxLength -TestObj $PhyDiskInfo.Model        -MinLen 4
  $SNLen = 
    Get-MaxLength -TestObj $PhyDiskInfo.SerialNumber -MinLen 13

  $fmtPhyDisk1 = 
     @{Expression={ '{0:N0}' -f $_.Number};Label="Drive`n No.";Width=5;
                                  Align='Center'},
     @{Expression={$_.Model};Label="`nName";Width=$DNLen},
     @{Expression={$_.SSD};Label="`nSSD";Width=5;Align='left'},
     @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
               Label="Disk Size`n / GB";Width=9;align='right'},
     @{Expression={$_.NumberOfPartitions};Label="Parti`ntions";
                                          Width=5},
     @{Expression={$_.PartitionStyle};Label="GPT`nMBR";Width=3},
     @{Expression={$_.IsBoot};Label="`nBoot";
                              Width=5;Align="Left"},
     @{Expression={$_.BusType};Label="Data`nBus";Width=4}

  $fmtPhyDisk2 = 
     @{Expression={ '{0:N0}' -f $_.Number};Label="Drive`n No.";Width=5;
                                  Align='Center'},
     @{Expression={$_.Model};Label="`nName";Width=$DNLen},
     @{Expression={$_.SerialNumber.Trim()};
         Label="`nSerial Number";Width=$SNLen;Align='left'},
     @{Expression={$_.HealthStatus};Label="`nStatus";Width=7}

  $PhyDiskTitle = "Physical Disk Information:" | Out-String
  
  $PhyDiskInfo1 = $PhyDiskInfo | 
           Format-Table -Property $fmtPhyDisk1 | Out-String
  $PhyDiskInfo2 = $PhyDiskInfo | 
           Format-Table -Property $fmtPhyDisk2 | Out-String

} #End Physical Disk Info Win 8.1 or greater

 #Logical Drive Info

  $DiskTypeHash = @{
   2="Removable"
   3="Fixed"
   4="Network"
   5="Compact"}

  $fmtDisk = 
     @{Expression={$_.Name};Label="Drive`nLetter";Width=6},
     @{Expression={$_.VolumeName};Label="Volume`nName";
                                  Width=10},
     @{Expression={$_.filesystem};Label="File`nSystem";
                                  Width=6},
     @{Expression={$DiskTypeHash.item([int]$_.DriveType)};
                      Label="Drive`nType";Width=9},
     @{Expression={$_.compressed};Label="`nCompressed";
                                  Width=10;Align='left'},
     @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
               Label="Disk Size`n / GB";Width=9;align='right'},
     @{Expression={ '{0:#,000.00}' -f ($_.FreeSpace/1gb)};
             Label="Free Space`n / GB";Width=10;align='right'},
     @{Expression={ '{0:N0}' -f $_.DiskIndex};
             Label="Drv`nNo";Width=4;Align='right'},
     @{Expression={$_.Bootable};Label="`nBoot";
                                Width=5;Align='left'}

  $DiskInfoTitle = "Logical Disk Information:" | Out-String

  $DiskInfo = Get-Drives | 
              Sort-Object  -Property DiskIndex, Name |
              Format-Table -Property $fmtDisk | Out-String

  If ($AdminPriv) {
    $fmtPart = 
       @{Expression={ '{0:N0}' -f ($_.Index)};
           Label="Part`nNo";Width=4;Align='right'},
       @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
           Label="Partition`nSize/GB";Width=9; Align='right'},
       @{Expression={$_.Alignment};
           Label="Part`nAlignment";Width=9; Align='left'},
       @{Expression={
         (&{IF ($_.PrimaryPartition) {"Yes"} else {""}})};
           Label="`nPrimary";Width=7; Align='Center'},
       @{Expression={
         (&{IF ($_.Bootable) {"Yes"} else {""}})};
           Label="`nBootable";Width=8; Align='Center'},
       @{Expression={
         (&{IF ($_.BootPartition) {"Yes"} else {""}})};
           Label="Boot`nPartition";Width=9; Align='Center'}

    $WMIArgs = @{Class    = 'Win32_DiskPartition'
                 Computer = $compname}
    $Partitions = 
       Get-WmiObject @WMIArgs | Sort-Object DiskIndex, Index

    $AMArgs = @{NotePropertyName = 'Alignment' 
                NotePropertyValue = ""}
    $Partitions | 
      Add-Member @AMArgs

    ForEach ($part in $Partitions) {
        $Part.Alignment = (
          &{IF ($Part.StartingOffset % 4096 -eq 0) 
             {"Aligned"} Else {"UnAligned"}}) 
    } #End ForEach ($part...

    $FTArgs = @{InputObject = $Partitions
               Property     = $fmtPart
               GroupBy      = 'Diskindex'}
     $Partitions = Format-Table @FTArgs | Out-String

  } #End If ($AdminPriv)

  Else {
   $Partitions = @"

   This information is Available ONLY when RunAs Administrator

"@ | Out-String
  } #End Else

  $PartInfoTitle = "Disk Partition Information:" | Out-String

#CD/DVD Info

  $CDTitle = "CD/DVD Information:" | Out-String

  $WMIArgs = @{Class    = 'Win32_CDROMDrive'
               Computer = $compname}
  $CD      = Get-WmiObject @WMIArgs

  If ($CD -ne $null) {

    $fmtCD  = @{Expression={$_.Drive};
                  Label="Drive Letter";Width=6},
              @{Expression={$_.Name};Label="`nName";Width=25},
              @{Expression={$_.CapabilityDescriptions};
                  Label="`nCapabilities";Width=40}
    $CDDVDINFO = $CD |  Format-Table -Wrap -Property $fmtCD | 
                        Out-String
  }
  Else{
    #---- All this to get a lousy Line Feed in the output! ---
    $CD        = New-Object PSObject
    $AMArgs    = @{Type  = "NoteProperty"
                   Name  = "Status"
                   Value = 'None-Avaliable'}
    $CD | Add-Member @AMArgs
    $fmtCD     = @{Expression={$_.Status}}
    $CDDVDINFO = $CD | Format-Table $fmtCD -HideTableHeaders | 
                       Out-String
  }

  #Mapped Drives Info

  $MDriveTitle = "Mapped Drive Information:" | Out-String
  $WMIArgs = @{Class    = 'Win32_MappedLogicalDisk'
               Computer = $CompName}

  $MappedDrives = Get-WMIObject @WMIArgs |
       Where-Object {$_.size -ge 0 -and 
                     $_.ProviderName -ne $null }

  If ($MappedDrives -eq $Null) {

    If ($AdminPriv) {
      $MDriveInfo = @"

    --No Mapped Drives--

    If drives were mapped using standard user privileges, 
    even from an administrator account, they will not show  
    up if this script is run as Administrator!

"@

    } #End If ($AdminPriv)

    Else {
          $MDriveinfo = @"

          --No Mapped Drives--

"@
    }

    $MDriveInfo = $MDriveInfo | Out-String

  } #End If ($MappedDrives...)

  Else {
        $fmtMDR = 
         @{Expression={$_.Name};Label="Drive`nLetter";Width=6},
         @{Expression={$_.Compressed};Label="Compressed";
                                      Width=5},
         @{Expression={$_.ProviderName};Label="Provider";
                                        Width=50},
         @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
               Label="Disk Size`n / GB";Width=9;align='right';}

        $MDriveInfo = $MappedDrives | 
                      Format-Table -Property $fmtMDR | 
                      Out-String

  } #End Else ($MappedDrives...)

#Shares Info

  $fmtShare   = 
    @{Expression={$_.Name};Label="`nShare Name";Width=12},
    @{Expression={$_.PSComputerName};Label="`nComputer Name";
                                     Width=15},
    @{Expression={$_.MaximumAllowed};Label="Max`nUsers";
                                     Width=5;align='Left'},
    @{Expression={$_.Path};Label="`nPath";Width=35}

  $ShareTitle = "Share Information:" | Out-String

  $WMIArgs = @{Class = 'Win32_Share'
               ComputerName = $CompName}
  $ShareInfo  = Get-WmiObject  @WMIArgs |
                Where-Object {$_.Type -eq 0} | 
                Sort-Object  -Property Name |
                Format-Table -Property $fmtShare | Out-String
  $tboxStorage = 
      New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {
    $tabStorage  = 
        New-Object -TypeName System.Windows.Forms.TabPage
  
    $GTArgs = @{tabObjectName = [ref]$tabStorage
                tboxObjectName = [ref]$tboxStorage
                tabName = 'Storage'
                TipText = 
                 'Drives, CD/DVD/BlueRay, Mapped Drives, Shares'}
  
    GenerateTab @GTArgs
  }

  If ((($ThisPCWinVer -ge 8.1) -and ($LocalMachine)) -or 
      ((-not $LocalMachine) -and ($ServerOS -ge 8.1))) {

    $tboxStorage.Text = 
      $PhyDiskTitle  + $PhyDiskInfo1 + $PhyDiskInfo2 +
      $PartInfoTitle + $Partitions   + $TrimStatus   
  }
 
    $tboxStorage.Text = $tboxStorage.Text +
      $DiskInfoTitle  + $DiskInfo     +
      $CDtitle        + $CDDVDINFO    + 
      $ShareTitle     + $ShareInfo    +
      $MDriveTitle    + $MDriveInfo
  

  Return ,$tboxStorage.Text

} #---------------- End Function Storage Tab ------------------

Function NetworkTab {

  If ($ServerOS -le 8.0) {
  
    $NetTable = @()  #Empty Array for Adapter Objects

    $fmtAdapter = 
        @{Expression={$_.Index};
            Label="Adapter`n Index";Width=7},
        @{Expression={$_.NICName};
            Label="`nAdapter Name"; Width= 50},
        @{Expression={ '{0:#,000}' -f ($_.Speed/1000000)};
            Label="Speed`n / MB";Width=5;Align='right'}

    $fmtNetIP = 
        @{Expression={$_.Index};
            Label="Adapter`n Index";Width=7},
        @{Expression={$_.MacAddr};
            Label="`nMAC Address"; Width= 17},
        @{Expression={$_.IPv4};   
            Label="`nIPv4 Address"; Width=15},
        @{Expression={$_.IPv6};   
            Label="`nIPv6 Address"; Width=25}

    $AdapterArgs = @{Class        = "win32_networkadapter" 
                     ComputerName = "$CompName" 
                     Filter       = "NetConnectionStatus = 2"}
    $Adapters = get-wmiobject @AdapterArgs  

    $IPConfigSetArgs = 
        @{Class        = "Win32_NetworkAdapterConfiguration"
          ComputerName = "$CompName" }
    $IPConfigSet = Get-WmiObject @IPConfigSetArgs

    ForEach ($Adptr in $Adapters) {

       $AdptrObj = New-Object -TypeName PSObject

       ForEach ($IP in $IPConfigSet) {

          If ($IP.Index -eq $Adptr.Index) {
            $AMArgs = @{Type  = "NoteProperty"
                        Name  = "Index"
                        Value = $Adptr.Index}
            $AdptrObj | Add-Member @AMArgs

            $AMArgs = @{Type  = "NoteProperty"
                        Name  = "NICName"
                        Value = $Adptr.Name}
            $AdptrObj | Add-Member @AMArgs

            $AMArgs = @{Type  = "NoteProperty"
                        Name  = "MACAddr"
                        Value = $IP.MACAddress}
            $AdptrObj | Add-Member @AMargs

            $AMArgs = @{Type  = "NoteProperty"
                        Name  = "Speed"
                        Value = $Adptr.Speed}
            $AdptrObj | Add-Member @AMargs

            ForEach ($IPAddr in $IP.IPAddress) {

              If ($IPAddr.Length -lt 25) {
                $AMArgs = @{Type  = "NoteProperty"
                            Name  = "IPv4"
                            Value = $IPAddr
                            ErrorAction = 'SilentlyContinue'}
               $AdptrObj | Add-Member @AMArgs 
              }
              Else {
                $AMArgs = @{Type  = "NoteProperty"
                            Name  = "IPv6"
                            Value = $IPAddr
                            ErrorAction = 'SilentlyContinue'}
               $AdptrObj | Add-Member @AMArgs 
              }

            } #End ForEach ($IPAddr...

          } #End IF ($IP.Index

       } #End ForEach ($IP...

       $NetTable += $AdptrObj

    } #End ForEach ($adptr...

    $NetInfo = $NetTable | Format-Table $fmtAdapter | Out-String
    $IPInfo  = $NetTable | Format-Table $fmtNetIP   | Out-String
  }

  Else {
    $SBCode = {Get-NetAdapter | 
               Where-Object {$_.Name -ne $Null}}

    If ($LocalMachine)   {
      $NetInfo = & $SBCode 
    }
    Else {
      $NAArgs = @{ComputerName = "$CompName" 
                  ScriptBlock  = $SBCode}
  
      $NetInfo = invoke-command @NAArgs 
    }

    $NameLen = Get-MaxLength -TestObj $NetInfo.Name -MinLen 10 
    $IFDLen = 80 - $NameLen - 20

    $fmtNet = 
       @{Expression={$_.Name};Label="Connection`nName";
                              Width=$NameLen},
       @{Expression={$_.Status};Label="`nStatus";Width=12},
       @{Expression={ '{0:#,000}' -f ($_.Speed/1000000)};
           Label="Speed`n / MB";Width=5;Align='right'},
       @{Expression={$_.InterfaceDescription};
           Label="`nAdapter Name";Width=$NameLen}

    $NetInfo = $NetInfo |
           Format-Table -Property $fmtNet -wrap | Out-String

    $fmtNetIP = 
       @{Expression={$_.InterfaceAlias};
           Label='Interface';Width=10},
       @{Expression={$_.AddressFamily};
           Label='IP Width'; Width= 5},
       @{Expression={$_.IPAddress};   
           Label='IP Address'; Width=30}

    $SBCode = {
        $IPArgs = @{ErrorAction = 'SilentlyContinue'
                    InterfaceAlias = "Ethernet","Wi-Fi"}
        Get-NetIPAddress @IPArgs  |
        Sort-Object  -Property InterfaceAlias, AddressFamily
     }

    If ($LocalMachine)   { 
      $IPInfo =  & $SBCode 

    }
    Else {
        $ICArgs = @{ComputerName = "$CompName" 
                    ScriptBlock  = $SBCode}
        $IPInfo = invoke-command @ICArgs     
    }

    $IPInfo = $IPInfo | Format-Table -Property $fmtNetIP | Out-String

  } #End Else

  $tboxNetwork = 
      New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {
    $tabNetwork  = 
        New-Object -TypeName System.Windows.Forms.TabPage
  
    $GTArgs = @{tabObjectName =  [ref]$tabNetwork
                tboxObjectName = [ref]$tboxNetwork
                tabName = 'Network'
                TipText = 'Status of Network Adapters (NICs)'}
    GenerateTab @GTArgs
  }

  $tboxNetwork.Text = "$NetInfo $IPInfo"

  Return ,$tboxNetwork.Text

} #--------------------- End Network Tab ---------------------

Function PrinterTab {

  $fmtPrt = 
      @{Expression={$_.DeviceID};  Label="Printer";Width=34},
      @{Expression={$_.DriverName};Label="Driver"; Width=24},
      @{Expression={$_.PortName};  Label="Port";   Width=20}

  $tboxPrinters = 
       New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {
    $tabPrinters  = 
         New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabPrinters
                tboxObjectName = [ref]$tboxPrinters
                tabName = 'Printers'
                TipText = "Printer Id, Driver and Port Info."}
    GenerateTab @GTArgs
  }

  $tboxPrinters.Text =
      Get-WMIObject  -Computer $CompName -Class Win32_Printer |
      Select-Object  -Property DeviceID,DriverName, PortName |
      Sort-Object    -Property DeviceId |
      Format-Table   -Property $fmtPrt -Wrap | Out-String

  Return ,$tboxPrinters.Text

} #--------------------- End Printers Tab ---------------------

Function WindowsTab {

  $GCIArgs = @{Class    = 'Win32_OperatingSystem'
               Computer = "$CompName"}
  $CurOS   = Get-CimInstance @GCIArgs
  $OSKey   = Get-WindowsProductKey $CompSysObj.name
  $WMIArgs = @{Class    = 'Win32_OperatingSystem'
               Computer = $CompName}
  $OS_Object      = Get-WMIObject @WMIArgs
  $LocalDateTime  = 
        $OS_Object.ConvertToDateTime($OS_Object.LocalDateTime)
  $LastBootUpTime = 
        $OS_Object.ConvertToDateTime($OS_Object.LastBootUpTime)
  $SysRoot        = Get-ChildItem  -Path Env:SystemRoot
  $UserProf       = Get-ChildItem -Path Env:UserProfile

  $PreUpTime      = @([String]($LocalDateTime - 
                               $LastBootUpTime)).split('.')
  If ($PreUptime.count -gt 2) {
    $UpTime = "$($PreUptime[0]) Day(s) $($PreUptime[1])"
  }
  Else {
    $UpTime = $PreUpTime[0]
  }

  $OSCompacted = "Unknown"
  
  If ($ServerOS -ge 10) {

    If ($LocalMachine) {
      $OSCompacted = $(Invoke-Expression "Compact.exe /CompactOS:Query")
    }
    Else {
      $ICArgs = @{ComputerName = "$CompName"
                  ScriptBlock = {Compact.exe /CompactOS:Query}}
      $OSCompacted = Invoke-Command @ICArgs
    } #End Else

    $OSCompacted = (($OSCompacted -like 
        "*system is in the Compact state*").count -eq 1)

  }   #End If ($ServerOS...

  $OSTabStartCNt = $CITable.Rows.Count
  Add-Row -ARIN 'OS Name'         -ARIV $CurOS.Caption
  Add-Row -ARIN '   Version'      -ARIV $CurOS.Version
  Add-Row -ARIN '   Bit-Width'    -ARIV $CurOS.OSArchitecture
  Add-Row -ARIN '   Key'          -ARIV $OSKey
  $ARAgs = @{ARIN = '   Registered Owner'
             ARIV = $CurOs.RegisteredUser}
  Add-Row @ARAgs

  If ( ($(Test-Path variable:CurOS.RegisteredOrganization) -eq
        $False) -or ($CurOS.Organization -eq "")) {
    If ($LocalMachine)   {
      $GIPArgs = @{Path = 
          "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\"
                   Name = 'RegisteredOrganization'}
      $RO = Get-ItemProperty @GIPArgs
    }
    Else {
     $ROArgs = @{ComputerName = "$CompName"
               ScriptBlock  = {
                   $GIPArgs = @{Path = 
          "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\"
                 Name = 'RegisteredOrganization'}
                   Get-ItemProperty @GIPArgs }}
    $RO = Invoke-Command @ROArgs
   }

   $RegOrg = $RO.RegisteredOrganization
  }
  Else {
         $RegOrg = $CurOS.RegisteredOrganization
  }

  $ARAgs = @{ARIN = '   Registered Org'
             ARIV = $RegOrg}
  Add-Row @ARAgs

  Add-Row -ARIN '   System Root'  -ARIV $SysRoot.Value
  $ARAgs = @{ARIN = '   Install Date'
             ARIV = $CurOS.InstallDate -f "MM/dd/yy"}
  Add-Row @ARAgs
  Add-Row -ARIN '   Date/Time'    -ARIV $LocalDateTime
  Add-Row -ARIN '   Last Boot'    -ARIV $LastBootUpTime
  Add-Row -ARIN '   Up Time'      -ARIV $UpTime
  Add-Row -ARIN '   OS Compacted' -ARIV $OSCompacted
  Add-Row -ARIN ' '
  Add-Row -ARIN 'Computer Name'   -ARIV $CompSysObj.Name
  Add-Row -ARIN 'Current User ID' -ARIV $CompSysObj.Username
  Add-Row -ARIN 'User Profile'    -ARIV $UserProf.Value
  Add-Row -ARIN 'Domain Name'     -ARIV $CompSysObj.Domain
  Add-Row -ARIN ' '

  Add-Row -ARIN 'Security Software'

  $SBCode = {
    $FWKey = "System\CurrentControlSet\Services\SharedAccess"
    $FWKey = "$FWKey\Parameters\FirewallPolicy\StandardProfile"
   [Microsoft.Win32.RegistryKey]::
     OpenRemoteBaseKey('LocalMachine',
     "$CompName").OpenSubKey("$FWKey").GetValue(
    "EnableFirewall")
  }

  If ($LocalMachine) {
    $FWallStatus = & $SBCode
  }
  Else {
    $ICArgs = @{ComputerName = "$CompName"
                  ScriptBlock = $SBCode}
  
    $FWallStatus = Invoke-Command @ICArgs
  }

  If ($FWallStatus -eq 1) {
    $FWallStatus = "On"
  }
  Else {
    $FWallStatus = "OFF"
  }

  Add-Row -ARIN "  Firewall"   -ARIV $FWallStatus

  $AV = Get-AntiVirus -CompName $CompName
  Add-Row -ARIN '  AV Program' -ARIV $($AV.displayName)
  Add-Row -ARIN '  AV State'   -ARIV $($AV.productState)
  Add-Row -ARIN ' '

  #--- Windows Update Information ---

  $WU = $("$CompName" | Get-WUsettings ) -split(":")
  
  Add-Row -ARIN 'Windows Update'
  $ARArgs = @{ARIN = '  Notification Level'
              ARIV = $WU[0]}
  Add-Row @ARArgs
  Add-Row -ARIN '  Update Days'        -ARIV $WU[1]
  Add-Row -ARIN '  Update Hour'        -ARIV $WU[2]
  $ARAgs = @{ARIN = '  Recommended Updates'
             ARIV = (&{If($WU[3] -eq $True) 
                         {"Yes"} Else {"No"}})}
  Add-Row @ARAgs             
  Add-Row -ARIN ' '

  #--- PowerShell Information ---

  If ($LocalMachine) {
    $PSVersInfo = $PSVersionTable
    $ExecPolicy = Get-ExecutionPolicy
  }
  Else {
    $PSVArgs = @{Computername = "$CompName"
                 Scriptblock = {$PSVersionTable}}
    $PSVersInfo = Invoke-Command  @PSVArgs
    $EPArgs  = @{Computername = "$CompName"
                 Scriptblock = {Get-ExecutionPolicy}}
    $ExecPolicy = Invoke-Command  @EPArgs
    $ExecPolicy = $ExecPolicy.Value
  }

  Add-Row -ARIN  "PowerShell Information"
  $ARAgs = @{ARIN = '  PS Version'
             ARIV = "$($PSVersInfo.PSVersion)"}
  Add-Row @ARAgs  
  $ARAgs = @{ARIN = '  Runtime Lang. Version'
             ARIV ="$($PSVersInfo.clrversion.Major)." + 
                   "$($PSVersInfo.clrversion.Minor)"}
  Add-Row @ARAgs  
             
  $ARAgs = @{ARIN = '  WS-Man Stack  Version'
             ARIV = $PSVersInfo.WsManStackVersion}
  Add-Row @ARAgs             
    $ARAgs = @{ARIN = '  PS Remoting   Version'
               ARIV = 
       "$($PSVersInfo.PSRemotingProtocolVersion.Major)." +
       "$($PSVersInfo.PSRemotingProtocolVersion.Minor)"}
  Add-Row @ARAgs             
  
  $ARAgs = @{ARIN = '  Execution Policy'
             ARIV = "$ExecPolicy"}
  Add-Row @ARAgs             

#-------------- .Net Information ----------------------

  $DotNetTitle = "Installed .Net Versions:" | Out-String

  $SBCode = {$DNArgs = @{Path = 
        'HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP'}
        Get-ChildItem @DNArgs -recurse |
        Get-ItemProperty -name Version,Release -EA 0 |
        Where { $_.PSChildName -match '^(?!S)\p{L}'} | 
        Sort-Object Version | 
        Select PSChildName, Version, Release}

  If ($LocalMachine) {
     $DotNet = & $SBCode
  }
  Else {
    $ICArgs = @{ComputerName = "$CompName"
                ScriptBlock = $SBCode}
    $DotNet = Invoke-Command @ICArgs
  }

  $NameLen = 
     Get-MaxLength -TestObj $DotNet.PSChildName -MinLen 9
  $VerLen  = 
     Get-MaxLength -TestObj $DotNet.Version     -MinLen 11
  $RNLen   = 
     Get-MaxLength -TestObj $DotNet.Release     -MinLen 11

  $fmtDotNet = 
      @{Expression={$_.PSChildName};Label=".Net Type";
                                    Width=$NameLen},
      @{Expression={$_.Version};Label="Version No:";
                                    Width=$VerLen},
      @{Expression={$_.Release};Label="Release No:";
                                    Width=$RNLen}

  $DOtNet = $DotNet | Format-Table $fmtDotNet | Out-String

  $GMArgs = @{TestObj =
      $CITable.Item[$OSTabStartCNt..$($CITable.Item.Count-1)]}
  $WinWidth = (Get-MaxLength @GMArgs ) + 3

  $fmtWin = 
    @{Expression={$_.Item};Label="Item  ";Width=$WinWidth},
    @{Expression={$_.Value};Label="Value";Width=78-$WinWidth}

  $WinInfo = 
    $citable.rows[$OSTabStartCnt..$($CITable.Rows.Count - 1)] |
    Format-Table -Property $fmtWin | Out-String

  #--- User Account Information ---

  $uai    = Get-UserAccounts -ComputerName $CompName

  $UIDLen = Get-MaxLength -TestObj $uai.Name   -MinLen 6
  $WGLen  = Get-MaxLength -TestObj $uai.Domain -MinLen 9

  $fmtAcct = 
     @{Expression={$_.Name};Label="`nUserID";Width=$UIDLen},
     @{Expression={$_.Domain};
         Label="Domain/`nWorkGroup";Width=$WGLen},
     @{Expression={If ($_.Disabled) {'Disabled'} 
                     Else {'Enabled'}};
         Label="`nStatus";align='left';Width=8},
     @{Expression={If ($_.PasswordRequired) {'   Yes'}
                     ELse {'   No'}};
         Label="|----P A`nRequired";align='left';Width=8},
     @{Expression={If ($_.PasswordExpires) 
                   {'  Yes'} Else {'  No'}};
        Label="S S W O`nExpires";align='left';Width=7},
     @{Expression={If ($_.PasswordChangeable) 
                   {'   Yes'} ELse {'   No'}};
        Label="R D S---|`nChangable";align='left';Width=9}

  $UATitle = "User Account Info:" | Out-String
  $uai   = $uai | Format-Table -Property $fmtAcct | Out-String

  $tboxWindows =  
      New-Object -TypeName System.Windows.Forms.Textbox


  If (-not($IsQuiet)) {
    $tabWindows  = 
        New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabWindows
                tboxObjectName = [ref]$tboxWindows
                tabName = 'Windows'
                TipText = 
          'Windows, Security S/W, PowerShell, .Net, User Accts.'}
  
    GenerateTab @GTArgs
  }
  $tboxWindows.Text = $WinInfo + $DotNetTitle + $DotNet + 
                      $UATitle + $uai

  Return ,$tboxWindows.Text

} #---------------------- End Windows Tab ---------------------

Function EnvironmentTab {

  If ($LocalMachine) {
    $WinEnvVars = Get-ChildItem -Path 'Env:'
  }
  Else {

  $WinEnvVars = 
      Invoke-Command -ComputerName "$CompName" -ScriptBlock {
                  Get-ChildItem  -Path 'Env:' }
  }

  $EnvLen   = Get-MaxLength -TestObj $WinEnvVars.Name -MinLen 10

  $fmtENV = 
    @{Expression={$_.Name};Label="Env Variable";Width=$EnvLen},
    @{Expression={$_.Value};Label="Value";Width=78-$EnvLen}
  $EnvInfo = 
       $WinEnvVars | Format-Table $fmtENV -Wrap | Out-String

  $tboxEnviron = 
      New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {  
    $tabEnviron  = 
        New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabEnviron
                tboxObjectName = [ref]$tboxEnviron
                tabName = 'Environment'
                TipText = 'Windows Environment Variables.'}
  
    GenerateTab @GTArgs
  }
  $tboxEnviron.Text = $EnvInfo

  Return ,$tboxEnviron.Text

} #-------------------- End EnvironmentTab --------------------

Function DriversTab {

  $DIArgs = @{Computer = $CompName 
              Class = 'Win32_PNPSignedDriver'}
  $PT = "*.?.????*.*"
  $DriverInfo = Get-WmiObject @DIArgs |
     Where-Object -Property DriverVersion -notlike -Value $PT |
     Where-Object -Property DeviceName -ne -Value $Null |
     Sort-Object  -Property DeviceName -Unique

  $DVLen = 
    Get-MaxLength -TestObj $DriverInfo.DriverVersion -MinLen 14
  $DDLen = 78 - $DVLen

  $fmtDRVR = @{Expression={$_.DeviceName};
                Label="Driver Description";Width=$DDLen},
             @{Expression={$_.DriverVersion};
                Label="Version Number";Width=$DVLen}

  $DrvTitle = 
     "Non-Windows Unique Drivers and Version Numbers:" |
     Out-String

  $DriverInfo = 
    $DriverInfo | Format-Table -Property $fmtDRVR | Out-String

  $tboxDrivers = 
      New-Object -TypeName System.Windows.Forms.Textbox
    
  If (-not($IsQuiet)) {
    $tabDrivers  = 
        New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabDrivers
                tboxObjectName = [ref]$tboxDrivers
                tabName = 'Drivers'
                TipText = $DrvTitle}
  
    GenerateTab @GTArgs
  }

  $tboxDrivers.Text = $DrvTitle + $DriverInfo

  Return ,$tboxDrivers.text

} #---------------------- End Drivers Tab  --------------------

Function ProgramsTab {

 $fmtPgms = @{Expression={$_.DisplayName};
                Label="Program Name";Width=40},
             @{Expression={$_.InstallDate};
                Label="Installed";Width=9},
             @{Expression={$_.InstallLocation};
                Label="Install Path";Width=31}

  If ($LocalMachine) {

  if (!([Diagnostics.Process]::GetCurrentProcess().Path -match
       '\\syswow64\\'))
{
  $unistallPath = 
    "\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\"
  $unistallWow6432Path =
    "\SOFTWARE\Wow6432Node\Microsoft\Windows\" +
    "CurrentVersion\Uninstall\"

 $PgmsInfo =
  @(
    if (Test-Path   -Path "HKLM:$unistallWow6432Path") {
      Get-ChildItem -Path "HKLM:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKLM:$unistallPath") {
      Get-ChildItem -Path "HKLM:$unistallPath"  }
    if (Test-Path   -Path "HKCU:$unistallWow6432Path") {
      Get-ChildItem -Path "HKCU:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKCU:$unistallPath") {
      Get-ChildItem -Path "HKCU:$unistallPath"  }
   ) |
   Where-Object {
     $_.GetValue('DisplayName') -and
    ($_.GetValue('UninstallString') -or
     $_.GetValue('NoRemove')) -and
    !$_.GetValue('SystemComponent') -and
    !$_.GetValue('ReleaseType') -and
    !$_.GetValue('ParentKeyName')
  } | Get-ItemProperty 

} #End If (!([Diagnostics.Process]::GetCurrentProcess().Path...
Else
{
  $PgmsInfo =
    "You are running 32-bit Powershell on 64-bit system.`n" +
    "Please run 64-bit Powershell instead." 
}

  }
  Else {
$PIArgs = @{ComputerName = "$CompName"
            ScriptBlock  = {
if (!([Diagnostics.Process]::GetCurrentProcess().Path -match
       '\\syswow64\\'))
{
  $unistallPath = 
    "\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\"
  $unistallWow6432Path =
    "\SOFTWARE\Wow6432Node\Microsoft\Windows\" +
    "CurrentVersion\Uninstall\"

 $PgmsInfo =
  @(
    if (Test-Path   -Path "HKLM:$unistallWow6432Path") {
      Get-ChildItem -Path "HKLM:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKLM:$unistallPath") {
      Get-ChildItem -Path "HKLM:$unistallPath"  }
    if (Test-Path   -Path "HKCU:$unistallWow6432Path") {
      Get-ChildItem -Path "HKCU:$unistallWow6432Path"  }
    if (Test-Path   -Path "HKCU:$unistallPath") {
      Get-ChildItem -Path "HKCU:$unistallPath"  }
   ) |
   Where-Object {
     $_.GetValue('DisplayName') -and
    ($_.GetValue('UninstallString') -or
     $_.GetValue('NoRemove')) -and
    !$_.GetValue('SystemComponent') -and
    !$_.GetValue('ReleaseType') -and
    !$_.GetValue('ParentKeyName')
  } | Get-ItemProperty 

} #End If (!([Diagnostics.Process]::GetCurrentProcess().Path...
Else
{
  $PgmsInfo =
    "You are running 32-bit Powershell on 64-bit system.`n" +
    "Please run 64-bit Powershell instead." 
}
  $PgmsInfo
 }} #End PIArgs
  
 $PgmsInfo = Invoke-Command @PIArgs

 } #End Else ($LocalMachine)
  

#Add Member to Object
  $PgmsInfo | 
    Add-Member -NotePropertyName BasePath -NotePropertyValue ""

  $ErrorActionPreference = "SilentlyContinue"

  $PgmsInfo | ForEach-Object {
     $Parts = $_.InstallLocation.Trim('"').Split('\')
     $_.BasePath = "$($Parts[0])\$($Parts[1])"
     $Temp = "\"
     For ($i = 2 ; $i -lt $Parts.Count ; $i++) {
        $Temp = $Temp + $Parts[$i] + "\"
     }
     $_.InstallLocation = $Temp.Replace('\\','\')
     $_.InstallDate = $_.InstallDate.Replace('/','')
  }

  $ErrorActionPreference = "Continue"

 $PgmsInfo = $PgmsInfo |  
   Sort-Object -Property BasePath, DisplayName |
   Format-Table -Property $fmtPgms -Wrap -GroupBy BasePath |
   Out-String

  $tboxInstPgms = 
      New-Object -TypeName System.Windows.Forms.Textbox

  If (-not($IsQuiet)) {    
    $tabInstPgms  = 
        New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabInstPgms
                tboxObjectName = [ref]$tboxInstPgms
                tabName = "Installed Programs"
                TipText = "Installed Programs"}
  
    GenerateTab @GTArgs
  }

  $tboxInstPgms.Text = $PgmsInfo

  Return ,$tboxInstPgms.Text

} #-------------------   End Programs Tab    ------------------

Function RegistryServicesTab {

  $ServicesTabStartCnt = $CITable.Rows.Count

  If ($LocalMachine) {
    $PropArgs = 
      @{Path= "HKLM:\SYSTEM\CurrentControlSet" +
              "\Control\Session Manager\Power" 
       Name = "HiberbootEnabled"
       ErrorAction = "SilentlyContinue"}
     
      $Prop = Get-ItemProperty @PropArgs 
  }
  Else {
      $ICArgs = @{ComputerName = "$CompName"
                 ScriptBlock  = {$PropArgs = 
      @{Path= "HKLM:\SYSTEM\CurrentControlSet" +
                  "\Control\Session Manager\Power" 
           Name = "HiberbootEnabled"
           ErrorAction = "SilentlyContinue"}
         
          Get-ItemProperty @PropArgs}}

      $Prop = Invoke-Command @ICArgs
  }

  If ($Prop -ne $Null) {
    $ARArgs = 
      @{ARIN = "Registry: HiberbootEnabled" 
        ARIV = ([Bool]$($Prop.HiberbootEnabled.Equals(1)))}
    Add-Row @ARArgs
  }

  If ($LocalMachine) {
    $PropArgs = 
        @{Path="HKLM:SYSTEM\CurrentControlSet\Control\Power" 
          Name = "HibernateEnabled"}
  
    $Prop = Get-ItemProperty @PropArgs
  }
  Else {
      $ICArgs = @{ComputerName = "$CompName"
                 ScriptBlock  = {$PropArgs = 
          @{Path="HKLM:SYSTEM\CurrentControlSet\Control\Power" 
            Name = "HibernateEnabled"
            ErrorAction = "SilentlyContinue"}
         
          Get-ItemProperty @PropArgs}}

      $Prop = Invoke-Command @ICArgs
  }

  $ARArgs = 
    @{ARIN = 'Registry: HibernateEnabled' 
      ARIV = ([Bool]$($Prop.HibernateEnabled.Equals(1)))}
  Add-Row @ARArgs

  $GSArgs = @{ComputerName = "$CompName"
              Name = @("HomeGroup*","USBDLM*","Macrium*",
                       "Remote*","MsKey*","VSS","Wsearch",
                       "DiagTrack")}
  Get-Service @GSArgs |
     ForEach-Object  {
        $ARArgs = @{ARIN = "Service:  $($_.DisplayName)" 
                    ARIV = $($_.Status)}
        Add-Row @ARArgs
     }

  Add-Row -ARIN ''
  Add-Row -ARIN '[  Security Items  ]'

  If ($LocalMachine) {
    $UACArgs = @{Name = 'EnableLUA' 
                 Path = 'HKLM:\SOFTWARE\Microsoft\Windows\' +
                        'CurrentVersion\Policies\System'
                               }
    $UAC = Get-ItemProperty @UACArgs
  }
  Else {
      $UACArgs = @{ComputerName = "$CompName" 
                   ScriptBlock  = {
                   $UACArgs = @{
                   Name = 'EnableLUA' 
                   Path = 'HKLM:\SOFTWARE\Microsoft\Windows\' +
                          'CurrentVersion\Policies\System'
                               }
                   Get-ItemProperty @UACArgs}
                }
      $UAC = invoke-command @UACArgs 
  }

  $ARArgs = @{ARIN = 'Registry: UAC Enabled' 
              ARIV = ($UAC.EnableLUA.Equals(1))}
  Add-Row @ARArgs
#-----------------------------------------------------------

  $MyServices =
   @("MBAM*","EMET*","WinDefend*","MpsSvc","wuauserv","BDESVC","TeamViewer*")
  If ($ServerOS -ge 8.0) { $MyServices += "fhsvc" }

  $GSArgs = @{Computer = "$CompName"
              Name     = $MyServices}
  Get-Service @GSArgs |
     ForEach-Object  {
        $ARArgs = @{ARIN = "Service:  $($_.DisplayName)" 
                    ARIV = $($_.Status)}
        Add-Row @ARArgs
     }

  $fmtServices = 
      @{Expression={$_.Item};Label="Item";Width=60},
      @{Expression={$_.Value};Label="Value";Width=20}

  $ServicesInfo = $CITable.Rows[$ServicesTabStartCnt..$(
                                 $CITable.Rows.Count - 1)] |
             Select-Object -Property Item,Value   |
             Format-Table  -Property $fmtServices | Out-String

  $tboxServices = 
      New-Object -TypeName System.Windows.Forms.Textbox
    

  If (-not($IsQuiet)) {
    $tabServices  = 
        New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabServices
                tboxObjectName = [ref]$tboxServices
                tabName = 'Registry/Services'
                TipText = 'Registry/Services...Hibernate,' +
                  ' HyperBoot, and Homegroup, MalwareBytes'}
  
    GenerateTab @GTArgs
  }

  $tboxServices.Text = $ServicesInfo

  Return ,$tboxServices.Text

} #----------------------- End Services Tab -------------------

Function BatteryTab {

  $Battery = $True

  Try {
       $WMIArgs = @{Class        = 'BatteryStatus' 
                    Namespace    = 'root\wmi'
                    ComputerName = "$CompName"
                    ErrorAction  = 'Stop'}
       $colBatStatus = Get-WMIObject @WMIArgs

  }
  Catch [Exception] 
  {
    $Battery = $False
  }

If ($Battery) {

  $Availability_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Running/Full Power',
  'Warning',
  'In Test',
  'Not Applicable',
  'Power Off',
  'Off Line',
  'Off Duty',
  'Degraded',
  'Not Installed',
  'Install Error',
  'Power Save - Unknown',
  'Power Save - Low Power Mode',
  'Power Save - Standby',
  'Power Cycle',
  'Power Save - Warning',
  'Paused',
  'Not Ready',
  'Not Configured',
  'Quiesced'
  )

  $BatteryStatus_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Fully Charged',
  'Low',
  'Critical',
  'Charging',
  'Charging and High',
  'Charging and Low',
  'Charging and Critical',
  'Undefined',
  'Partially Charged'
  )

  $Chemestry_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Lead Acid',
  'Nickel Cadmium',
  'Nickel Metal Hydride',
  'Lithium-ion',
  'Zinc Air',
  'Lithium Polymer'
  ) 
  
  $StatusInfo_ReturnValue = (
  '',
  'Other',
  'Unknown',
  'Enabled',
  'Disabled',
  'Not Applicable'
  ) 

  $WimArgs = @{ComputerName = "$CompName"
               Class        = 'Win32_Battery'
               Namespace    = 'root\CIMV2'}
  $colItems = Get-WMIObject @WimArgs

  If (-not ($colItems -eq $Null)) {

    $BatteryTabStartCnt = $CITable.Rows.Count

    If ($colBatStatus.PowerOnline) {
      Add-Row -ARIN 'On Mains' -ARIV $colbatstatus.poweronline
      If ($colItems.TimeToFullCharge -ne $Null) {
        $ARArgs = @{ARIN = 'Time To Full Charge' 
                    ARIV = $colItems.TimeToFullCharge}
        Add-Row @ARARGS
      }
      Add-Row -ARIN 'Charging' -ARIV $colBatStatus.Charging
    }

    Else {
      $ARArgs = @{ARIN = 'Availability' 
                  ARIV =
             $Availability_ReturnValue[$colItems.Availability]}
      Add-Row @ARARGS

      If ($colItems.BatteryRechargeTime -ne $null) {
      $ARArgs = @{ARIN = 'Battery Recharge Time' 
                  ARIV = $colItems.BatteryRechargeTime}
      Add-Row @ARARGS
      }

      If ($colItems.ConfigManagerErrorCode -ne $Null) {
        $ARArgs = @{ARIN = 'Configuration Manager Error Code' 
                    ARIV = $colItems.ConfigManagerErrorCode}
        Add-Row @ARArgs
      }

      If ($colItems.ConfigManagerUserConfig -ne $Null) {
        $ARArgs = @{ARIN = 
                 'Configuration Manager User Configuration' 
                    ARIV = $colItems.ConfigManagerUserConfig}
        Add-Row @ARArgs
      }

      $ARArgs = @{ARIN = 'Description' 
                  ARIV = $colItems.Description}
      Add-Row @ARArgs

      If ($colItems.DesignCapacity -ne $Null) {
        $ARArgs = @{ARIN = 'Design Capacity' 
                    ARIV = $colItems.DesignCapacity}
        Add-Row @ARArgs
      }

      $ARArgs = @{ARIN = 'Design Voltage' 
                  ARIV = $($colItems.DesignVoltage/1000 )}
      Add-Row @ARArgs

      $ARArgs = @{ARIN = 'Device ID' 
                  ARIV = $colItems.DeviceID}
      Add-Row @ARArgs

      If ($colItems.ErrorCleared -ne $Null) {
        $ARArgs = @{ARIN = 'Error Cleared' 
                    ARIV = $colItems.ErrorCleared}
        Add-Row @ARArgs
      }

      If ($colItems.ErrorDescription -ne $Null) {
        $ARArgs = @{ARIN = 'Error Description' 
                    ARIV = $colItems.ErrorDescription}
        Add-Row @ARArgs
      }

      $ARArgs = @{ARIN = 'Estimated Charge Remaining' 
                  ARIV = 
                    "$($colItems.EstimatedChargeRemaining)%"}
      Add-Row @ARArgs

      $TS = New-TimeSpan -Minutes $($colItems.EstimatedRunTime)
      Add-Row -ARIN 'Estimated Run Time'  -ARIV $ts 

      If ($colItems.ExpectedBatteryLife -ne $Null) {
        $ARArgs = @{ARIN = 'Expected Battery Life' 
                    ARIV = $colItems.ExpectedBatteryLife}
        Add-Row @ARArgs
        $ARArgs = @{ARIN = 'Expected Life'
                    ARIV = $colItems.ExpectedLife}
        Add-Row @ARArgs
      }

      If ($colItems.FullChargeCapacity -ne $Null) {
        $ARArgs = @{ARIN = 'Full Charge Capacity' 
                    ARIV = $colItems.FullChargeCapacity}
        Add-Row @ARArgs
      } 

      If ($colItems.InstallDate -ne $Null) {
        $ARArgs = @{ARIN = 'Installation Date' 
                    ARIV = $colItems.InstallDate}
        Add-Row @ARArgs
      }

      If ($colItems.LastErrorCode -ne $Null) {
        $ARArgs = @{ARIN = 'Last Error Code' 
                    ARIV = $colItems.LastErrorCode}
        Add-Row @ARArgs
      }

      If ($colItems.MaxRechargeTime -ne $Null) {
        $ARArgs = @{ARIN = 'Maximum Recharge Time' 
                    ARIV = $colItems.MaxRechargeTime}
        Add-Row @ARArgs
      }

      Add-Row -ARIN 'Name' -ARIV $colItems.Name 

      If ($colItems.PNPDeviceID -ne $Null) {
        $ARArgs = @{ARIN = 'PNP Device ID' 
                    ARIV = $colItems.PNPDeviceID}
        Add-Row @ARArgs
      }

      If ($colItems.SmartBatteryVersion -ne $Null) {
        $ARArgs = @{ARIN = 'Smart Battery Version'
                    ARIV = $colItems.SmartBatteryVersion}
        Add-Row @ARArgs
      }

      Add-Row -ARIN 'Status' $colItems.Status 
      If ($colItems.StatusInfo -ne $Null) {
        $ARArgs = @{ARIN = 'Status Information'
                    ARIV = 
                $StatusInfo_ReturnValue[$colItems.StatusInfo]}
        Add-Row @ARArgs
      }

      If ($colItems.TimeOnBattery -ne $Null) {
        $TS = New-TimeSpan -Seconds $colItems.TimeOnBattery
        Add-Row -ARIN 'Time On Battery' -ARIV $TS  
      }
    } #End Else..If ($colBatStatus.PowerOnline)

    $BatteryChg = Test-BatteryHealth
    If ($BatteryChg -eq -1) { $BatteryChg = "Undetermined"}

    $ARArgs = @{ARIN = 'Battery Last Full Charge' 
                ARIV = "$BatteryChg% of Original Capacity"}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Current Battery Charge mWhr' 
                ARIV = $colBatStatus.RemainingCapacity}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Battery Status' 
                ARIV = 
           $BatteryStatus_ReturnValue[$colItems.BatteryStatus]}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Caption'
                ARIV = $colItems.Caption}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Chemistry' 
                ARIV =
                 $Chemestry_ReturnValue[$colItems.Chemistry]}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Power Management Capabilities' 
                ARIV = 
               ([String]$colItems.PowerManagementCapabilities)}
    Add-Row @ARArgs

    $ARArgs = @{ARIN = 'Power Management Supported' 
                ARIV =
                 (&{If($colItems.PowerManagementSupported) 
                    {"True"} Else {"False"}})}
    Add-Row @ARArgs

    $GMLArgs = @{TestObj = 
       $CITable.Item[$BatteryTabStartCnt..$(
                     $CITable.Rows.Count -1)]}
    $ItemLen = Get-MaxLength @GMLArgs
    $ItemLen += 2

    $fmtBattery = 
      @{Expression={$_.Item};Label="Item  ";Width=$ItemLen},
      @{Expression={$_.Value};Label="Value";Width=80-$ItemLen}

    $BatteryInfo = 
      $CITable.Rows[$BatteryTabStartCnt..$(
                    $CITable.Rows.Count -1)] |
      Format-Table -Property $fmtBattery | Out-String

    $tboxBattery = 
       New-Object -TypeName System.Windows.Forms.TextBox
    
  If (-not($IsQuiet)) {
    $tabBattery  = 
       New-Object -TypeName System.Windows.Forms.TabPage
    $GTArgs = @{tabObjectName =  [ref]$tabBattery
              tboxObjectName = [ref]$tboxBattery
              tabName = 'Battery'
              TipText = 'Protable Devices Battery Information'}

    GenerateTab @GTArgs
  }

    $tboxBattery.Text = $BatteryInfo

    Return ,$tboxBattery.text   
 
  } #End If (-not($colItems -eq $Null))   

} #End If ($Battery)

} #----------------------- End Battery Tab  --------------------

Function GenerateTab {

 param (
         [Parameter(Mandatory=$True)]
          [System.Windows.Forms.TabPage] $tabObjectName,
         [Parameter(Mandatory=$True)]
          [System.Windows.Forms.Textbox] $tboxObjectName,
         [Parameter(Mandatory=$True)]
          [string] $tabName,
         [Parameter(Mandatory=$False)]
          [string] $TipText
        )

  $tabObjectName.Name          = "$tabName"
  $tabObjectName.Location      = $tabLocation
  $tabObjectName.Padding       = $System_Windows_Forms_Padding
  $tabObjectName.Size          = $tabSize
  $tabObjectName.TabIndex      = 0
  $tabObjectName.Text          = $TabName
  If ($TipText -ne $null) {
    $tabObjectName.ToolTipText = $TipText
  }

  $tabObjectName.DataBindings.DefaultDataSourceUpdateMode = 0
  $tabObjectName.UseVisualStyleBackColor = $True

  $tboxObjectName.Location   = $tboxLocation
  $tboxObjectName.Size       = $tboxSize
  $tboxObjectName.MultiLine  = $True
  $tboxObjectName.ForeColor  = $tboxForeColor
  $tboxObjectName.BackColor  = $tboxBackColor
  $tboxObjectName.ScrollBars = 'Both'
  $tboxObjectName.Font       = $tboxFont
  $tboxObjectName.ReadOnly   = $True

  $MainTabCtrl.Controls.Add($tabObjectName)
  $tabObjectName.Controls.add($tboxObjectName)

}   #---------------------- End GenerateTab -------------------

Function Add-Row {

  param (
         [Parameter(Mandatory=$False)]
         [alias("ARIN")]
         [string]$ItemName,
         [Parameter(Mandatory=$False)]
         [alias("ARIV")]
         [string]$ItemValue
        )

  $CITRow       = $CITable.NewRow() #Create Row Variable
  $CITRow.Item  = $ItemName       #Assign items to row variable
  $CITRow.Value = $ItemValue
  $CITable.Rows.Add($CITRow)  #Add Table Row using Row Variable

}  #------------------ End Function Add-Row -------------------

Function Combine-Object {

  param(
	 [Parameter(Mandatory=$True)] $object1,
	 [Parameter(Mandatory=$True)] $object2
	)

#	 Courtesy of: http://powershell.com/cs/media/p/7924.aspx

	$propertylistObj1 = 
      @($object1 | Get-Member -ea Stop -memberType *Property |
                   Select-Object -ExpandProperty Name)
	$propertylistObj2 = 
      @($object2 | Get-Member -memberType *Property |
                   Select-Object -ExpandProperty Name |
                   Where-Object { $_ -notlike '__*'})

	$propertylistObj2 | ForEach-Object {
		if ($propertyListObj1 -contains $_) {
			$name = '_{0}' -f $_
		}
        else {
			$name = $_
		}
 
      $AMArgs = @{NotePropertyName = "$name"
                  NotePropertyValue = ($object2.$_)}

	  $object1 = $object1 | 
         Add-Member @AMArgs -PassThru
	}

	Return ,$object1

} #----------------- End Function Combine-Object --------------

Function Get-AdminStatus {

If (-NOT ([Security.Principal.WindowsPrincipal] `
 [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole(`
 [Security.Principal.WindowsBuiltInRole] "Administrator"))
  {"User"}
Else
  {"Administrator"}

} # -----------  End Function Get-AdminStatus -----------------

Function Get-AntiVirus{

<#+--------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddawy |
  +--------------------------------------------------+
#>
  [CmdletBinding()]
  param (
    [parameter(ValueFromPipeline=$true,
      ValueFromPipelineByPropertyName=$true)]
     [string]$CompName="$env:COMPUTERNAME"
  )

  PROCESS{
    $WMIArgs = @{Class = 'Win32_OperatingSystem'
                 ComputerName = "$CompName"}
    $os = Get-WmiObject @WMIArgs

   if ([int]$os.BuildNumber -ge 6001 ) {
     $WMIArgs = @{Namespace = 'ROOT\SecurityCenter2'
                  Class = 'AntiVirusProduct'
                  ComputerName = "$CompName"}
     $av = Get-WmiObject  @WMIArgs
   }
   else {
     $WMIArgs = @{Namespace = 'ROOT\SecurityCenter'
                  Class = 'AntiVirusProduct'
                  ComputerName = "$CompName"}
     $av = Get-WmiObject  @WMIArgs
   }

   Return ,$av

  } #End Process

} #--------------- End Function Get-AntiVirus  ----------------

Function Get-BiosType {

<#
.Synopsis
   Determines underlying firmware (BIOS) type and returns an integer indicating UEFI, Legacy BIOS or Unknown.
   Supported on Windows 8/Server 2012 or later
.DESCRIPTION
   This function uses a complied Win32 API call to determine the underlying system firmware type.
.EXAMPLE
   If (Get-BiosType -eq 1) { # System is running UEFI firmware... }
.EXAMPLE
    Switch (Get-BiosType) {
        1       {"Legacy BIOS"}
        2       {"UEFI"}
        Default {"Unknown"}
    }
.OUTPUTS
   Integer indicating firmware type (1 = Legacy BIOS, 2 = UEFI, Other = Unknown)
.FUNCTIONALITY
   Determines underlying system firmware type
#>

[OutputType([UInt32])]
Param(
   [Parameter(Mandatory=$True)]
     [String] $CompName)

 If ($LocalMachine) {
     
   $ATArgs = @{Language       = 'CSharp'
               ErrorAction    = 'SilentlyContinue' 
               TypeDefinition = @'
      
    using System;
    using System.Runtime.InteropServices;

    public class FirmwareType
    {
     [DllImport("kernel32.dll")]
     static extern bool GetFirmwareType(ref uint FirmwareType);

     public static uint GetFirmwareType()
     {
         uint firmwaretype = 0;
         if (GetFirmwareType(ref firmwaretype))
             return firmwaretype;
         else
             return 0;   // API call failed, just return 'unknown'
     }
    }
'@
    } #End $ATArgs

    $ErrorActionPreference = 'Stop'
    Try {
      If (-not [System.Management.Automation.PSTypeName][FirmwareType]) {
      }
    }
    Catch {  #Type does not Exist Create it!
      Add-Type @ATArgs
    }
    $ErrorActionPreference = 'SilentlyContinue'

    [FirmwareType]::GetFirmwareType()
  }
  Else {
      $ICArgs = @{ComputerName = "$CompName"
                  ScriptBlock  = {
       Add-Type -Language CSharp -TypeDefinition @'
   
       using System;
       using System.Runtime.InteropServices;
   
       public class FirmwareType
       {
           [DllImport("kernel32.dll")]
           static extern bool GetFirmwareType(ref uint FirmwareType);
   
           public static uint GetFirmwareType()
           {
               uint firmwaretype = 0;
               if (GetFirmwareType(ref firmwaretype))
                   return firmwaretype;
               else
                   return 0;   // API call failed, just return 'unknown'
           }
       }
'@
       [FirmwareType]::GetFirmwareType()
   
   }}  #End $ICArgs
   
    Invoke-Command @ICArgs

 } #End Else

}  #End  ---------------Function Get-BIOSTYPE ----------------


Function Get-Drives {

#	 Courtesy of: http://powershell.com/cs/media/p/7924.aspx

  Get-WmiObject -computer $CompName -Class Win32_DiskPartition |
	ForEach-Object {
		$partition = $_
		$logicaldisk = 
           $partition.psbase.GetRelated('Win32_LogicalDisk')
		if ($logicaldisk -ne $null) {
			Combine-Object $logicaldisk $partition
		}
	} |  Select-Object -Property Name, VolumeName, FileSystem, 
                                 DriveType, Size, Compressed, 
                                 FreeSpace,  DiskIndex, 
                                 Index, Bootable

  #Note: This returns the Select-Object output to the caller!
  #      The Select-Object could be dropped but unneeded data 
  #      would be returned to caller...inefficient!

} #--------------    End Function Get-Drives  -----------------

Function Get-NetworkComputers {

$wgrp = (
 [adsi]"WinNT://$((Get-WMIObject Win32_ComputerSystem).Domain)"
        ).Children.path

  $WGComps = $Null
  
  ForEach ($Item in $wgrp) {
  
     If ($Item -ne $Null) {
       $temp = $Item.ToString() -Split('/')
       [Array]$WGComps += "$($temp[$($Temp.Count)-1])"
     } #If ($Item...
  
  }    #End ForEach
  
   Return $WGComps

} #-------------- End Function Get-NetworkComputers -----------

Function Get-MaxLength {
<#
.SYNOPSIS
   Finds the length of the longest item in collection.

.DESCRIPTION
   Use this Function to get the length of the longest item in a
   collection for use in format strings or other places where needed

.PARAMETER TestObj
    The qualified object to be tested. See example!

.Parameter MinLen
    The minimum length of the item (if using for formatting) which
    should be the Label (title) length. Note if the object item
    being tested does not have a Length property you MUST specify
    the label length!



.OUTPUTS
    Returns a numerical value

.NOTES


.EXAMPLE
   $NameLen = Get-MaxLength -TestObj $DotNet.PSChildName
   $VerLen  = Get-MaxLength -TestObj $DotNet.Version
   $RNLen   = Get-MaxLength -TestObj $DotNet.Release -MinLen 11

     #--- .Net Information ---

 $fmtDotNet = 
  @{Expression={$_.PSChildName};Label=".Net Type";Width=$NameLen},
  @{Expression={$_.Version};Label="Version No:";Width=$VerLen},
  @{Expression={$_.Release};Label="Release No:";Width=$RNLen}

  $Dotnet | Format-Table $fmtDotNet
#>

  Param(
    [Parameter(Mandatory=$True)]
     [object] $TestObj,
    [Parameter(Mandatory=$False)]
     [int] $MinLen = 0
  )

   $ErrorActionPreference = "SilentlyContinue"

   foreach ($x in $TestObj) {
     If ($x.Trim().length -gt $MinLen) {
       $MinLen = $x.Trim().length
     }
   }

   $ErrorActionPreference = "Continue"

   Return ,$MinLen

}  #-----------  End Function Get-MaxLength  ------------------

Function Get-UserAccounts{

<#+--------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddawy |
  +--------------------------------------------------+
#>
  [CmdletBinding()]
  param (
   [parameter(ValueFromPipeline=$true,
     ValueFromPipelineByPropertyName=$true)]
    [string]$ComputerName  ="$env:COMPUTERNAME"
  )

  PROCESS{
   $WMIArgs = @{Class = 'Win32_UserAccount'
                ComputerName = $ComputerName  }
   Get-WmiObject @WmiArgs |
    Select-Object -Property AccountType, Description, Disabled,
        Domain, FullName,InstallDate, LocalAccount, Lockout, 
        Name, SID, SIDType, PasswordChangeable,PasswordExpires,
        PasswordRequired
  } #End Process

}  #-------------  End Function Get-UserAccounts  -------------

Function Get-WindowsProductKey {

  param (
          [Parameter(Mandatory=$True)]
            [string]$CompName
        )

 $Reg = [WMIClass] ("\\" + $CompName + 
                    "\root\default:StdRegProv")
 $values = [byte[]]($reg.getbinaryvalue(2147483650,
           "SOFTWARE\Microsoft\Windows NT\CurrentVersion",
           "DigitalProductId").uvalue)
 $lookup = [char[]]("B","C","D","F","G","H","J","K","M",
                    "P","Q","R","T","V","W","X","Y",
                    "2","3","4","6","7","8","9")
 $keyStartIndex      = [int]52;
 $keyEndIndex        = [int]($keyStartIndex + 15);
 $decodeLength       = [int]29
 $decodeStringLength = [int]15
 $decodedChars       = New-Object char[] $decodeLength
 $hexPid             = New-Object System.Collections.ArrayList

 for ($i = $keyStartIndex; $i -le $keyEndIndex; $i++) {
   [void]$hexPid.Add($values[$i])
 }

 for ( $i = $decodeLength - 1; $i -ge 0; $i--) {

   if (($i + 1) % 6 -eq 0) {
     $decodedChars[$i] = '-'
   }
   else
   {
     $digitMapIndex = [int]0
     for ($j = $decodeStringLength - 1; $j -ge 0; $j--)
     {
      $byteValue = [int](($digitMapIndex * [int]256) -bor 
                          [byte]$hexPid[$j])
      $hexPid[$j] = [byte] ([math]::Floor($byteValue / 24))
      $digitMapIndex = $byteValue % 24
      $decodedChars[$i] = $lookup[$digitMapIndex]
     }
   }
 }   #End For ($i...)

 $WindowsKey = ''
 $decodedChars | ForEach-Object { $WindowsKey+=$_}

 Return ,$WindowsKey

}  #----------  End Function Get-WindowsProductKey ------------

Function Get-WUsettings{

  [CmdletBinding(
      SupportsShouldProcess=$True,
      ConfirmImpact="Low"
  )]
  Param(
         [ValidateNotNullOrEmpty()]
         [parameter(ValueFromPipeline=$true,
                    ValueFromPipelineByPropertyName=$true)]
           [String[]]$ComputerName 
  )

  Begin{
    $User = [Security.Principal.WindowsIdentity]::GetCurrent()
    $Role = (New-Object Security.Principal.WindowsPrincipal $user
            ).IsInRole([Security.Principal.WindowsBuiltinRole]::
            Administrator)
    if(!$Role){
#    Write-Warning $("To perform some operations you must run " +
#                  "an elevated Windows PowerShell console.")
    } #End If !$Role  
  }

  Process{
    Write-Debug "STAGE 0: Prepare environment"
    Foreach($Computer in $ComputerName) {
      If (Test-Connection -ComputerName $ComputerName -Quiet){
        Write-Debug "STAGE 1: Get WU settings"
        If ($pscmdlet.ShouldProcess($Computer,"Get WU settings")){
          Write-Verbose "Get WU settings for $ComputerName"
          If($LocalMachine){
              #Supports local instance only
              $objSession = 
                New-Object -ComObject "Microsoft.Update.Session"
          }
          else{
            $objSession =  
               [activator]::CreateInstance([type]::
               GetTypeFromProgID("Microsoft.Update.Session",$ComputerName))
          }
      
          $WUSettings = 
            (New-Object -ComObject "Microsoft.Update.AutoUpdate").settings 
          $WUNotifyLevel = $WUSettings | 
                 select -ExpandProperty NotificationLevel
          $WUDays        = $WUSettings | 
                 select -ExpandProperty ScheduledInstallationDay
          $WUHour        = $WUSettings | 
                 select -ExpandProperty  ScheduledInstallationTime
          $WURecUpdts        = $WUSettings | 
                 select -ExpandProperty  IncludeRecommendedUpdates
          $NotifyLevel   =
            Switch ($WUNotifyLevel){ 
                  1 {"Never Check for updates"} 
                  2 {"Check for updates but let me choose " +
                       "whether to download and install them"} 
                  3 {" Download updates but let me choose " +
                      "whether to install them"} 
                  4 {"Install updates automatically"} 
                  default {
                     "Updates status could not be determined"}
          } #End Switch

          $WUDays = Switch ($WUDays) {
                       0 {"Every Day"}
                       1 {"Sunday"}
                       2 {"Monday"}
                       3 {"Tuesday"}
                       4 {"Wednesday"}
                       5 {"Thursday"}
                       6 {"Friday"}
                       7 {"Saturday"}
                    } #End Switch  
                                      
          "$NotifyLevel" + ":" + "$WUDays" + ":" + "$WUHour" +
                       ":" + "$WURecUpdts"
             
        } #End If($pscmdlet...
      }   #End If(Test-Connection...
      else{
        Write-Warning $("Cannot connect to $Computer. " +
         "Find out if it exists. If so, ensure it is online " +
         "or check whether it is affected by firewall settings.")
        Write-Output "$Computer, could not connect" |
         Out-File "C:\Scripts\WUSettingsConnectionErrors.txt" -Append
      }

    }   #End For-Each($Computer)
  }     #End Process
}  #----------------- End Function Get-WUsettings ------------

Function Show-BalloonTip  
{
  param
  (
    [Parameter(Mandatory=$true)]
       [String] $Text,   
    [Parameter(Mandatory=$True)]
       [String] $Title,
    [Parameter(Mandatory=$True)]          
    [ValidateSet('None', 'Info', 'Warning', 'Error')]
       [String] $Icon = 'Info',
    [Parameter(Mandatory=$False)]
       [Int32] $Timeout = [Int32]2000
  )
 
  Add-Type -AssemblyName System.Windows.Forms

  $GVArgs = @{Name        = "Balloon"
              ErrorAction = "SilentlyContinue"}
  If (-not (Get-Variable @GVArgs))
  {
    $script:balloon = New-Object System.Windows.Forms.NotifyIcon
  }

  $path                    = Get-Process -id $pid | 
                             Select-Object -ExpandProperty Path
  $balloon.Icon            = 
       [System.Drawing.Icon]::ExtractAssociatedIcon($path)
  $balloon.BalloonTipIcon  = $Icon
  $balloon.BalloonTipText  = $Text
  $balloon.BalloonTipTitle = $Title
  $balloon.Visible         = $true

  $balloon.ShowBalloonTip($Timeout)

} #----------- End Function Show-BalloonTip -------------------


Function Test-BatteryHealth { 

  $WMIArgs = @{Class        = "BatteryFullChargedCapacity" 
               Namespace    = 'ROOT\WMI'
               ComputerName = "$CompName"}
  $fullchargecapacity = 
      (Get-WmiObject @WmiArgs).FullChargedCapacity 

  $WMIArgs = @{Class        = "BatteryStaticData" 
               Namespace    = 'ROOT\WMI'
               ComputerName = "$CompName"}

  $designcapacity = (Get-WmiObject @WmiArgs).DesignedCapacity 
 
  if ($fullchargecapacity -eq $designcapacity) { 
      $Batteryhealth=-1
  } 
  Else {
        $batteryhealth = 
            ($fullchargecapacity / $designcapacity) * 100 
        if ($batteryhealth -gt 100) {$batteryhealth = 100} 
  }

  return [decimal]::round($batteryhealth)  

} #----------------  End Test-BatteryHealth -------------------

Function Test-SSD {

<#
.SYNOPSIS
   Detects if the passed Physical Disk Id is a Solid State Disk (SSD) or a
   spindle disk. Returns true for an SSD and false for anything else.

.DESCRIPTION
   Use Get-PhysicalDisk to get the Physical Disk Ids for the system you wish
   to test. This script supports values being passed throught the pipeline.
   
   The methods used for detecting are by reading the Nominal Media Rotation
   Rate and Seek Penalty. These values are measured through method calls
   into the Kernel32.dll. If either of the Win32 DLL calls return true then
   the script will return false. If an exception occurs in either of the
   Win32 DLL calls, the return value will be dependant on the remaining call.

.PARAMETER PhysicalDiskId
    The LUN based physical disk id.

.NOTES
    Author: Grant Carthew
  
.LINK
  https://gist.github.com/grantcarthew/c74bbfd3eba167cd3a7a

.EXAMPLE
   Get-PhysicalDisk | Select-Object -ExpandProperty DeviceId | Test-SSD

   This example will test all the disks on the local computer.

.EXAMPLE
   if (Test-SSD -PhysicalDiskId 0) { $delay = 0 }

   This example will change the delay variable on a script if the
   first physical disk is an SSD.
#>

[CmdletBinding(SupportsShouldProcess=$true,
               ConfirmImpact="Low")]
[OutputType([boolean])]
Param
(
    [Parameter(Mandatory=$true,
               ValueFromPipeline=$true,
               ValueFromPipelineByPropertyName=$true,
               Position=0)]
    [Int]
    $PhysicalDiskId
)

Begin {
    $code = @"
using Microsoft.Win32.SafeHandles;
using System;
using System.Runtime.InteropServices;
using System.Text;
 
namespace Util
{
    public class DetectSSD
    {
        // For CreateFile to get handle to drive
        private const uint GENERIC_READ = 0x80000000;
        private const uint GENERIC_WRITE = 0x40000000;
        private const uint FILE_SHARE_READ = 0x00000001;
        private const uint FILE_SHARE_WRITE = 0x00000002;
        private const uint OPEN_EXISTING = 3;
        private const uint FILE_ATTRIBUTE_NORMAL = 0x00000080;
 
        // CreateFile to get handle to drive
        [DllImport("kernel32.dll", SetLastError = true)]
        private static extern SafeFileHandle CreateFileW(
            [MarshalAs(UnmanagedType.LPWStr)]
            string lpFileName,
            uint dwDesiredAccess,
            uint dwShareMode,
            IntPtr lpSecurityAttributes,
            uint dwCreationDisposition,
            uint dwFlagsAndAttributes,
            IntPtr hTemplateFile);
 
        // For control codes
        private const uint FILE_DEVICE_MASS_STORAGE = 0x0000002d;
        private const uint IOCTL_STORAGE_BASE = FILE_DEVICE_MASS_STORAGE;
        private const uint FILE_DEVICE_CONTROLLER = 0x00000004;
        private const uint IOCTL_SCSI_BASE = FILE_DEVICE_CONTROLLER;
        private const uint METHOD_BUFFERED = 0;
        private const uint FILE_ANY_ACCESS = 0;
        private const uint FILE_READ_ACCESS = 0x00000001;
        private const uint FILE_WRITE_ACCESS = 0x00000002;
 
        private static uint CTL_CODE(uint DeviceType, uint Function,
                                     uint Method, uint Access)
        {
            return ((DeviceType << 16) | (Access << 14) |
                    (Function << 2) | Method);
        }
 
        // For DeviceIoControl to check no seek penalty
        private const uint StorageDeviceSeekPenaltyProperty = 7;
        private const uint PropertyStandardQuery = 0;
 
        [StructLayout(LayoutKind.Sequential)]
        private struct STORAGE_PROPERTY_QUERY
        {
            public uint PropertyId;
            public uint QueryType;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 1)]
            public byte[] AdditionalParameters;
        }
 
        [StructLayout(LayoutKind.Sequential)]
        private struct DEVICE_SEEK_PENALTY_DESCRIPTOR
        {
            public uint Version;
            public uint Size;
            [MarshalAs(UnmanagedType.U1)]
            public bool IncursSeekPenalty;
        }
 
        // DeviceIoControl to check no seek penalty
        [DllImport("kernel32.dll", EntryPoint = "DeviceIoControl",
                   SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool DeviceIoControl(
            SafeFileHandle hDevice,
            uint dwIoControlCode,
            ref STORAGE_PROPERTY_QUERY lpInBuffer,
            uint nInBufferSize,
            ref DEVICE_SEEK_PENALTY_DESCRIPTOR lpOutBuffer,
            uint nOutBufferSize,
            out uint lpBytesReturned,
            IntPtr lpOverlapped);
 
        // For DeviceIoControl to check nominal media rotation rate
        private const uint ATA_FLAGS_DATA_IN = 0x02;
 
        [StructLayout(LayoutKind.Sequential)]
        private struct ATA_PASS_THROUGH_EX
        {
            public ushort Length;
            public ushort AtaFlags;
            public byte PathId;
            public byte TargetId;
            public byte Lun;
            public byte ReservedAsUchar;
            public uint DataTransferLength;
            public uint TimeOutValue;
            public uint ReservedAsUlong;
            public IntPtr DataBufferOffset;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public byte[] PreviousTaskFile;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 8)]
            public byte[] CurrentTaskFile;
        }
 
        [StructLayout(LayoutKind.Sequential)]
        private struct ATAIdentifyDeviceQuery
        {
            public ATA_PASS_THROUGH_EX header;
            [MarshalAs(UnmanagedType.ByValArray, SizeConst = 256)]
            public ushort[] data;
        }
 
        // DeviceIoControl to check nominal media rotation rate
        [DllImport("kernel32.dll", EntryPoint = "DeviceIoControl",
                   SetLastError = true)]
        [return: MarshalAs(UnmanagedType.Bool)]
        private static extern bool DeviceIoControl(
            SafeFileHandle hDevice,
            uint dwIoControlCode,
            ref ATAIdentifyDeviceQuery lpInBuffer,
            uint nInBufferSize,
            ref ATAIdentifyDeviceQuery lpOutBuffer,
            uint nOutBufferSize,
            out uint lpBytesReturned,
            IntPtr lpOverlapped);
 
        // For error message
        private const uint FORMAT_MESSAGE_FROM_SYSTEM = 0x00001000;
 
        [DllImport("kernel32.dll", SetLastError = true)]
        static extern uint FormatMessage(
            uint dwFlags,
            IntPtr lpSource,
            uint dwMessageId,
            uint dwLanguageId,
            StringBuilder lpBuffer,
            uint nSize,
            IntPtr Arguments);
 
        // Method for no seek penalty
        public static bool HasSeekPenalty(string sDrive)
        {
            SafeFileHandle hDrive = CreateFileW(
                sDrive,
                0, // No access to drive
                FILE_SHARE_READ | FILE_SHARE_WRITE,
                IntPtr.Zero,
                OPEN_EXISTING,
                FILE_ATTRIBUTE_NORMAL,
                IntPtr.Zero);

            if (hDrive == null || hDrive.IsInvalid)
            {
                string message = GetErrorMessage(Marshal.GetLastWin32Error());
                throw new System.Exception(message);
            }

            uint IOCTL_STORAGE_QUERY_PROPERTY = CTL_CODE(
                IOCTL_STORAGE_BASE, 0x500,
                METHOD_BUFFERED, FILE_ANY_ACCESS); // From winioctl.h
 
            STORAGE_PROPERTY_QUERY query_seek_penalty =
                new STORAGE_PROPERTY_QUERY();
            query_seek_penalty.PropertyId = StorageDeviceSeekPenaltyProperty;
            query_seek_penalty.QueryType = PropertyStandardQuery;
 
            DEVICE_SEEK_PENALTY_DESCRIPTOR query_seek_penalty_desc =
                new DEVICE_SEEK_PENALTY_DESCRIPTOR();
 
            uint returned_query_seek_penalty_size;
 
            bool query_seek_penalty_result = DeviceIoControl(
                hDrive,
                IOCTL_STORAGE_QUERY_PROPERTY,
                ref query_seek_penalty,
                (uint)Marshal.SizeOf(query_seek_penalty),
                ref query_seek_penalty_desc,
                (uint)Marshal.SizeOf(query_seek_penalty_desc),
                out returned_query_seek_penalty_size,
                IntPtr.Zero);
 
            hDrive.Close();
 
            if (query_seek_penalty_result == false)
            {
                string message = GetErrorMessage(Marshal.GetLastWin32Error());
                throw new System.Exception(message);
            }
            else
            {
                return query_seek_penalty_desc.IncursSeekPenalty;
            }
        }
 
        // Method for nominal media rotation rate
        // (Administrative privilege is required)
        public static bool HasNominalMediaRotationRate(string sDrive)
        {
            SafeFileHandle hDrive = CreateFileW(
                sDrive,
                GENERIC_READ | GENERIC_WRITE, // Administrative privilege is required
                FILE_SHARE_READ | FILE_SHARE_WRITE,
                IntPtr.Zero,
                OPEN_EXISTING,
                FILE_ATTRIBUTE_NORMAL,
                IntPtr.Zero);
 
            if (hDrive == null || hDrive.IsInvalid)
            {
                string message = GetErrorMessage(Marshal.GetLastWin32Error());
                throw new System.Exception(message);
            }
 
            uint IOCTL_ATA_PASS_THROUGH = CTL_CODE(
                IOCTL_SCSI_BASE, 0x040b, METHOD_BUFFERED,
                FILE_READ_ACCESS | FILE_WRITE_ACCESS); // From ntddscsi.h
 
            ATAIdentifyDeviceQuery id_query = new ATAIdentifyDeviceQuery();
            id_query.data = new ushort[256];
 
            id_query.header.Length = (ushort)Marshal.SizeOf(id_query.header);
            id_query.header.AtaFlags = (ushort)ATA_FLAGS_DATA_IN;
            id_query.header.DataTransferLength =
                (uint)(id_query.data.Length * 2); // Size of "data" in bytes
            id_query.header.TimeOutValue = 3; // Sec
            id_query.header.DataBufferOffset = (IntPtr)Marshal.OffsetOf(
                typeof(ATAIdentifyDeviceQuery), "data");
            id_query.header.PreviousTaskFile = new byte[8];
            id_query.header.CurrentTaskFile = new byte[8];
            id_query.header.CurrentTaskFile[6] = 0xec; // ATA IDENTIFY DEVICE
 
            uint retval_size;
 
            bool result = DeviceIoControl(
                hDrive,
                IOCTL_ATA_PASS_THROUGH,
                ref id_query,
                (uint)Marshal.SizeOf(id_query),
                ref id_query,
                (uint)Marshal.SizeOf(id_query),
                out retval_size,
                IntPtr.Zero);
 
            hDrive.Close();
 
            if (result == false)
            {
                string message = GetErrorMessage(Marshal.GetLastWin32Error());
                throw new System.Exception(message);
            }
            else
            {
                // Word index of nominal media rotation rate
                // (1 means non-rotate device)
                const int kNominalMediaRotRateWordIndex = 217;
 
                if (id_query.data[kNominalMediaRotRateWordIndex] == 1)
                {
                    return false;
                }
                else
                {
                    return true;
                }
            }
        }
 
        // Method for error message
        private static string GetErrorMessage(int code)
        {
            StringBuilder message = new StringBuilder(255);
 
            FormatMessage(
              FORMAT_MESSAGE_FROM_SYSTEM,
              IntPtr.Zero,
              (uint)code,
              0,
              message,
              (uint)message.Capacity,
              IntPtr.Zero);
 
            return message.ToString();
        }
    }
}
"@
    Add-Type -TypeDefinition $code

    $hasRotationRate = $true
    $hasSeekPenalty = $true
}

Process {

    $driveString = "\\.\PhysicalDrive" + $PhysicalDiskId
    Write-Verbose -Message "Current disk item id is: $PhysicalDiskId"
    Write-Verbose -Message "Current disk string is: $driveString"
    Write-Verbose -Message "Calling Win32 DLL Method 'DeviceIoControl' in 'HasNominalMediaRotationRate'."
    if ($PSCmdlet.ShouldProcess("Physical Disk $PhysicalDiskId","Read Nominal Media Rotation Rate Property")) {
        try {
            $hasRotationRate = [Util.DetectSSD]::HasNominalMediaRotationRate([string]$driveString)
        } catch {
            Write-Verbose -Message "HasNominalMediaRotationRate detection failed with the following error;"
            Write-Verbose -Message $Error[0].Exception.Message
            $hasRotationRate = $true
        }
    }

    Write-Verbose -Message "Calling Win32 DLL Method 'DeviceIoControl' in 'HasSeekPenalty'."
    if ($PSCmdlet.ShouldProcess("Physical Disk $PhysicalDiskId","Read Seek Penalty Property")) {
        try {
            $hasSeekPenalty = [Util.DetectSSD]::HasSeekPenalty([string]$driveString)
        } catch {
            Write-Verbose -Message "HasSeekPenalty detection failed with the following error;"
            Write-Verbose -Message $Error[0].Exception.Message
            $hasSeekPenalty = $true
        }
    }

    # Only return true if the disk has no rotation rate or no seek penalty.
    Write-Output -InputObject (!$hasRotationRate -or !$hasSeekPenalty)

}

End {
}

} #------------------------ End Function Test-SSD -------------

Function ValidateWinVer {

  Param (
         [Parameter (Mandatory=$True)]
           [Int] $MajorVer,
         [Parameter (Mandatory=$False)]
           [Int] $MinorVer = 0
  )

  $ClrVers = @{"10.0" = "10";
                "6.3" = "8.1";
                "6.2" = "8.0";
                "6.1" = "7";
                "6.0" = "Vista"}

  $CurOS   = Get-CimInstance -Class Win32_OperatingSystem
  $VersionItems = $CurOS.Version.split('.')

  #Set Global ThisPCWinVer Variable for later reference
  $SVArgs = @{Name = "ThisPCWinVer"
              Scope = "Global"
              Value = 
   [Single]$ClrVers["$($VersionItems[0]).$($VersionItems[1])"]}
  Set-Variable @SVArgs

  If (([int]$VersionItems[0] -lt $MajorVer) -or 
     (([int]$VersionItems[0] -eq $MajorVer) -and 
      ([int]$VersionItems[1] -lt $MinorVer))) {
    $Message = "You are running Windows " + 
      $ClrVers["$($VersionItems[0]).$($VersionItems[1])"] +
      "`nthe program requires at least Windows " +
      $CLrVers["$MajorVer.$MinorVer"] +" to run."

     [Windows.Forms.MessageBox]::Show($Message,"Program Terminated:", 
     [Windows.Forms.MessageBoxButtons]::OK , 
     [Windows.Forms.MessageBoxIcon]::Information)

    Return ,$False

  } #End If
  Else {
    Return "OK",$True   #Mimic MsgBox OK button!
  }

} #------------------- End ValidateWinVer ---------------------

Function Write-ComputerInfoFile {

   "Computer Mentors System Information - Version" +
   " $($PGMVersion)`r`n`r`n" +
   "---     Hardware      ---    `r`n" + $HWInfo         +
   "---       Video       ---    `r`n" + $VideoInfo      +
   "---      Storage      ---`r`n`r`n" + $StorageInfo    +
   "---      Network      ---    `r`n" + $NetworkInfo    +
   "---     Printers      ---    `r`n" + $PrinterInfo    +
   "---      Windows      ---    `r`n" + $WindowsInfo    +
   "---    Environment    ---    `r`n" + $EnvironInfo    +
   "---      Drivers      ---`r`n`r`n" + $DriversInfo    +
   "--- Installed Program ---    `r`n" + $InstPgmsInfo   +
   "--- Registry/Services ---    `r`n" + 
        $ServicesInfo   >  "$InfoFileName"

  If (-not ($BatteryInfo -eq $Null)) {
   "--- Battery Status    ---    `r`n" + 
                         $BatteryInfo  >> "$InfoFileName"
  }

} #End ---------------  Write-ComputerInfoFile  ---------------

#------------------------ Add a Helper ------------------------

$SWAArgs = @{name = “Win32ShowWindowAsync” 
             namespace = 'Win32Functions'}

$showWindowAsync = Add-Type –memberDefinition @”
[DllImport("user32.dll")]
public static extern bool ShowWindowAsync(
            IntPtr hWnd, int nCmdShow);
“@ @SWAargs -PassThru

Function Show-PowerShell() {
     [void]$showWindowAsync::ShowWindowAsync(
                   (Get-Process –id $pid).MainWindowHandle, 10)
}  #End Function Show-PowerShell

Function Hide-PowerShell() {
     [void]$showWindowAsync::ShowWindowAsync(
                   (Get-Process –id $pid).MainWindowHandle, 2)
}  #End Function Hide-PowerShell

<#
  +------------------------------------------------------------+
  |                        Main Program                        |
  +------------------------------------------------------------+
#>

$PGMVersion = 18.10  #--- Remember to Update! ----

$WinVersionTable = @{
   "6.1.7601"   = "7.0"
   "6.2.9200"   = "8.0"
   "6.3.9600"   = "8.1"
   "10"         = "10"
}  

Add-Type -AssemblyName  System.Windows.Forms
Add-Type -AssemblyName  System.Drawing

Hide-Powershell       #--- Hide the CMD Window ---
Clear-Host
$IsQuiet = $Quiet.IsPresent

  $BTArgs = @{Text    = "Checking Local Machine Setup"
              Title   = "I'm Working on $Env:COMPUTERNAME"
              Icon    = 'Info'
              Timeout = [Int32]500}
  Show-BalloonTip  @BTArgs

$TermMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
     "Program Terminated:", 
  [Windows.Forms.MessageBoxButtons]::OK , 
  [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

#---   Check Local (Client) Windows Version   ---
If (-not($(ValidateWinVer -MajorVer 6 -MinorVer 1)[1])) {
  $Message = 
      "You are running Windows Prior to version 7:`n`n" + 
      "This program REQUIRES Windows 7 or later to run!"
  & $TermMsg
  Exit 
}

If ( (Get-AdminStatus) -eq "User") {
  $AdminPriv = $False
}
Else {
  $AdminPriv = $True
}

Try {
  $WSManStatus = $True
  $TWSMArgs = @{ComputerName = "$Env:COMPUTERNAME"
                ErrorAction = "Stop"}
  Test-WSMan @TWSMArgs | Out-Null
}
Catch {
 $WSManStatus = $False
}

If (-not $WSManStatus) {

  If ($AdminPriv) {
    Start-Service -Name "WinRM"
  } #End If ($AdminP...
  Else {
      $Message = 
        "You are running with Standard User Privleges:`n`n" + 
        "The WimRM Service is not started, please run the Setup" +
        " Program.`n" +
        " or use RunAs Administrator to run the program!"
     & $TermMsg
     Exit
  } #End Else

}  #End If ($WSMan...

If ($PSVersionTable.PSVersion.Major -lt 4) {
  
  $Message = "You are running PowerShell version: " +
             "$($PSVersionTable.PSVersion)`n`n" + 
             "The program requires Version 4.0 to run.`n`n" +
             "Please upgrade our PowerShell."
  & $TermMsg
  Exit
} #End If ($PSVersionTable...

$StatusFile = 
 "$([environment]::GetFolderPath("mydocuments"))\" +
 "CMsPCInfo-Errors.log"

If ($ClearLog.IsPresent) {
  $RIArgs = @{Path = "$StatusFile"
              ErrorAction = "SilentlyContinue"}
  Remove-Item @RIArgs
}

Remove-Variable "TermMsg"

$InfoMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
      "$CompName Processing Terminated:", 
  [Windows.Forms.MessageBoxButtons]::OK , 
  [Windows.Forms.MessageBoxIcon]::Information) | Out-Null
}

ForEach ($CompName in $Computers) {  #Loop through List!

  $BTArgs = @{Text    = "Checking remote access..."
              Title   = "I'm Working on $CompName"
              Icon    = 'Info'
              Timeout = [Int32]500}
  Show-BalloonTip  @BTArgs

  $LocalMachine = $("$CompName" -eq "$Env:COMPUTERNAME")
  
  $THosts = Get-Item WSMan:\localhost\Client\TrustedHosts
  $THosts = $THosts.Value.ToString() 
  
  If (-not($THosts.contains("$CompName"))) {
    If ($AdminPriv) {
    
      $SIArgs = @{Path  = "WSMan:\localhost\Client\TrustedHosts"
                  Value = (&{If ($THosts -eq ''){"$CompName"
                            }Else{"$THosts,$CompName"}})
                  Force = $True
                  ErrorAction = 'SilentlyContinue'}
      Set-Item  @SIargs
    }
    Else {
      $Message = 
        "You are running with Standard User Privleges:`n`n" + 
        "The Computer: $CompName" +
        "is NOT in the Trusted Hosts list and can not be added.`n" +
        "Use RunAs Administrator to run the program!"
      If ($IsQuiet) {
        "$(Get-Date -f "MM/dd/yy HH:mm:ss") $Message`n`n" >> "$StatusFile"
      }
      Else { & $InfoMsg }

      Continue     #Move to next computer in list!
    } #Else
  
  }  #If (-not(THosts...
  
  If (-Not((Get-NetworkComputers) -contains "$CompName")) {
     
    $Message = "A Computer named: $CompName " +
               "can not be found on this network.`n`n" + 
               "Please make sure the machine is powered up, " +
               " the name is spelled correctly and try again."

    If ($IsQuiet) {
      "$(Get-Date -f "MM/dd/yy HH:mm:ss") $Message`n`n" >> "$StatusFile"
    }
    Else { & $InfoMsg } 

    Continue     #Move to next computer in list!     
  }
  
  
  Try {
     $GCIArgs = @{Computer    = "$CompName"
                  ClassName   = 'Win32_OperatingSystem'
                  ErrorAction = 'Stop'}
    $ServerOS = Get-CimInstance @GCIArgs
  }
  Catch  {
    $Message = 
      "Sorry the Target: $CompName (Server)`n" +
      "can NOT be contacted via RPC Server`n`n" + 
      "The program requires the Target: $CompName (Server)`n" +
      "be running Windows 7+ and PRO or better " +
      "SKU as it's operating system.`n`n"+
      "You must also have both the Target: $CompName (Server)`n" +
      "and this machine: $Env:COMPUTERNAME (Client)`n" +
      "properly configured for PowerShell Remoting!"
    If ($IsQuiet) {
      "$(Get-Date -f "MM/dd/yy HH:mm:ss") $Message`n`n" >> "$StatusFile"
    }
    Else { & $InfoMsg }

    Continue     #Move to next computer in list!
  }
  
  $OSVerParts = $ServerOS.version.Split('.')
  If ([int]$OSVerParts[0] -gt 6) {
    $TestOS = $OSVerParts[0]
  }
  Else {
    $TestOS = $ServerOS.Version
  }
  $ServerOS = [Single]$($WinVersionTable.get_item($TestOS))
  
  # Created Computer Information Table (CIT)
  $CITable = 
  New-Object -TypeName System.Data.DataTable "Computer Information"
  #Create Columns for table
  $CITItem  = 
      New-Object -TypeName System.Data.DataColumn Item,([string])
  $CITValue = 
      New-Object -TypeName System.Data.DataColumn Value,([string])
  #Add Columns to table
  $CITable.columns.add($CITItem)
  $CITable.columns.add($CITValue)
 
 #--------- Create FileSpec for current computer's info ------- 
  $InfoFileName = 
    "$([environment]::getfolderpath("mydocuments"))\" +
    "CMsPCInfo for $CompName.txt"

  GenerateForm
  
  Remove-Variable "CIT*","ServerOS"
  
} #ForEach ($CompName in $Computers)

#  ------------------ End Main Program ------------------------