﻿Function Set-WindowSizePosition {

<#
  .Synopsis
    Configure a program window to a specified size and location,
    and opening the program if necessary.

  .Description
    Completely configure a program window to a specified size
    (height/width) and positioning that window on a single and/or
    multiple monitor setup at a given x,y coordinate.

  .Parameter AppName
    The name of the application. This must be the exact spelling
    as is returned by Get-Process.

  .Parameter AppWindWidth
    The desired width of the application window. To place on a
    second screen add the width of the 1st screen.
    Note: Use the Working Resolution of your displays! 

  .Parameter AppWindHeight
    The desired height of the application window.

  .Parameter AppWindLeft
    The position of the left border of the application window.

  .Parameter AppWindTop
    The position of the top border the application window.
  
  .Parameter DelaySeconds
    The number of seconds to wait before continuing with the
    program when an application has to be opened. Different
    applications take variable times to start and the program
    can not proceed until the application has been initialized.

  .Outputs
    Application opened and window positioned and sized on screen

  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 21 Jul 2016
     Last Updated : 
     Current Vers : 2.0

  .Example
   $SWSPArgs = @{
   AppName       = "Excel"
   AppWindWidth  = [Int]($EffScrnWidth/2)
   AppWindHeight = $EffScrnHeight
   AppWindLeft   = $EffScrnWidth 
   AppWindTop    = 0
   DelaySeconds  = 1}
   Set-WindowSizePosition @SWSPArgs 

   Setup an Excel window on screen 2 of a 2 screen setup at a width of
   1/2 of the screen.


  .Example
   $SWSPArgs = @{
   AppName       = "Excel"
   AppWindWidth  = $EffScrnWidth
   AppWindHeight = $EffScrnHeight
   AppWindLeft   = 0 
   AppWindTop    = 0
   DelaySeconds  = 1}
   Set-WindowSizePosition @SWSPArgs -Verbose 4>&1 >> "$DebugFile"
       
   Setup Excel with size of $EffScrnWidth and Height of $EffScrnHeight
   located on screen one in upper left corner and if Excel is not 
   running wait 1 second before attemption to process. Output actual
   coordinates to the WorkEnv.Log file in the users Documents folder.

#>


  Param (
    [Parameter(Mandatory=$True)]
      [String] $AppName,
    [Parameter(Mandatory=$True)]
      [Int] $AppWindWidth,
    [Parameter(Mandatory=$True)]
      [Int] $AppWindHeight,
    [Parameter(Mandatory=$True)]
      [Int] $AppWindLeft,
    [Parameter(Mandatory=$True)]
      [Int] $AppWindTop,
    [Parameter(Mandatory=$False)]
      [Int] $DelaySeconds
  )

# Win7 Powershell script to resize Minecraft to 1280x720 (HD for fraps youtube capture)
# use by typing the following at a command prompt:
# > PowerShell -ExecutionPolicy Bypass -File minecraft-sethd.ps1
# refs:
# http://stackoverflow.com/questions/2556872/how-to-set-foreground-window-from-powershell-event-subscriber-action
# http://richardspowershellblog.wordpress.com/2011/07/23/moving-windows/
# http://www.suite101.com/content/client-area-size-with-movewindow-a17846

Add-Type @"
using System;
using System.Runtime.InteropServices;
public class Win32 {
[DllImport("user32.dll")]
[return: MarshalAs(UnmanagedType.Bool)]
public static extern bool GetWindowRect(IntPtr hWnd, out RECT lpRect);
[DllImport("user32.dll")]
[return: MarshalAs(UnmanagedType.Bool)]
public static extern bool GetClientRect(IntPtr hWnd, out RECT lpRect);
[DllImport("user32.dll")]
[return: MarshalAs(UnmanagedType.Bool)]
public static extern bool MoveWindow(IntPtr hWnd, int X, int Y, int nWidth, int nHeight, bool bRepaint);
}
public struct RECT
{
public int Left;   // x position of upper-left corner
public int Top;    // y position of upper-left corner
public int Right;  // x position of lower-right corner
public int Bottom; // y position of lower-right corner
}
"@

  $AppWindow  = New-Object RECT
  $ScrnWindow = New-Object RECT
  
  $h = (Get-Process | where {$_.ProcessName -eq "$AppName"})
  
  If ($h -eq $Null) {
    & ($AppList.$AppName) 
<#
       +-----------------------------------------------+
       | Adjust the seconds conunt below based on the  |
       | loading speed of your applications on your    |
       | computer. Set to time to load the slowest     |
       | loading application in the $AppList hashtable.|
       +-----------------------------------------------+
#>
    If (Test-Path -Path variable:DelaySeconds) {
      Start-Sleep -Seconds $DelaySeconds 
    }

    $h = (Get-Process | where {$_.ProcessName -eq "$AppName"})
  }
  Else {
    Set-WindowStyle -InputObject $h -Style 'MAXIMIZE'
  }

  If ($h -ne $null) {
    $h = $h.MainWindowHandle
    [void] [Win32]::GetWindowRect($h,[ref]$AppWindow)
    Write-Verbose -Message "$AppName"
    Write-Verbose -Message $(
      "App : Left $($AppWindow.Left)  Width $($AppWindow.Right) " +
      "Top $($AppWindow.Top)  Height $($AppWindow.Bottom)")
    Write-Verbose -Message $(
      "New : Left $($AppWindLeft)  Width $($AppWindWidth) " +
      "Top $($AppWindTop)  Height $($AppWindHeight)")
    #Position Top left of screen 2! 1920 x 1080 screen
    #To address a 2nd screen start at 1920!
    [void] [Win32]::MoveWindow(
         $h, $AppWindLeft, $AppWindTop, 
             $AppWindWidth, $AppWindHeight, $true )
  } #End If ($h -ne...

} #End Function Set-WindowSizePosition

Function Set-WindowStyle {

<#
.LINK
  https://gist.github.com/jakeballard/11240204
#>


[CmdletBinding(DefaultParameterSetName = 'InputObject')]
param(
	[Parameter(Position = 0, Mandatory = $True, ValueFromPipeline = $True, ValueFromPipelinebyPropertyName = $True)]
	[Object[]] $InputObject,
	[Parameter(Position = 1)]
	[ValidateSet('FORCEMINIMIZE', 'HIDE', 'MAXIMIZE', 'MINIMIZE', 'RESTORE', 'SHOW', 'SHOWDEFAULT', 'SHOWMAXIMIZED', 'SHOWMINIMIZED', 'SHOWMINNOACTIVE', 'SHOWNA', 'SHOWNOACTIVATE', 'SHOWNORMAL')]
	[string] $Style = 'SHOW'
)

BEGIN {
	$WindowStates = @{
		'FORCEMINIMIZE'   = 11
		'HIDE'            = 0
		'MAXIMIZE'        = 3
		'MINIMIZE'        = 6
		'RESTORE'         = 9
		'SHOW'            = 5
		'SHOWDEFAULT'     = 10
		'SHOWMAXIMIZED'   = 3
		'SHOWMINIMIZED'   = 2
		'SHOWMINNOACTIVE' = 7
		'SHOWNA'          = 8
		'SHOWNOACTIVATE'  = 4
		'SHOWNORMAL'      = 1
    }

$Win32ShowWindowAsync = Add-Type -MemberDefinition @'
[DllImport("user32.dll")] 
public static extern bool ShowWindowAsync(IntPtr hWnd, int nCmdShow); 
'@ -Name "Win32ShowWindowAsync" -Namespace Win32Functions -PassThru
	
}

  PROCESS {
		foreach ($process in $InputObject) {
		  $Win32ShowWindowAsync::ShowWindowAsync(
            $process.MainWindowHandle, $WindowStates[$Style]) | 
            Out-Null
		}
  }

}
