﻿<#
  .Synopsis
   Locate files in a specified directory that exceed the
   specified length.

  .Description
   Locate files in a specified directory that exceed the
   specified length. Output to the screen or a .csv file
   readable by excel.    

  .Parameter BasePath
   The path where the search should begin. All child
   directories will also be searched.

  .Parameter MinLen
   The minimum path length to be included in the results.

  .Parameter WriteCSV
   Should the output be written to a .csv file in the
   users documents folder

  .Outputs
   List to the console or .csv file.
   Status Messages indicating invalid path, no matching
   files, location of the output file if specified. 

  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 06 Mar 2017
     Last Updated : 
     Current Vers : 1.0

  .Example
   Get-LongFileNames.ps1 -BasePath "C:\Windows\" 

   Will return a list of all file paths >= 240 (default)
   characters in C:\Windows and below.

  .Example
   Get-LongFileNames.ps1 -BasePath "C:\Windows\" -MinLen 266

   Will return a list of all file paths >= 265
   characters in C:\Windows and below.

  .Example
   Get-LongFileNames.ps1 -BasePath "C:\Windows\" -WriteCSV

   Will write a csv file of all file paths >= 240
   characters in C:\Windows and below to the users
   Documents folder.

#>


# Two blank linea above required for Comment-Based Help

Param (
  [Parameter(Mandatory=$False)]
    [String] $BasePath,
  [Parameter(Mandatory=$False)]
    [Int32]  $MinLen = 240,
  [Parameter(Mandatory=$False)]
    [Switch] $WriteCSV
)

Function Show-Msg {
  Param ( [Parameter(Mandatory=$True, 
           HelpMessage="Message box content.")]
            [string]$Msg ,
          [Parameter(Mandatory=$False,
           HelpMessage="Message box title.")]
            [string]$Title = "Information"
        )          
[Windows.Forms.MessageBox]::Show("$Msg", "$Title", 
     [Windows.Forms.MessageBoxButtons]::OK , 
     [Windows.Forms.MessageBoxIcon]::Information) 

}  #End Function Show-Msg

# --- Main PGM ---

Add-Type -AssemblyName System.Windows.Forms

If ($BasePath -eq '') {
   $BasePath = 
   Read-Host "Enter your starting drive:\path with a final \`n"
}

$ErrorActionPreference = "SilentlyContinue"
Clear-Host

$GCIArgs = @{LiteralPath = "$BasePath" 
             Filter      = "*.*"
             Recurse     = $true
            }

If (-Not(Test-Path -LiteralPath "$BasePath")) {
  $SMArgs = @{Msg   = "The Path: $BasePath`n`n" +
                      "is not a valid path " + 
                      "on this machine."

              Title = "No Files Found"}
  Show-Msg @SMArgs | Out-Null
  Exit
}

$FileList = 
  Get-ChildItem @GCIArgs |
  Where-Object {$_.FullName.Length -ge $MinLen} | 
  Select @{Label="Length";Expression={$_.FullName.Length}},Fullname 

If ($FileList -ne $Null) {

  if ($WriteCSV.IsPresent) {

    #--- Create FileSpec for the CSV file ---
    $CSVPath = 
      "$([environment]::getfolderpath("mydocuments"))\" +
      "LongFileNames.csv"

    $FileList | Export-Csv -LiteralPath "$CSVPath"

    $SMArgs = @{Msg = "The list of files >= $MinLen " +
                      "characters was written to:`n`n" +
                      "$CSVPath"
              Title = "Output File Location"}
    Show-Msg @SMArgs | Out-Null 

  }
  Else {
    $FileList | Format-Table -Wrap
  }

} #End if $FileList.Count 

Else {
  $SMArgs = @{Msg   = "No files of a length >= $MinLen " +
                      "were found in the directory:`n`n" +
                      "$BasePath"
              Title = "No Files Found"}
  Show-Msg @SMArgs | Out-Null
}