﻿<#
.Synopsis
 Run a backup definition file (BDF) or several after
 disabling network and un-necessary processes & services.
 Then shutting down the computer.

.Description
 Run a backup definition file (BDF) or several after
 disabling network and un-necessary processes & services.
 Then shutting down the computer. The program writes
 both it's own log file and to the Event log, 
 Windows Logs/Application.

 Program will also setup a task to popup a reminder to backup
 based on a user specified interval and a run once job to dispaly
 the programs log file the next time the computer is booted.

.Parameter BDFiles
 Array of Backup Definition Files to execute.

.Parameter Network
 If the Source and/or destination of the Backup is on the
 Network specify this switch to prevent the network from being
 disabled. Does NOT work on Windows 7! 
 Note: if DESTINATION is a network drive the program will NOT 
       disable the network. If the Source is a network drive
       you can only do a File backup and the program will NOT
       disable the network. 

.Parameter CleanDisk
 If present this will run a disk cleanup with the current
 settings specified in /sageset 65535 and then reboot the 
 machine before running the specified BDF files.
 Make sure you have this sageset set to the options you want!
 See the user documentation for further information.

.Parameter ClearLog
 Clears the Log file before processing the current backup list.
 Note: This parameter is processed BEFORE CleanDisk so that
       upon reboot after CleanDisk the log file is not cleared
       of any messages from CleanDisk!

.Parameter NoShutdown
 Keeps the program from shutting the machine down at it's
 conclusion. For Testing purposes only!

.Parameter PowerPlan
 If specified the program will change the active Power Plan
 to the specified plan. Upon program completion the Power Plan
 will be changed back to the one active when the program
 was started. If the specified Power Plan can not be found no 
 change will be made and a message will be placed in the
 log file.

.Parameter Setup
 Creates the scheduled tasks and shortcuts necessary to run the
 program. Provides a menu interface to prompt for the necessary 
 Backup Definition Files (BDF) and Backup Name & reminder
 interval.

.Outputs
 Macrium-Reflect.log in the [Users Documents]\Reflect
 directory. Example:
07/25/19 04:16:01 Disk Cleanup started. 
07/25/19 04:16:30 Disk Cleanup ended. 
07/25/19 04:16:30 Initiated reboot to finish disk cleanup. 
07/25/19 04:28:50 Disk Cleanup started. 
07/25/19 04:29:18 Disk Cleanup ended. 
07/25/19 04:29:18 Initiated reboot to finish disk cleanup.
07/25/19 04:29:45 Network Disabled. 
07/25/19 04:29:58 Backup Set started. Program Version 4.04
07/25/19 04:30:11 Couldn't stop service MBAMService
07/25/19 04:30:21 Backup Started...
07/25/19 04:49:30 Backup Succeeded (G:\BEKDocs\Reflect\Boot Drive.xml).
					Summary File: DDAE4CBD575767DE-00-00.html
07/25/19 04:49:30 Backup Started...
07/25/19 05:03:27 Backup Succeeded (G:\BEKDocs\Reflect\Data Drive.xml).
					Summary File: 627BE9752AFE3EBF-00-00.html
07/25/19 05:03:27 Backup Set ended. Total Elapsed Time: 00:33:29
+------------------------------------------------------+
| Check the C:\ProgramData\Macrium\Reflect directory   |
| for .html files with the name indicated for details. |
+------------------------------------------------------+
07/25/19 05:03:27 Network reactivated!

Sample Errors:
03/26/18 11:45:50 Backup Set started. Program Version 4.04
03/26/18 11:45:51 Couldn't stop process GarminService
03/26/18 11:45:53 Drive I: not found. Please connect drive.
03/26/18 11:45:53 G:\BEKDocs\Reflect\Junk.xml File Not Found!
03/26/18 11:45:53 Backup Set ended. Total Elapsed Time: 00:00:03


 An entry is also written to the Event Log: Windows Log\Application
 Level         Date and Time          Source
 Information   3/10/2018 10:43:33 PM  Macrium Reflect

.Notes
 Programmer   : RetiredGeek (WSL/askWoody) 
                aka: ComputerMentor
 Created      : 10 Mar 2018
 Last Updated : 24 Sep 2019
 Current Vers : 04.05
 
 Some code borrowed from Macrium Reflect generated .PS1 files.

 Tested w/Macrium Reflect Free Version 6.3 Build 1885
          Macrium Reflect Paid Home    6.3 Build 1865

 Note: The program will not work properly on Version 7 either
       Home Trial or Free as Reflect does not exit when the
       backup job terminates. It will make the backup correctly
       but it will not write the final log messages or shut
       down the machine!

 Tested on: Windows 10 Pro  Build 17763.rs5_release.180914-1434
            Windows 10 Pro  Build 18362.19h1_release.190318-1202
            Windows 10 Home Build 16299.rs3_release.170928-1534
            Windows 10 Insider    17127.rs4_release.180318-0047

 Designed to be run from a Scheduled Task with the 
 Highest Privleges option selected! If you use separate User &
 Admin accounts you must install and run from the Admin Acct.!

 On Win 7 the neither the Network or MS Security Essentials
 will be stopped!

.Example
    [d:\path\]Start-ReflectBackup.ps1 -Setup
    
    Runs the graphical setup routine. Make sure you have already
    created the necessary Backup Definition Files (BDFs) and
	setup your Cleanmgr.exe options.  

.Example
    [d:\path\]Start-ReflectBackup.ps1 

    Will prompt for the BD Files to execute.

.Example
    $SRBArgs = @{Network = $True
                 BDFiles = $("Boot Drive.xml",
                             "Data Drive.xml"}
    [d:\path\]Start-ReflectBackup.ps1 @SRBArgs

    Runs the specified BDF files w/o Network!

.Example
    $SRBArgs = @{BDFiles = $("NAS Pictures.xml",
                             "NAS-NoPics.xml")
                 Network = $True 
                }
    [d:\path\]Start-ReflectBackup.ps1 @SRBArgs
	
	Keeps the Network active if the Source or Destination of
	the backup requires such access.

.Example
    $SRBArgs = @{BDFiles = $("NAS Pictures.xml",
                             "NAS-NoPics.xml")
                 CleanDisk  = $True 
                 ClearLog   = $True
                }
    [d:\path\]Start-ReflectBackup.ps1 @SRBArgs
	
	Clears the backup drive of .mrimg & .mrbak files and will
    run Cleanmgr.exe /sagerun 65535 and then reboot before
	starting the backup.
    It will also clear the Log file before proceeding.

#>


# Two blank linea above required for Comment-Based Help 

Param(
      [Parameter(Mandatory=$False)]
        [String[]] $BDFiles ,
      [Parameter(Mandatory=$False)]
        [Switch]   $Network ,
      [Parameter(Mandatory=$False)]
        [Switch]   $CleanDisk ,
      [Parameter(Mandatory=$False)]
        [Switch]   $ClearLog,
      [Parameter(Mandatory=$False)]
        [String]   $PowerPlan,
      [Parameter(Mandatory=$False)]
        [Switch]   $NoShutdown,
      [Parameter(Mandatory=$False)]
        [Switch]   $Setup      
)

Function Backup {

  Param(
        [Parameter(Mandatory=$True)]
        [string] $BDFName
  )

  "$(Get-DTStr) Backup Started..." >> "$LogFile"

  $strArgs = "-e -w -Full `"$XMLFilePath\$BDFName`""
  $SPArgs  = @{FilePath     = $ReflectPath 
               ArgumentList = $strArgs 
               PassThru     = $True
               Wait         = $True
              }

  $BkUpResult   = (Start-Process @SPargs).ExitCode

  $StatMsg = "Backup Succeeded","Backup Failed",
             "XML File Invalid"

  $BkUpSummary = 
    Get-ChildItem -Path C:\ProgramData\Macrium\Reflect\*.html |
    Sort-Object lastwritetime -Descending | Select-Object -first 1 Name

  $(Get-DTStr) + " " + ($StatMsg[($BkUpResult)]) +
     " ($($XmlFilePath)\$($BDFName)).`r`n`t`t`t`t`t" +
     "Summary File: $($BkUpSummary.Name)" >> "$LogFile"

  return $BkUpResult  

} #End Function Backup

Function New-NoUACTask {

<#
  .Synopsis
   Create Task to get around UAC Prompt for
   programs requiring Administrative Privileges.

  .Description
   Create Task with the supplied parameters to get around 
   UAC Prompt for programs requiring Administrative Privileges. 

  .Parameter TaskName
   String value to be used as the Task Scheduler task name.

  .Parameter TaskCommand
   Full drive, path, filename of the executable command.

  .Parameter TaskArg
   Any arguments to the TaskCommand

  .Parameter TaskDesc
   A brief description of the command for later reference.

  .Parameter Laptop
   True/False should the timing restrictions for battery operated
   devices be enabled/disables in the task definition.

  .Outputs
    A task scheduler task with administrative privileges.

  .Notes
   Function     : New-NoUACTask                                      
   Description  : Create Task to get around UAC Prompt                  
   Programmer   : RetiredGeek (WSL) aka: ComputerMentor                 
   Created      : 21 Nov 2014                                           
   Last Updated :                                           
     12/06/14 : Added laptop settings to allow on battery operations. 
                Set Stop on Idle end to FALSE    
     01/26/17 : Added Splatting & comment based help.                                

  .Example
   $NoUACArgs = 
     @{TaskCommand = "c:\windows\system32\windowspowershell" +
                     "\v1.0\powershell.exe"
       TaskName    = "PowerShell CMD Shell as Administrator" 
       LapTop      = $False
       TaskArgs    = "g:\scripts\mypsscript.ps1"
       TaskDesc    = "Run the mypsscript.ps1 program as Administrator" 
      }
    
   New-NoUACTask @NoUACArgs

#>


# Two blank linea above required for Comment-Based Help

param (
        [Parameter(Mandatory=$true)]
          [String]$TaskName,
        [Parameter(Mandatory=$true)]
          [String]$TaskCommand,
        [Parameter(Mandatory=$false)]
          [String]$TaskArg,
        [Parameter(Mandatory=$false)]
          [String]$TaskDesc="No Description Supplied",
        [Parameter(Mandatory=$false)]
          [Boolean]$Laptop=$false,
        [Parameter(Mandatory=$false)]
          [Switch]$Trigger
      )

  $TaskSet = New-ScheduledTaskSettingsSet -Compatibility Win7 
  $TaskSet.DisallowStartIfOnBatteries = $Laptop
  $TaskSet.StopIfGoingOnBatteries     = $Laptop
  $TaskSet.IdleSettings.StopOnIdleEnd = $false
  
  $TAArgs = @{Execute  = '"' + $TaskCommand + '"'
              Argument = $TaskArg 
             }
  
  $TaskAction = New-ScheduledTaskAction @TAArgs 

  $RSTArgs = @{Action      = $TaskAction  
               TaskName    = "$TaskName" 
               User        = "$env:USERDOMAIN\$env:username" 
               RunLevel    = "Highest" 
               Description = "$TaskDesc" 
               Settings    = $TaskSet
               Force       = $True}

  If($Trigger.IsPresent) {
    $Auto = New-JobTrigger -AtLogOn
    Register-ScheduledTask @RSTargs -Trigger $Auto | Out-Null
  }
  Else {
        Register-ScheduledTask @RSTargs | Out-Null
  }
  
}   #End Function New-NoUACTask

Function New-NoUACShortcut {

<#
  .Synopsis
   Creates a shortcut to run the specified program with
   supplied arguments and places it in the location selected.

  .Description
    

  .Parameter ShortcutName
   The Name of the Task/Shortcut to create

  .Parameter ShortcutCommand
   The Command to be Executed by the Shortcut

  .Parameter DestinationPath
   The storage location to store the created shortcut

  .Parameter ShortcutIcon
   Is the path & number of the icon to use.
   ex: "C:\MyIcon.ico,0"
   Mutually exclusive with UseTaskCommandIcon!

  .Parameter UseTaskCommandIcon
   Will select the first icon in the ShortcutCommand file.
   Mutually exclusive with ShortcutIcon!

  .Outputs
   A shortcut (.lnk) file in the specified location. 

  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 24 Nov 2014                                              
     Current Vers : 1.4                                                      
     Last Updated : 28 Nov 2014 

  .Example
   $NoUACSCArgs = 
     @{ShortcutName       = "My Admin Level CMD" 
       DestinationPath    = C:\Userid\Desktop 
       ShortcutCommand    = "C:\Windows\...\program.exe"
       UseTaskCommandIcon = $True
      }

   New-NoUACShortcut @NoUACSCArgs

#>


# Two blank linea above required for Comment-Based Help

param ( [Parameter(Mandatory=$true)]
          [String]$ShortcutName,
        [Parameter(Mandatory=$false)]
          [String]$ShortcutCommand,
        [Parameter(Mandatory=$false)]
          [String]$DestinationPath="$env:UserProfile\Desktop", 
        [Parameter(Mandatory=$false)]
          [String]$ShortcutIcon="NO",
          [switch]$UseTaskCommandIcon
      )

  $WshShell = New-Object -comObject WScript.Shell
  $Shortcut = $WshShell.CreateShortcut(
              "$DestinationPath\$ShortcutName.lnk")
  $Shortcut.TargetPath = "C:\Windows\System32\schtasks.exe" 
  $Shortcut.Arguments = "`/run `/TN " + '"' + $ShortcutName + '"'
  $Shortcut.WindowStyle = 7
  
  If ($ShortcutIcon -ne 'NO') {
    $Shortcut.IconLocation = $ShortcutIcon
  }
  Else {
      If ($UseTaskCommandIcon.IsPresent) {
        $Shortcut.IconLocation = $ShortcutCommand
      }
  }
  $Shortcut.Save()
  
}    #End Function New-NoUACShortcut

Function Get-DTStr {

  $(Get-Date).ToString("MM/dd/yy") + " " +
  $(Get-Date).ToString("hh:mm:ss")

} #End Function Get-DTStr

Function Set-BackupName {
<#
  .Synopsis
   Check to see if the Passed BackupName is already 
   in use.

  .Description
   Check to see if the Passed BackupName is already 
   in use. Provide a popup message if the name is in
   use by either a Scheduled Task and/or Shortcut. 
   Returns a value of $False if the name is in use or
   $True if the name is not found.   

  .Parameter BackupName

  .Outputs
   Message Box.
    
  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 24 Apr 2018
     Last Updated : 
     Current Vers : 1.0

  .Example
   Set-BackupName -BackupName "SystemBackup"

#>


# Two blank linea above required for Comment-Based Help

Param (
     [Parameter(Mandatory=$FalSe)]
        [String] $BackupName 
)

#-- Already loaded in Main ---
#  Add-Type -AssemblyName "System.Windows.Forms"

  $StatusMsg = {
    [Windows.Forms.MessageBox]::Show($Msg,
                                 "Status Message:", 
    [Windows.Forms.MessageBoxButtons]::OK , 
    [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

  If ("" -eq $BackupName) {
    $Msg = "Backup Name is BLANK, Please Correct"
    & $StatusMsg
    Return ,$False
  }

  Else {

    $ValidBkUpName = $True
    $Msg = $Null
    $TaskExists = Get-ScheduledTask | 
      Where-Object {$_.TaskName -like "MR $BackupName" }
  
    if($TaskExists) {$ValidBkUpName = $False
                     $Msg = "Scheduled Task Name"}
  
    $GCIArgs = @{Path    = 
                           "$([environment]::getfolderpath(
                           "mydocuments"))\" +
                           $BackupName + ".lnk"
                 Recurse = $True}
    $x = Get-ChildItem @GCIArgs 
    If (-not ($null -eq $x) ) { 
      $ValidBkUpName = $False 
  	If ($null -eq $Msg ) {
  	  $Msg = "Shortcut Name"
  	}
  	Else { $Msg = "$Msg and Shortcut Name"}
    }
    
    If (-not($Null -eq $Msg)) {
   
      $Msg = "The name: $BackupName is" +
  	      " already being used`nby a $Msg."
  		  
      & $StatusMsg	
      
      Return ,$False	  
    }
    Else { Return ,$True }
  
  } #Else - If ($Null -eq $BackupName)
  
} #End Set-BackupName  

Function Set-Setup {

    Function Test-OKBtnStatus {
    
         If ( (-not($Null -eq $tboxBkUpName.Text)) -and
              (-not($Null -eq $tboxDays.Text))     -and
              ($BDFListBox.SelectedItems.Count -ne 0)){
              $OKButton.Visibility = "Visible"
         }
    	 Else { $OKButton.Visibility = "Hidden"}
    	 
    } #End Test-OKBtnStatus 
 
  $BDFiles = Get-ChildItem -Path "$XMLFilePath" -Filter "*.xml" 

  $GWOArgs = @{Namespace = 'root\cimv2\power'
               Class     = 'Win32_PowerPlan' }

  $CurrentPPS = $(Get-WmiObject  @GWOArgs).ElementName

  $SSDArgs = @{BDFItems    = $($BDFiles.name) 
               PPItems     = $CurrentPPS
               MultiSelect = $True
               Title       = "Backup Job Parameters " +
                             "- Pgm Vers: $PgmVersion"}

  $SelectedBDFs = Show-SetupDialog @SSDArgs

  If ($Null -eq $SelectedBDFs ) {
    $Message = "User Canceled Setup`n`n" +
               "No changes made to your system."
    & $TermMsg
    Exit
  }

  #Prepend MR to BkUpName to distinguish it easily!
  $Script:BkUpName = "MR $Script:BkUpName"

  $BDFList = '$('

  ForEach ($x in $SelectedBDFs) {
        $BDFList += "'" + $x + "',"
  }
  $BDFList = $BDFList.Substring(0,$BDFList.Length - 1) + ')'

  $Options = ""
  If ($ClearLogFlag)          { $Options += " -ClearLog" }
  If ($CleanDiskFlag)         { $Options += " -CleanDisk" }
  If ($NetworkFlag )          { $options += " -Network" }
  If ($Script:SetupPP -ne "") { $Options += $(" -PowerPlan " +
                                  "'" + $Script:SetupPP + "'")}

#--- Set up Scheduled Task for Running Backup ---

  $NoUACArgs = 
   @{TaskName    = "$Script:BkUpName"
     TaskCommand = "$([environment]::SystemDirectory)" +
                   "\WindowsPowerShell\v1.0\powershell.exe"
     TaskArg     = "-WindowStyle Minimized -Command " + '"' +
                   $PSScriptRoot + "\Start-ReflectBackup.ps1" + 
                   '"' + "$Options" + " -BDFiles $($BDFList)"
     Laptop      = $False
    }

  New-NoUACTask @NoUACArgs 

#--- Setup Shortcut for running Backup Scheduled Task ---

  $NoUACSCArgs = 
         @{ShortcutName       = "$Script:BkUpName" 
           DestinationPath    = "$env:UserProfile\Desktop" 
           ShortcutCommand    = "$($NoUACArgs.TaskCommand)"
           UseTaskCommandIcon = $True
          }
     
  New-NoUACShortcut @NoUACSCArgs

#--- Set up Task for Backup Reminder ---

  $NoUACArgs = 
   @{TaskName    = "$Script:BkUpName Reminder"
     TaskCommand = "$([environment]::SystemDirectory)" +
                   "\WindowsPowerShell\v1.0\powershell.exe"
     TaskArg     = "-WindowStyle Minimized -Command " + '"' +
                   $PSScriptRoot + 
                   "\Display-TimeToImageMsg.ps1" + 
                   '"'  + " -Threshold $Script:DaysToNextBkUp"
     Laptop      = $False
     Trigger     = $True
    }

  New-NoUACTask @NoUACArgs 

#  Add-Type -AssemblyName "System.Windows.Forms"

  $StatusMsg = {
    [Windows.Forms.MessageBox]::Show($Msg,
                                 "Status Message:", 
    [Windows.Forms.MessageBoxButtons]::OK , 
    [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

    $Msg = "Tasks:`t$Script:BkUpName`n" +
                 "`t$Script:BkUpName Reminder`n" +
           "Shortcut:`t$($Script:BkUpName).lnk`n`n`t Created."
    & $StatusMsg

} #End Function Set-Setup

 Function Show-SetupDialog {

   [CmdletBinding()]
   [OutputType([String[]])]

   Param( 
     [Parameter(Mandatory=$True)] [Alias('BI')]
        [array] $BDFItems, 
     [Parameter(Mandatory=$True)] [Alias('PI')]
        [array] $PPItems, 
     [Parameter(Mandatory=$False)][Alias('T')]
        [String] $Title = "", 
     [Parameter(Mandatory=$False)][Alias('MS')]
        [Switch] $MultiSelect
   )           

    $cboxHeight   =  35
    $cboxWidth    = 400
    $cboxFontSize =  18
    $VSpacing     =  10
    $WinHeight    = 700
    $WinWidth     = 500
    $OKLeftMargin =  70


    [xml]$xaml = @"
    <Window
      xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
      xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
      Name                  = "Window" 
      Title                 = "$Title"
      WindowStartupLocation = "CenterScreen"
      Width                 = "$WinWidth" 
      Height                = "$WinHeight" 
      ShowInTaskbar         = "True"
      Background            = "Blue" 
      FontSize              = "18">

    <Window.Resources>

      <Style x:Key="TBoxStyle" TargetType="TextBox">
       <Setter Property="Height"     Value="30"  />
       <Setter Property="Width"      Value="250"   />
       <Setter Property ="VerticalAlignment"   Value="Top"  />
       <Setter Property ="HorizontalAlignment" Value="Left" />
       <Setter Property="Background" Value="White"          />
       <Setter Property="Foreground" Value="Blue"           />
       <Setter Property="FontSize"   Value="18"  />
       <Setter Property="IsReadOnly" Value ="$False"        />
       <Setter Property="VerticalScrollBarVisibility" 
                                                Value="Auto"/>
      </Style>

      <Style x:Key="CkBoxStyle" TargetType="CheckBox">
       <Setter Property="VerticalAlignment"   Value="Top"  />
       <Setter Property="HorizontalAlignment" Value="Left" />
       <Setter Property="Height"     Value="$cboxHeight"   />
       <Setter Property="Width"      Value="$cboxWidth"    />
       <Setter Property="FontSize"   Value="$cboxFontSize" />
       <Setter Property="Foreground" Value="White"/>
      </Style>

      <Style x:Key="LabelStyle" TargetType="Label">
       <Setter Property="VerticalAlignment"   Value="Top"  />
       <Setter Property="HorizontalAlignment" Value="Left" />
       <Setter Property="Height"     Value="$cboxHeight"   />
       <Setter Property="Width"      Value="$cboxWidth"    />
       <Setter Property="FontSize"   Value="$cboxFontSize" />
       <Setter Property="Foreground" Value="White"/>
      </Style>
	  
    </Window.Resources>

      <Grid>
          <Label Name    = "BackupName_Label" 
                 Content = "Backup Name:" 
                 Style   = "{StaticResource LabelStyle}"
                 Margin  = "10,$VSpacing,0,0" />
   
          <TextBox Name   = "tboxBkUpName" 
                   Style  = "{StaticResource TBoxStyle}"
                   Margin = "220,$VSpacing,0,0" />
   
          <Label Name    = "Reminder_label" 
                 Content = "Reminder Days:" 
                 Style   = "{StaticResource LabelStyle}"
                 Margin  = "10,$(($VSpacing+=40)),0,0" />
   
          <TextBox Name="tboxDays" 
                     Style  = "{StaticResource TBoxStyle}"
                     Text   = '30'
                     Margin = "220,$VSpacing,0,0" />

          <Label Name = "Label_DBF" 
                 Content = "Select BDF Files for this backup task:"
                 Style   = "{StaticResource LabelStyle}" 
                 Margin  = "10,$(($VSpacing+=40)),0,0"/>

          <ListBox Name="BDFFiles" 
                   Height = "150" Width = "370" 
                   HorizontalAlignment  = "Left" 
                   VerticalAlignment    = "Top" 
                   Background           = "White"
                   Foreground           = "Black" 
                   Margin = "10,$(($VSpacing+=40)),0,0"/>

          <Label Name = "PPlan_Label" 
                 Content = "Select desired Power Plan: (optional)"
                 Style   = "{StaticResource LabelStyle}" 
                 Margin  = "10,$(($VSpacing+=165)),0,0"/>

          <ListBox Name="PwrPlans" 
                   Height = "150" Width = "370" 
                   HorizontalAlignment  = "Left" 
                   VerticalAlignment    = "Top" 
                   Background           = "White"
                   Foreground           = "Black" 
                   Margin     = "10,$(($VSpacing+=40)),0,0"/>

          <CheckBox Name     = "CBoxClearLog" 
                   IsChecked = "$($Script:ClearLogFlag)"
                   Content   = "Clear Log File?" 
                   Style     = "{StaticResource CkBoxStyle}"
                   Margin    = "10,$(($VSpacing+=165)),0,0"/>

          <CheckBox Name     = "CBoxNetwork" 
                   IsChecked = "$($Script:NetworkFlag)"
                   Content   = "Leave Network Active?" 
                   Style     = "{StaticResource CkBoxStyle}"
                   Margin    = "10,$(($VSpacing+=40)),0,0"/>

          <CheckBox Name     = "CBoxCleanDisk" 
                   IsChecked = "$($Script:CleanDiskFlag)"
                   Content   = "Run Disk Cleanup before backup?" 
                   Style     = "{StaticResource CkBoxStyle}"
                   Margin    = "10,$(($VSpacing+=40)),0,0"/>

          <Button Name="OKButton" 
                  Content="OK" 
                  Height = "30" Width = "75" 
                  HorizontalAlignment = "Left" 
                  VerticalAlignment   = "Top"
                  Background          = "Green"
                  Foreground          = "Yellow"
                  Visibility          = "Hidden"
                  Margin = "$OKLeftMargin,$($WinHeight-80),0,0"/>

          <Button Name="CancelButton" 
                  Content="Cancel" 
                  Height = "30" Width = "75" 
                  HorizontalAlignment = "Left" 
                  VerticalAlignment   = "Top" 
                  Background          = "Red"
                  Foreground          = "Yellow"
                  Visibility          = "Visible"
                  Margin = "$($($WinWidth) - $($OKLeftMargin+75)),$($WinHeight-80),0,0"/>
      </Grid>
    </Window>
"@
 
   $reader=(New-Object System.Xml.XmlNodeReader $xaml)
   $Window=[Windows.Markup.XamlReader]::Load( $reader )

   $tboxBkUpName = $Window.FindName("tboxBkUpName")
   $tboxBkUpName.Focus() = $True
   $tboxBkUpName.TabIndex = "1"
   $tboxBkUpName.Add_LostFocus({
     If (-not ($CancelButton.IsMouseOver)) {
       If (Set-BackupName -BackupName $tboxBkUpName.Text) {
         $Script:BkUpName = $tboxBkUpName.Text
       }
       Else {$tboxBkUpName.Text = ""}
       Test-OKBtnStatus
     } #End If ($CancelButton.IsFocused)
   })  #End $tboxBkUpName.Add_LostFocus

   $tboxDays = $Window.FindName("tboxDays")
   $tboxDays.TabIndex = "2"
   $tboxDays.Add_GotFocus({
     If ($tboxBkUpName.Text -eq ""){
       $TboxBkUpName.Focus() = $True
     }
   })
   $tboxDays.Add_LostFocus({
     If ($Null -eq $tboxDays.Text) { 
       $Script:DaystoNextBkUp = "30"}
     Else {
       $Script:DaysToNextBkUp = $tboxdays.Text}
       Test-OKBtnStatus
    })

   $BDFListBox = $Window.FindName("BDFFiles")
   $BDFListBox.TabIndex = "3"
   # Valid MultiSelect Values: Single, Multiple, Extended
   If ($MultiSelect) { $BDFListBox.SelectionMode = "Extended"}
   ForEach ($Item in $BDFItems) { 
              [void] $BDFListBox.Items.Add($Item) }
   $BDFListBox.Add_SelectionChanged({
     If($BDFListBox.SelectedItems.Count -gt 0){Test-OKBtnStatus}
    })

   $PPListBox = $Window.FindName("PwrPlans")
   $PPListBox.TabIndex = "4"
   # Valid MultiSelect Values: Single, Multiple, Extended
   If ($MultiSelect) { $PPListBox.SelectionMode = "Single"}
   ForEach ($Item in $PPItems) { 
              [void] $PPListBox.Items.Add($Item) }
   $PPListBox.Add_SelectionChanged({$Script:SetupPP = 
                                    $PPListBox.SelectedItems})

   $CBoxClearLog = $Window.FindName("CBoxClearLog")
   $CboxClearLog.TabIndex = "5"
   $CBoxCLearLog.Add_Checked(   {$Script:ClearLogFlag = $True} )
   $CBoxClearLog.Add_UnChecked( {$Script:ClearLogFlag = $False})

   $CBoxNetwork = $Window.FindName("CBoxNetwork")
   $CBoxNetwork.TabIndex = "6"
   $CBoxNetwork.Add_Checked(   {$Script:NetworkFlag = $True} )
   $CBoxNetwork.Add_UnChecked( {$Script:NetworkFlag = $False})

   $CBoxCleanDisk = $Window.FindName("CBoxCleanDisk")
   $CboxCleanDisk.TabIndex = "7"
   $CBoxCleanDisk.Add_Checked(   {$Script:CleanDiskFlag = $True} )
   $CBoxCleanDisk.Add_UnChecked( {$Script:CleanDiskFlag = $False})

   $OKButton = $Window.FindName("OKButton")
   $OKButton.TabIndex = "8"
   $OKButton.Visibility = "Hidden"
   $OKButton.Add_Click({
         $Script:OKButtonClick     = $True
         $Script:CancelButtonCLick = $False
         $Window.Close() })

   $CancelButton = $Window.FindName("CancelButton")
   $CancelButton.TabIndex = "9"
   $CancelButton.Add_GotFocus({
     If ($tboxBkUpName.Text -eq ""){
       $Script:CancelButtonClick = $True
       $Script:OKButtonClick     = $False
       $Window.Close()}
   })
   $CancelButton.Add_Click({
         $Script:CancelButtonClick = $True
         $Script:OKButtonClick     = $False
         $Window.Close() })

   $Window.Title = $Title

   $Window.ShowDialog() | Out-Null
   
   If ($Script:OKButtonClick) { $BDFListBox.SelectedItems }

} #End Function Show-SetupDialog

#-----------  Main Program --------------

$PGMVersion     = 4.05
$ReflectPath    = "C:\Program Files\Macrium\Reflect\Reflect.exe"
$XMLFilePath    = 
      "$([environment]::getfolderpath("mydocuments"))\Reflect" 
$LogFile        = "$($XMLFilePath)\Macrium-Reflect.log"
$BkUpDrive      = ""  #Populated by the program from the BDFs.
$DaysToNextBkUp = 30
$IgnoreNetwork  = $False
$NetworkDrive   = $False
$CleanDiskFlag  = $False
$ClearLogFlag   = $False
$NetworkFlag    = $False
$SetupPP        = ""        #Populated by the program.
$StartTime      = Get-Date  #Save to calculate elapsed time.

Add-Type -AssemblyName "System.Windows.Forms"

$TermMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
                                   "Program Terminated:", 
  [Windows.Forms.MessageBoxButtons]::OK , 
  [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

$BDFMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
                                   "Reflect BDF Error:", 
  [Windows.Forms.MessageBoxButtons]::OK , 
  [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

If ($host.Name -eq 'ConsoleHost' -or
    $host.Name -eq 'Visual Studio Code Host') {
 
   try{<#+------------------------------------------------+
         | Check that the proper assemblies are loaded    |
         | Required for PS Console and Visual Code, while |
         | only Systems.Windows.Forms needed for PSISE!   |
         +------------------------------------------------+
       #>
     $ATArgs = @{AssemblyName = "PresentationCore",
                                "PresentationFramework",
                                "WindowsBase"
                 ErrorAction = 'Stop'}
     Add-Type @ATArgs
   } 
   catch {
     $Message = 
      "Failed to load Windows Presentation Framework" +
      " and/or other assemblies required for this program!"
      & $TermMsg
     Exit 
   }
   
 } #End If ($host...

$identity = 
   [Security.Principal.WindowsIdentity]::GetCurrent()
$principal = 
   New-Object Security.Principal.WindowsPrincipal $identity

If (-NOT ($principal.IsInRole(
    [Security.Principal.WindowsBuiltinRole]::Administrator))) 
  {$Message = "Program requires Administrative Privledges`n" +
              "Please exit powershell and restart using"     +
              " `"the`nRun as administrator`" option on the" +
              " right-click options menu."
   & $TermMsg
   Exit
  } 

If ($Setup.IsPresent) { 

  $CurOS = Get-CimInstance -ClassName  Win32_OperatingSystem
  $Parts = $curos.version.split('.') 

  If (( [Int]$Parts[0] -lt 6) -or 
      (([Int]$Parts[0] -eq 6) -and 
       ([Int]$Parts[1] -lt 2))) {
    $Message = "You are running Windows 7 or less.`n" +
               "Windows 8.0 or greater is required for`n" +
               "Automatic setup.`n`n" +
               "Please refer to the documentation in the`n" +
               "zip file for instructions to manually`n" +
               "create the necessary Tasks and shortcuts."
    & $TermMsg
  }

  Else { Set-Setup }

  Exit
}

Try {

   $GCIArgs = @{ClassName = "Win32_Battery"
                ErrorAction = 'Stop'}
   If ((Get-CimInstance @GCIArgs ).BatteryStatus -eq 1) {
     $Message = "Your machine is currently running on battery" + 
                " power.`n`nPlease plug it in then click OK"   +
                " otherwise click Cancel."
     $OnBattery = 
       [Windows.Forms.MessageBox]::Show($Message,
                                      "Laptop on Battery:", 
       [Windows.Forms.MessageBoxButtons]::OKCancel , 
       [Windows.Forms.MessageBoxIcon]::Warning) 
     If ( $OnBattery -eq "Cancel" ) {Exit}
   
   } #End If ((Get-Cim...

} #End Try

Catch {}  #If error machine does not have a battery!

If ($ClearLog.IsPresent) {
  Remove-Item -Path "$LogFile" -Force
}

If ($CleanDisk.IsPresent ) {

  $RebootArgs = ""

  If ($Network.IsPresent)    { $RebootArgs += " -Network"    }
  If ($NoShutdown.IsPresent) { $RebootArgs += " -NoShutdown" }
  If ($PowerPlan -ne "")     { $RebootArgs += 
                               (" -PowerPlan '" + $PowerPlan + "'" )}

  $BDFList = ' -BDFiles $('

  ForEach ($x in $BDFiles) {
        $BDFList += "'" + $x + "',"
  }

  $BDFList = $BDFList.Substring(0,$BDFList.Length - 1) + ')'
  $RebootArgs += $BDFList

  "$(Get-DTStr) Disk Cleanup started. "  >> $LogFile
  $SPArgs = @{FilePath     = "CleanMgr.exe" 
              ArgumentList = "/sagerun 65535"
              WindowStyle  = "Hidden" 
              Wait         = $True}
  Start-Process @SPArgs
  $GPArgs = @{Name        = "cleanmgr,dismhost"
              ErrorAction = "SilentlyContinue"}
  Get-Process @GPArgs  | Wait-Process

 "$(Get-DTStr) Disk Cleanup ended. "  >> $LogFile 

  "$(Get-DTStr) Initiated reboot to finish disk " +
               "cleanup. "  >> $LogFile 

  $NIPArgs = 
   @{Path  = "HKLM:\Software\Microsoft\Windows\" +
                    "CurrentVersion\RunOnce" 
     Name  = "Start-ReflectBackup"
     Value = ('C:\Windows\System32\WindowsPowerShell\' +
              'v1.0\Powershell.exe ' +
                '-executionPolicy Unrestricted ' +
                '-WindowStyle Minimized ' +
                "-Command $PsScriptRoot\" +
                "Start-ReflectBackup.ps1 $RebootArgs")
     Force = $True}
  Set-ItemProperty @NIPArgs | Out-Null
  
  Restart-Computer -Force

  Exit

} #End If($CleanDisk...

<#+-------------------------------------------------------+
  | Check passed Backup Definition Files (BDFs) for the   |
  | backup drive and directory of the backup device then  |
  | insure that those locations exist!                    |
  +-------------------------------------------------------+
#>

$BDFErrorCnt = 0

$BkUpDefFileDir = 
  "$([environment]::getfolderpath("mydocuments"))\Reflect"

ForEach ($BDF in $BDFiles) {

  $BDFFSpec = "$BkUpDefFileDir\$BDF"

  If (Test-Path -Path "$BDFFSpec") {
    [xml]$BDFData = Get-Content -force -raw -Path "$BDFFSpec"
    $DestDir = $BDFData.backup_definition.destination.directory
    $SrcDir  = $BDFData.backup_definition.source.folder_filter.folder
    If ($SrcDir -ne "") {
      If ($SrcDir.Substring(0,2) -eq '\\') {$NetworkDrive = $True}
    }
    Clear-Variable "BDFData"

    If ($DestDir.GetType().Name -eq 'String') {

      $BkUpDrive = $DestDir.Substring(0,2)
      #Note: If DestDir is a network drive $BkUpDrive = \\
      If ($BkUpDrive -eq '\\') {$NetworkDrive = $True}

    } #End If ($DestDir.GetType...
    
    Else { #Macrium Reflect find Disk Letter by UID

        $xUID = $Destdir.'#text' -Split("}") 
        $UID = $xUID[0] + '}\'
        $VolInfo = Get-Volume -UniqueId $UID
        $BkUpDrive = $VolInfo.DriveLetter + ':'
        $DestDir   = $BkUpDrive + $xuid[1]
        
    } #End Else
  
    If (-not (Test-Path -Path "$DestDir")) {

      $Message = 
      "Backup Destination Directory: $DestDir " +
      "`nspecified in: $BDF Not Found!" +
      "`nDrive may not be connected or the directory does not" +
      "exist.`n`n Please correct and retry."
      & $BDFMsg
      $BDFErrorCnt += 1

    } #End If (-not (Test-Path -Path "$DestDir"))

  } #End If (Test-Path -Path "$BDFFSpec")

  Else {
      $Message =
        "Backup Defintion File $BDFFSpec not found!"
      & $BDFMsg
      $BDFErrorCnt += 1
  }
  
} #ForEach ($FN...

If ($BDFErrorCnt -gt 0) { Exit }

#----- End BDF Error Check ----------------

"$(Get-DTStr) Backup Set started. " +
     "Program Version " + $PGMVersion -f "00.00" >> $LogFile

#Change Power Plan (Mostly for Laptops)

If ($PowerPlan -ne "") {

  #--- Save currently Active Power Plan ---
  $GWOArgs = @{Namespace = 'root\cimv2\power'
               Class     = 'Win32_PowerPlan' 
               Filter    = "IsActive = $True"}
  $SavedPowerPlan = Get-WmiObject  @GWOArgs
  
  #Find PowerPlan for Backup and Activate
  Try {
        $NewPowerPlan = $False
        $GWOArgs = @{Namespace  = 'root\cimv2\power'
                    Class       = 'Win32_PowerPlan' 
                    Filter      = 'ElementName = "' + 
                                  $PowerPlan + '"'
                    ErrorAction = "Stop"}
        $NewPowerPlan = Get-WmiObject  @GWOArgs
        $NewPowerPlan.Activate()
        "$(Get-DTStr) Activated: $($NewPowerPlan.ElementName)" +
        " power plan." >> $LogFile
        $NewPowerPlan = $True
      }
  Catch {"$(Get-DTStr) Specified: $PowerPlan " +
         "Power Plan not found!`n" +
         "         The $($SavedPowerPlan.ElementName)" + 
         " power plan remains in effect." >> $LogFile} 

} #End If ($PowerPlan...

#Disable Network?

Try {
      If ( (-not($Network.IsPresent)) -and
           (-not($NetworkDrive))) {
        $NICs = Get-NetAdapter -Name * | 
                Where-Object {$_.Status -eq "Up"}
        $NICs | disable-NetAdapter -Confirm:$False
       "$(Get-DTStr) Network Disabled." >> $LogFile

      }
}

Catch [System.Management.Automation.CommandNotFoundException] {
     "$(Get-DTStr) Get-NetAdapter CmdLet NOT available." +
     "Network remains active!" >> $LogFile
     $IgnoreNetwork = $True
}

<#+-----------------------------------------------------------+
  | Turn off Windows Defender real-time monitoring if network |
  | off! This code can be deleted if you are not using WD or  |
  | modified to disable your prefered product's RTM.          |
  +-----------------------------------------------------------+
#>

If ( (-not($Network.IsPresent)) -or $IgnoreNetwork) {

  Try{
       Set-MpPreference -DisableRealtimeMonitoring $True
  }
  Catch {
   "$(Get-DTStr) Windows Defender isn't installed or attempt" +
   " to disable real time monitoring failed!" >> $LogFile
  }

} #End If ( (-not($Net...

<#
  +-----------------------------------------------------------+
  | Stop Services:                                            |
  | Use get-service to find names to customize the list below |
  | for the machine you're working on. If there are multiple  |
  | services for a particular item use the *.                 |
  +-----------------------------------------------------------+
#>

$SVCName = @{'Apple'        = 'Apple Mobile*'
             'Bonjour'      = 'Bonjour*'
             'Macrium'      = 'MacriumService'
             'Synergy'      = 'Synergy'
             'TeamViewer'   = 'TeamViewer*'
             'Remote Mgmt'  = 'WinRM'
             'Win Search'   = 'WSearch'
             'Win Update'   = 'wuauserv'}

ForEach ($SN in $SVCName.Values) {

  Try {
    Get-Service -name $SN -EA Stop > $Null
    $FoundSvc = $True
  }
  Catch {
         "$(Get-DTStr) Service $SN not running! >> $LogFile" 
         $FoundSvc = $False
  }

  If ($FoundSvc) {
    Try   { Stop-Service -Name $SN -Force -EA Stop > $Null }
    Catch [Microsoft.PowerShell.Commands.ServiceCommandException]
          {"$(Get-DTStr) Couldn't stop service $SN" >> $LogFile}
  } #End If ($FoundSvc..

} #End ForEach ($SN...

<#
  +-----------------------------------------------------------+
  | Stop Processes:                                           |
  | Use get-process to find names to customize the list below |
  | for the machine you're working on. If there are multiple  |
  | processes for a particular item use the *. Note: if there |
  | is a service involved stop the service first before       |
  | checking what processes are running!                      |
  +-----------------------------------------------------------+
#>

$ProcName = @{'Apple'      = 'AppleMobile*'
              'Foxit'      = 'FoxitConnected*'
              'Garmin'     = 'GarminService'
              'HP'         = 'hpwu*'
              'Nitro'      = 'NitroPDF*'
              'OneNote'    = 'ONENOTE*'
              'RoboForm'   = 'robotask*'
              'Synergy'    = 'Synergy'
              'TeamViewer' = 'TeamViewer*'
              'WinPatrol'  = 'WinPatrol*'
             }

ForEach ($PN in $ProcName.Values) {

  Try {$TestPN = Get-Process -Name "$PN" -EA Stop }
  Catch {}
  if ($Null -ne $TestPN) {
  
    ForEach ($PNToStop in $TestPN) {
      Try {
        $PNToStop | Stop-Process  -Force -EA Stop
      }
      Catch {"$(Get-DTStr) Couldn't stop process" +
             " $PNToStop." >> $LogFile}
  
    } #End ForEach ($PN...
  
  } #If ($Null...

} #End ForEach ($PN...

#*** Call Backup function w/BDF files to execute Backup ***

ForEach ($BDF in $BDFiles) {

  If (Test-Path -Path "$XMLFilePath\$BDF") {

    $ExitCode = Backup -BDFName "$BDF"
      
    $NELArgs  = @{LogName = "Application" 
                  Source  = "Macrium Reflect"}
    
    If (! [System.Diagnostics.EventLog]::SourceExists(
           'Macrium Reflect')) { New-EventLog @NELArgs }
    
    $WLArgs = @{EntryType = 'Information'
                EventId   = 0 
                LogName   = 'Application' 
                Message   = "Macrium Reflect - Successful Backup" 
                Source    = "Macrium Reflect"}
     
    If ($ExitCode -ne 0) {# Backup NOT completed successfully...
      $WLArgs.EventId = 1
      $WLArgs.Message = "Macrium Reflect - Backup Failed"
    }
    
    Write-EventLog @WLArgs
  
  } #End If (Test-Path...

  Else {
  
    "$(Get-DTStr) $XMLFilePath\$BDF" +
    " File Not Found!" >> "$LogFile"
  }

} #End ForEach ($BDF...

$EndTime = Get-Date
"$(Get-DTStr) Backup Set ended. Total Elapsed Time: " +
 "{00:hh}:{00:mm}:{00:ss}" -f ($EndTime - $StartTime) >> $LogFile

@"
+------------------------------------------------------+
| Check the C:\ProgramData\Macrium\Reflect directory   |
| for .html files with the name indicated for details. |
+------------------------------------------------------+
"@  >> $LogFile

#--- Set RunOnce to display Log File on next boot, ---

  $NIPArgs = 
   @{Path         = "HKCU:\Software\Microsoft\Windows\" +
                    "CurrentVersion\RunOnce" 
     Name         = "Show Backup Log"
     Value        = "$LogFile"
     Force        = $True}
  Set-ItemProperty @NIPArgs | Out-Null


#------------------ Clean-Up -------------------------

#Restore original Power Plan
If ($PowerPlan -ne "" -And ($Null -ne $NewPowerPlan)){
  $SavedPowerPlan.Activate()
  "$(Get-DTStr) $($SavedPowerPlan.ElementName) " +
  "power plan reactivated." >> $LogFile
} 

#Turn Network back on as re-boot will not do it!
If ((-not($Network.IsPresent)) -and (-not ($IgnoreNetwork))) {
  $NICs| Enable-NetAdapter 
  "$(Get-DTStr) Network reactivated!" >> $LogFile
}

If (-not $NoShutdown.IsPresent) {
  Stop-Computer  #Turn off computer:
}