﻿<#
  .Synopsis
   Copy all Non-Microsoft Drivers to a designated location.

  .Description
   Copy all Non-Microsoft Drivers to a designated location.
   Program will check for the existance of the location and
   offer to create it if it does not exist.

   Files will be written to the User's Documents folder under
   Customizations\[Computer Name]\DriverBackup
   Ex: d:\[documents]\Customizations\DellXPS8920\DriverBackup

   Use the associated PowerShell Import-Drivers.ps1 to
   reload ALL of the saved drivers. You can also load individual
   drivers by opening the driver file and double-clicking on
   the .inf file.

  .Parameter DocsFolder
   The base destination folder (directory) will default to the
   user's documents directory as specified in the environment.
   If you wish to over-ride this value use the -DocsFolder
   parameter to specify the base drive:\path.

   In either event the sub-folders (directories):
     Customizations\[ComputerName]\DriverBackup
   will be appended.

  .Outputs
   Exported driver files to specified directory.
   Status message dialog boxes:
     Prompt to create directory.
     Prompt to clear directory.
     Prompt with results.

  .Notes
     Programmer   : RetiredGeek (askWoody) aka: ComputerMentor
     Created      : 17 Oct 2019
     Last Updated : 18 Oct 2019
     Current Vers : 2.1

  .Example
   PS> [d:\path\]Export-Drivers.ps1
   Will create the backups in the default location of:
     [Documents]\Customizations\[COMPUTERNME]\DriverBackup

  .Example
   PS> [d:\path\]Export-Drivers.ps1 -DocsFolder N:\MyBackups

   Will create the backups in:
     N:\MyBackups\Customizations\[COMPUTERNME]\DriverBackup

#>


# Two blank linea above required for Comment-Based Help

Param (
       [Parameter(Mandatory=$False)]
        [Switch] $ClearDir,
       [Parameter(Mandatory=$False)]
        [String] $DocsFolder =
          "$([environment]::getfolderpath("mydocuments"))"
)

Function Show-Msg {

  Param ( [Parameter(Mandatory=$True,
           HelpMessage="Message box content.")]
            [string]$Message ,
          [Parameter(Mandatory=$False,
           HelpMessage="Message box title.")]
            [string]$MsgTitle = "Program Terminated:",
          [Parameter(Mandatory=$False,
           HelpMessage="Message box buttons.")]
            [string]$Btns = "OK",
          [Parameter(Mandatory=$False,
           HelpMessage="Message box Icon.")]
            [string]$Icon="Information"
        )

  [Windows.Forms.MessageBox]::Show("$Message", "$MsgTitle",
  [Windows.Forms.MessageBoxButtons]::$Btns,
  [Windows.Forms.MessageBoxIcon]::$Icon)

}  #End Function Show-Msg

<#
  +-----------------------------------------------------+
  |                  Main Program                       |
  +-----------------------------------------------------+
#>

Clear-Host

# *** Setup ***

Try{ #--- Check that the proper assemblies are loaded
     #--- Required for PS Console.
     #--- Only Systems.Windows.FormsNeeded for PSISE!

  $ATArgs = @{AssemblyName = "PresentationCore",
                             "PresentationFramework",
                             "WindowsBase",
                             "System.Windows.Forms"
              ErrorAction = 'Stop'}
Add-Type @ATArgs
}
Catch {
   $SMArgs =
     @{Message  =
         "Failed to load Windows Presentation Framework" +
         " assemblies required for this program!"
       MsgTitle = "Program Terminated:"
       Icon     = "Stop"
       Btns     = "Ok"}
   $Ans = Show-Msg @SMargs
  Exit
}

#  *** Check for Administrator Access! ******
$identity =
   [Security.Principal.WindowsIdentity]::GetCurrent()
 $principal =
   New-Object Security.Principal.WindowsPrincipal $identity

If (-NOT ($principal.IsInRole(
    [Security.Principal.WindowsBuiltinRole]::Administrator))) {
  $SMArgs =
     @{Message =  "You are running PowerShell with User " +
                  "Privileges!`nThis program requires "   +
                  "Administrator Privileges to work.`n`n" +
                  "Please restart PowerShell as Administrator."
       MsgTitle = "Program Terminated:"
       Icon     = "Stop"
       Btns     = "Ok"}
   $Ans = Show-Msg @SMargs
  Exit
  }

$EXDVRArgs =
  @{Online      = $True
    Destination = "$DocsFolder"       + "\Customizations\" +
                  "$env:COMPUTERNAME" + "\DriverBackup"}

If ( (Test-Path -Path $EXDVRArgs.Destination) -eq $False) {

  $SMArgs =
     @{Message  = "$($EXDVRArgs.Destination)`nDoes not exist!" +
                  "`n`nWould you like to create it?"
       MsgTitle = "Invalid Drive\Path:"
       Icon     = "Warning"
       Btns     = "YesNo"}
   $Ans = Show-Msg @SMargs

  If ($Ans -eq "Yes") {
    $NIArgs = @{Path     = $($EXDVRArgs.Destination)
                ItemType = "Directory"}
    $Null = New-item @NIArgs
  }
  Else { Exit }

} #End If ( (Test-Path...

Else {

  $GCIArgs = @{Path = $($EXDVRArgs.Destination)}
  If ($(Get-ChildItem @GCIArgs).Count -gt 0) {

    If (-Not $ClearDir.IsPresent) {
      $SMArgs =
         @{Message  = "Do you wish to Clear the direcory " +
                      "(folder)`n`n$($EXDVRArgs.Destination)?"
           MsgTitle = "Clear Existing Driver Backups:"
           Icon     = "Information"
           Btns     = "YesNo" }
      $Ans = Show-Msg @SMargs
    }
    Else { $Ans = "Yes" }

    If ($Ans -eq "Yes") {
      $RIArgs = @{Path    = "$($EXDVRArgs.Destination)\*.*"
                  Force   = $True
                  Recurse = $True }
      Remove-Item @RIArgs
    }

  } #End If (... -gt 0)

} #End Else

Try {

   $Null = Export-WindowsDriver @EXDVRArgs -ErrorAction "Stop"

   $DvrCnt =
        $(Get-ChildItem -Path $($EXDVRArgs.Destination)).Count

   $SMArgs =
      @{Message  = "$DvrCnt Drivers Successfully backed " +
                   "up to:`n`n$($EXDVRArgs.Destination)"
        MsgTitle = "Program Completed:"
        Icon     = "Information"
        Btns     = "OK"}
    $Ans = Show-Msg @SMargs

}  #End Try

Catch {
   $SMArgs =
      @{Message  = "Unexpected Error!" +
                   "Try running this command:`n`n" +
                   "Export-WindowsDriver -Online " +
                   "-Destination $($EXDVRArgs.Destination)" +
                   "`n`nCopy the error messages and post on " +
                   "askWoody.com as PM to RetiredGeek"
        MsgTitle = "Program Terminated:"
        Icon     = "Stop"
        Btns     = "OK"}
    $Ans = Show-Msg @SMargs
} #End Catch