﻿<#
  .Synopsis
   Will attempt to load drivers from your backups created
   with the Export-Drivers.ps1 program.

  .Description
   Will attempt to load drivers from your backups created
   with the Export-Drivers.ps1 program. A dialog box will
   appear detailing how many driver restores were attempted
   and how many were successful, it will also give you the
   change to re-boot your computer immediately or wait until
   a later time. 
   
   Note: the loaded drivers will NOT take affect until the 
         machine is re-booted!
 

  .Parameter DrvsDir
   The root drive:\directory (folder) where the program can
   find your backed up driver files. To the specified root
   \Customizations\[MachineName]\DriverBackup will be appended.

  .Parameter LogFile
   Switch denoting if a Log File with the output of pnputil.exe
   is to be saved. File will be saved to the user's Documnets
   directory/folder.

  .Outputs
   Status dialog boxes.
   ImportDrivers.Log File in the user's Documents directory/folder
   with output from pnputil.exe. 

   Sample:
   1 : G:\BEKDocs\Customizations\DELLXPS8920\DriverBackup\ant_libusb.inf_amd64_54173307afc55815\ant_libusb.inf
   Microsoft PnP Utility

   Processing inf :            ant_libusb.inf
   Successfully installed the driver.
   Driver package added successfully.
   Published name :            oem48.inf


   Total attempted:              1
   Number successfully imported: 1

  .Notes
     Programmer   : RetiredGeek (askWoody) aka: ComputerMentor
     Created      : 20 Oct 2019
     Last Updated : 22 Oct 2019
     Current Vers : 01.00

  .Example
   PS> [d:\path\]Import-Drivers.ps1

   Will import your saved drivers from the standard folder used
   by the Export-Drivers.ps1 program.

   Ex: C:\Users\[Userid]\Documents\Customizations\[MachineName]\
       DriverBackup

  .Example
   PS> [d:\path\]Import-Drivers.ps1 -DrvsDir "d:\path"

   Will import your saved drivers from the specified root path
   appending Customizations\[MachineName]\DriverBackup

   Ex: d:\path\Customizations\[MachineName]\DriverBackup

  .Example
   PS> [d:\path\]Import-Drivers.ps1 -LogFile

   Same as two example #1 above but will also write a log file
   to the user's documents directory/folder.

#>


# Two blank linea above required for Comment-Based Help

Param (
       [Parameter(Mandatory=$False)]
        [String] $DrvsDir =
          "$([environment]::getfolderpath("mydocuments"))",
       [Parameter(Mandatory=$False)]
        [Switch] $LogFile
)

Function Show-Msg {

  Param ( [Parameter(Mandatory=$True,
           HelpMessage="Message box content.")]
            [string]$Message ,
          [Parameter(Mandatory=$False,
           HelpMessage="Message box title.")]
            [string]$MsgTitle = "Program Terminated:",
          [Parameter(Mandatory=$False,
           HelpMessage="Message box buttons.")]
            [string]$Btns = "OK",
          [Parameter(Mandatory=$False,
           HelpMessage="Message box Icon.")]
            [string]$Icon="Information"
        )

  [Windows.Forms.MessageBox]::Show("$Message", "$MsgTitle",
  [Windows.Forms.MessageBoxButtons]::$Btns,
  [Windows.Forms.MessageBoxIcon]::$Icon)

}  #End Function Show-Msg

<#
  +-----------------------------------------------------+
  |                  Main Program                       |
  +-----------------------------------------------------+
#>

Clear-Host

# *** Setup ***

Try{ #--- Check that the proper assemblies are loaded
     #--- Required for PS Console.
     #--- Only Systems.Windows.FormsNeeded for PSISE!

  $ATArgs = @{AssemblyName = "PresentationCore",
                             "PresentationFramework",
                             "WindowsBase",
                             "System.Windows.Forms"
              ErrorAction = 'Stop'}
Add-Type @ATArgs
}
Catch {
   $SMArgs =
     @{Message  =
         "Failed to load Windows Presentation Framework" +
         " assemblies required for this program!"
       MsgTitle = "Program Terminated:"
       Icon     = "Stop"
       Btns     = "Ok"}
   $Ans = Show-Msg @SMargs
  Exit
}

#  *** Check for Administrator Access! ******
$identity =
   [Security.Principal.WindowsIdentity]::GetCurrent()
 $principal =
   New-Object Security.Principal.WindowsPrincipal $identity

If (-NOT ($principal.IsInRole(
    [Security.Principal.WindowsBuiltinRole]::Administrator))) {
  $SMArgs =
     @{Message =  "You are running PowerShell with User " +
                  "Privileges!`nThis program requires "   +
                  "Administrator Privileges to work.`n`n" +
                  "Please restart PowerShell as Administrator."
       MsgTitle = "Program Terminated:"
       Icon     = "Stop"
       Btns     = "Ok"}
   $Ans = Show-Msg @SMargs
  Exit
  }

$LogFileName = "$DrvsDir" + "\ImportDrivers.Log"
$DrvsDir = "$DrvsDir" + "\Customizations\" +
           "$env:COMPUTERNAME" + "\DriverBackup"

If ( (Test-Path -Path "$DrvsDir") -eq $False) {

  $SMArgs =
     @{Message  = "$DrvsDir`nDoes not exist!"
       MsgTitle = "Invalid Drive\Path:"
       Icon     = "Stop"
       Btns     = "OK"}
  $Null = Show-Msg @SMargs

  Exit 

} #End If ( (Test-Path...

Else {

  $GCIArgs = @{Path = "$DrvsDir"}
  If ($(Get-ChildItem @GCIArgs).Count -eq 0) {

    $SMArgs =
       @{Message  = "The direcory (folder):`n`n" +
                    "$DrvsDir`n`n?" +
                    "is Empty!"
         MsgTitle = "No Drivers to Import:"
         Icon     = "Stop"
         Btns     = "OK" }
    $Null = Show-Msg @SMargs

    Exit
  } #End If (... -gt 0)

} #End Else

If ( (Test-Path -Path "$LogFileName") -eq $True) {
  Remove-Item -Path "$LogFileName"
}

$GCIArgs = @{Path    = "$DrvsDir" 
             Filter  = "*.inf" 
             Recurse = $True
             File    = $True}

$Drivers = Get-ChildItem @GCIArgs

#Initialize Counters
$attempted  = 0
$successful = 0

ForEach ($DriverInf in $Drivers){

  $x =  pnputil.exe -i -a "$($DriverInf.FullName)"
  $RetLine = $x.Count - 3
  $attempted  += $($x[$RetLine].split(':'))[1].TrimStart(' ')
  $successful += 
            $($x[$($RetLine+1)].split(':'))[1].TrimStart(' ')

  If ($LogFile.IsPresent) {            
    "$attempted : $($DriverInf.FullName)" >> "$LogFileName"
    $x                                    >> "$LogFileName"
  } #End If ($LogFile....
                           
} #End ForEach ($DriverInf...

If ($LogFile.IsPresent) {
  $LogStatus = "`n`nLog written to:`n $LogFileName"
}
Else { $LogStatus = "" }

$SMArgs =
   @{Message  = "You attempted to load $attempted drivers." +
                "`n`n$successful drivers were "             +
                "successfully loaded.`n`nSelect Yes to "    +
                "Re-boot now or No to Re-boot later."       +
                "$LogStatus"
     MsgTitle = "Import Driver Results:"
     Icon     = "Information"
     Btns     = "YesNo" }
$Ans = Show-Msg @SMargs

If ($Ans -eq "Yes") { Restart-Computer -Force }
