﻿<#
.SYNOPSIS
   Produce information about the local computer in a tabbed
   dialog. Similar to Piriform's Speccy or BelArc Advisor.

.DESCRIPTION
   CMsLocalPCInfo.ps1 provides information on your hardware;
   windows; video; storage; network; printers; drivers;
   programs; windows update, selected registry/service settings
   and battery information for portables.

.Parameter Setup
   Creates a Scheduled Task to run program as Administrator.
   It also creates a shortcut to the Scheduled Task on the Desktop.

.OUTPUTS
    Optional output to File: Written to users default Documents
    Directory. With button to write output to a text file.

.NOTES
  Computer Mentors Local PC System Information W10 Only
  Version          : 5.11
  Programmed by    : RetiredGeek @ Windows Secrets Lounge (WSL)
  Code adapted from:
           Powershell PC Info Script V1.0b
              Coded By:Trenton Ivey(kno)
                     hackyeah.com
  Incorporated Coding changes recommed by Cafed00d @ WSL

  Thanks to Stephen Owen, MS MVP, for his gracious gift of
  his time and expertise in reviewing the code and pointing
  out several areas needing work. Also for turning me on to
  XAML! @FoxDeploy.com

  Thanks to Rick Corbet (@askWoody.com) for testing and many
  very useful suggestions.

  Updated:
  10/13/18 Changed to Windows 10 Only. Changed name to
           CmsLocalPCInfoW10 {minus 200+ lines of code}
                                                        (V1)
  10/22/18 Added Windows Defender Tab.                  (V2)
  11/05/18 Added Windows Defender Sandbox Status        (V3)
  12/30/18 Fix error w/Home ver. in WinUpdate           (V3.1)
  10/03/19 Add Windows Activation status                (V3.2)
  10/10/19 Add TPM & Disk Encryption Status             (V3.3)
  10/11/19 Fixed problem w/Multiple Batteries           (V3.4)
  12/23/19 Replaced special folders code                (V3.5)
  02/17/20 Fixed problem w/WinRM not running            (v3.6)
  04/22/20 Nested tab interface to reduce               (v5.11)
           scrolling and make information easier
           to locate. Windows Defender tab hidden
           if Defender is not the Active A/V program
           Output imporvements suggested by
             Rick Corbett @askWoody.com
  04/23/20 Fixed some problems when running in User     (v5.12)
           mode, added sub menus to the Registry/Services tab.

  Tested on: PS Vers 5.1.18362.628 & 7.0.0
  +-----------------------------------------------------------+
  |Note: there are spacing issues between PS5 & PS7. Thus,    |
  |      if you compare outputs you will see more blank lines |
  |      in the PS5 version vs the PS7 version!               |
  +-----------------------------------------------------------+

.EXAMPLE
    PS C:\>CMsPCInfo.ps1

    Returns data for the local computer.

.EXAMPLE
    PS C:\>CMSPCInfoW10.ps1 -Setup

    Creates a scheduled task to run the program as administrator
    and a desktop shortcut to run the scheduled task.

#>


#NOTE: TWO blank lines above required for comment based help!

Param (
       [Parameter(Mandatory=$False)]
        [Switch] $Setup
)

#--------------------  Functions   -----------------------------

Function Add-Row {

  param (
         [Parameter(Mandatory=$False)]
         [alias("ARIN")]
         [string]$ItemName,
         [Parameter(Mandatory=$False)]
         [alias("ARIV")]
         [string]$ItemValue
        )

  $CITRow       = $CITable.NewRow() #Create Row Variable
  $CITRow.Item  = $ItemName         #Assign items to row Var.
  $CITRow.Value = $ItemValue
  $CITable.Rows.Add($CITRow)    #Add Table Row using Row Var.

}  #------------------ End Function Add-Row -------------------

Function AudioTab {

  $SICodes = @{1 = "Other"
               2 = "Unknown"
               3 = "Enabled"
               4 = "Disabled"}

  $fmtAudio = @{Expression={$_.Name};
                 Label="`nDevice Name"; Width=45},
               @{Expression={$_.Manufacturer};
                 Label="`nManufacturer";Width=15},
               @{Expression={$_.Status};
                 Label="`nStatus";      Width=6},
               @{Expression={$SICodes[[Int]$_.StatusInfo]};
                 Label="Status`nInfo";  Width=8; Align='Left'}

  $Audio = Get-CimInstance -ClassName Win32_SoundDevice |
           Format-Table $fmtAudio |
           Out-String -Width $OStrWidth

  Return ,$Audio

} #----------------------- End Audio Tab ----------------------

Function BatteryTab {

  $Battery = $True

  Try {
       $GCIArgs = @{Class        = 'BatteryStatus'
                    Namespace    = 'root\wmi'
                    ErrorAction  = 'Stop'}
       $colBatStatus = Get-CimInstance @GCIArgs

  }
  Catch [Exception]
  {
    $Battery = $False
    Return ,$Null
  }

If ($Battery) {

  $Availability_ReturnValue = @{
    1 = 'Other'
    2 =	'Unknown'
    3 =	'Running or Full Power'
    4 =	'Warning'
    5 =	'In Test'
    6 =	'Not Applicable'
    7 =	'Power Off'
    8 =	'Off Line'
    9 =	'Off Duty'
   10 = 'Degraded'
   11 = 'Not Installed'
   12 = 'Install Error'
   13 = 'Power Save - Unknown'
   14 = 'Power Save - Low Power Mode'
   15 = 'Power Save - Standby'
   16 = 'Power Cycle'
   17 = 'Power Save - Warning'
  }

  $BatteryStatus_ReturnValue = @{
    1 = 'Other'
    2 = 'Unknown'
    3 = 'Fully Charged'
    4 = 'Low'
    5 = 'Critical'
    6 = 'Charging'
    7 = 'Charging and High'
    8 = 'Charging and Low'
    9 = 'Charging and Critical'
   10 = 'Undefined'
   11 = 'Partially Charged'
  }

  $Chemestry_ReturnValue = @{
    1 = 'Other'
    2 = 'Unknown'
    3 = 'Lead Acid'
    4 = 'Nickel Cadmium'
    5 = 'Nickel Metal Hydride'
    6 = 'Lithium-ion'
    7 = 'Zinc Air'
    8 = 'Lithium Polymer'
  }

  $StatusInfo_ReturnValue = @{
    1 = 'Discharging'
    2 = 'On A/C'
    3 = 'Fully Charged'
    4 = 'Low'
    5 = 'Critical'
    6 = 'Charging'
    7 = 'Charging High'
    8 = 'Charging Low'
    9 = 'Charging Critical'
    10 = 'Undefined'
    11 = 'Partially Charged'
  }

  $GCIArgs = @{Class      = 'Win32_Battery'
               Namespace  = 'root\CIMV2'}
  $colItems = Get-CimInstance @GCIArgs

  If (-not ($Null -eq $colItems)) {

    ForEach ($CI in $colItems) {

      If ($CI.Caption -eq "Internal Battery") {

      $BatteryTabStartCnt = $CITable.Rows.Count

      If ($colBatStatus.PowerOnline) {
        Add-Row -ARIN 'On Mains' -ARIV $colbatstatus.poweronline
        If ($Null -ne $CI.TimeToFullCharge) {
          $ARArgs= @{ARIN = 'Time To Full Charge'
                     ARIV = $CI.TimeToFullCharge}
          Add-Row @ARARGS
        }
        Add-Row -ARIN 'Charging' -ARIV $colBatStatus.Charging
      }

      Else {
        $ARArgs= @{ARIN = 'Availability'
                   ARIV =
               $Availability_ReturnValue[[int]$CI.Availability]}
        Add-Row @ARARGS

        If ($Null -ne $CI.BatteryRechargeTime) {
        $ARArgs= @{ARIN = 'Battery Recharge Time'
                   ARIV = $CI.BatteryRechargeTime}
        Add-Row @ARARGS
        }

        If ($Null -ne $CI.ConfigManagerErrorCode) {
          $ARArgs= @{ARIN = 'Configuration Manager Error Code'
                     ARIV = $CI.ConfigManagerErrorCode}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.ConfigManagerUserConfig) {
          $ARArgs= @{ARIN =
                   'Configuration Manager User Configuration'
                     ARIV = $CI.ConfigManagerUserConfig}
          Add-Row @ARArgs
        }

        $ARArgs= @{ARIN = 'Description'
                  ARIV = $CI.Description}
        Add-Row @ARArgs

        If ($Null -ne $CI.DesignCapacity) {
          $ARArgs= @{ARIN = 'Design Capacity'
                     ARIV = $CI.DesignCapacity}
          Add-Row @ARArgs
        }

        $ARArgs= @{ARIN = 'Design Voltage'
                  ARIV = $($CI.DesignVoltage/1000 )}
        Add-Row @ARArgs

        $ARArgs= @{ARIN = 'Device ID'
                   ARIV = $CI.DeviceID}
        Add-Row @ARArgs

        If ($Null -ne $CI.ErrorCleared) {
          $ARArgs= @{ARIN = 'Error Cleared'
                     ARIV = $CI.ErrorCleared}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.ErrorDescription) {
          $ARArgs= @{ARIN = 'Error Description'
                     ARIV = $CI.ErrorDescription}
          Add-Row @ARArgs
        }

        $ARArgs= @{ARIN = 'Estimated Charge Remaining'
                   ARIV =
                      "$($CI.EstimatedChargeRemaining)%"}
        Add-Row @ARArgs

        $TS = New-TimeSpan -Minutes $($CI.EstimatedRunTime)
        Add-Row -ARIN 'Estimated Run Time'  -ARIV $ts

        If ($Null -ne $CI.ExpectedBatteryLife) {
          $ARArgs= @{ARIN = 'Expected Battery Life'
                     ARIV = $CI.ExpectedBatteryLife}
          Add-Row @ARArgs
          $ARArgs= @{ARIN = 'Expected Life'
                     ARIV = $CI.ExpectedLife}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.FullChargeCapacity) {
          $ARArgs= @{ARIN = 'Full Charge Capacity'
                     ARIV = $CI.FullChargeCapacity}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.InstallDate) {
          $ARArgs= @{ARIN = 'Installation Date'
                     ARIV = $CI.InstallDate}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.LastErrorCode) {
          $ARArgs= @{ARIN = 'Last Error Code'
                     ARIV = $CI.LastErrorCode}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.MaxRechargeTime) {
          $ARArgs= @{ARIN = 'Maximum Recharge Time'
                     ARIV = $CI.MaxRechargeTime}
          Add-Row @ARArgs
        }

        Add-Row -ARIN 'Name' -ARIV $CI.Name

        If ($Null -ne $CI.PNPDeviceID) {
          $ARArgs= @{ARIN = 'PNP Device ID'
                     ARIV = $CI.PNPDeviceID}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.SmartBatteryVersion) {
          $ARArgs= @{ARIN = 'Smart Battery Version'
                     ARIV = $CI.SmartBatteryVersion}
          Add-Row @ARArgs
        }

        Add-Row -ARIN 'Status' $CI.Status
        If ($Null -ne $CI.StatusInfo) {
          $ARArgs= @{ARIN = 'Status Information'
                     ARIV =
         $StatusInfo_ReturnValue[[int]$CI.StatusInfo]}
          Add-Row @ARArgs
        }

        If ($Null -ne $CI.TimeOnBattery) {
          $TS = New-TimeSpan -Seconds $CI.TimeOnBattery
          Add-Row -ARIN 'Time On Battery' -ARIV $TS
        }
      } #End Else..If ($colBatStatus.PowerOnline)

      $BatteryChg = Test-BatteryHealth
      If ($BatteryChg -le 0) { $BatteryChg = "Undetermined"}

      $ARArgs= @{ARIN = 'Battery Last Full Charge'
                 ARIV = "$BatteryChg% of Original Capacity"}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Current Battery Charge mWh'
                 ARIV = $colBatStatus.RemainingCapacity}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Battery Status'
                 ARIV =
        $BatteryStatus_ReturnValue[[int]$CI.BatteryStatus]}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Caption'
                 ARIV = $CI.Caption}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Chemistry'
                 ARIV =
               $Chemestry_ReturnValue[[int]$CI.Chemistry]}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Power Management Capabilities'
                 ARIV =
                 ([String]$CI.PowerManagementCapabilities)}
      Add-Row @ARArgs

      $ARArgs= @{ARIN = 'Power Management Supported'
                 ARIV =
                   (&{If($CI.PowerManagementSupported)
                      {"True"} Else {"False"}})}
      Add-Row @ARArgs

      } #End If($CI.Caption...

    } #End ForEach-Object ($CI...

    $GMLArgs = @{TestObj =
       $CITable.Item[$BatteryTabStartCnt..$(
                     $CITable.Rows.Count -1)]}
    $ItemLen = Get-MaxLength @GMLArgs
    $ItemLen += 2

    $fmtBattery =
      @{Label="Item  ";Width=$ItemLen    ;Expression={$_.Item}},
      @{Label="Value";Width=(90-$ItemLen);Expression={$_.Value}}

    $BatteryInfo =
      $CITable.Rows[($BatteryTabStartCnt)..(
                    $CITable.Rows.Count -1)] |
      Format-Table -Property $fmtBattery     |
      Out-String   -Width $OStrWidth

    Return ,$BatteryInfo

  } #End If (-not($colItems -eq $Null))

} #End If ($Battery)

} #----------------------- End Battery Tab  -------------------

Function CDDVDTab {

  $CDTitle = "CD/DVD Information:" | Out-String

  $CD      = Get-CimInstance -ClassName 'Win32_CDROMDrive'

  If ($Null -ne $CD) {

    $fmtCD  = @{Expression={$_.Drive};
                  Label="Drive Letter";Width=6},
              @{Expression={$_.Name};Label="`nName";Width=25},
              @{Expression={$_.CapabilityDescriptions};
                  Label="`nCapabilities";Width=40}
    $CDDVDINFO = $CD | Format-Table -Wrap -Property $fmtCD |
                       Out-String   -Width $OStrWidth
  }
  Else{
    #---- All this to get a lousy Line Feed in the output! ---
    $CD        = New-Object PSObject
    $AMArgs    = @{Type  = "NoteProperty"
                   Name  = "Status"
                   Value = 'None-Available'}
    $CD | Add-Member @AMArgs
    $fmtCD     = @{Expression={$_.Status}}
    $CDDVDInfo = $CD | Format-Table $fmtCD -HideTableHeaders |
                       Out-String -Width $OStrWidth
  }

  Return $($CDTitle + $CDDVDInfo)

} #-------------- End CDDVDTab ---------------------------------

Function DriversTab {

$DriverInfo =
  Get-CimInstance -ClassName 'Win32_PNPSignedDriver'         |
  Where-Object -Property DriverProviderName  -ne "Microsoft" |
  Where-Object -Property DeviceName -ne -Value $Null         |
  Sort-Object  -Property DeviceName -Unique

$DriverCnt = $DriverInfo.Count

  $DVLen =
    Get-MaxLength -TestObj $DriverInfo.DriverVersion -MinLen 14
  $DDLen = $OstrWidth - $DVLen

  $fmtDRVR = @{Label="`nDriver Description";Width=$DDLen;
                                 Expression={$_.DeviceName}},
             @{Label="Version Number";    Width=$DVLen;
                                 Expression={$_.DriverVersion}}

  $DrvTitle = "$($DriverCnt) Non-Windows Unique Drivers and " +
              "Version Numbers:" | Out-String

  $DriverInfo =
    $DriverInfo | Format-Table -Property $fmtDRVR -Wrap |
                  Out-String   -Width $OStrWidth

  Return ,$($DrvTitle + $DriverInfo)

} #---------------------- End Drivers Tab  --------------------

Function EnvironmentTab {

  $WinEnvVars = Get-ChildItem -Path 'Env:'
  $EnvLen = Get-MaxLength -TestObj $WinEnvVars.Name -MinLen 10

  $fmtENV =
    @{Label="Env Variable";Width=$EnvLen;Expression={$_.Name}},
    @{Label="Value"; Width=(89-$EnvLen);Expression={$_.Value}}

  $EnvInfo =
       $WinEnvVars | Format-Table $fmtENV -Wrap |
                     Out-String -Width $OStrWidth

  Return ,$EnvInfo

} #-------------------- End EnvironmentTab --------------------

Function FileSystemTab {

  $FSTabStartCount = $CITable.Rows.Count

  Try {
    $RegArgs =
      @{Path = 'HKLM:\system\CurrentControlSet\Control\FileSystem'
        Name = 'DisableDeleteNotification'
        ErrorAction = "Stop"}
    $TrimVal = $($(Get-ItemProperty @RegArgs).DisableDeleteNotification)

  #--- PS IIF construct! ---
    $TrimStatus = (&{If([int]$TrimVal -eq 0)
                       {"Enabled"}
                  Else {"Disabled....Fix THIS!"}})
  }
  Catch {
    $TrimStatus = "Error: Unable to retrieve registry entry!"
  }

  Add-Row -ARIN  'Trim Status' -ARIV $TrimStatus

  Try {
    $RegArgs.Name = 'NtfsDisable8dot3NameCreation'
    $EDTStatus    =
     $($(Get-ItemProperty @RegArgs).NtfsDisable8dot3NameCreation)
    $EightDotThreeNaming = (&{If([int]$EDTStatus -eq 0)
                                {'Enabled'} Else {'Disabled'}})
  }
  Catch {
    $EightDotThreeNaming =
       "Error: Unable to retrieve registry entry!"
  }

  Add-Row -ARIN  '8.3 Naming' -ARIV $EightDotThreeNaming

  Try {
    $RegArgs.Name = 'LongPathsEnabled'
    $LPStatus    =
      $($(Get-ItemProperty @RegArgs).LongPathsEnabled)
    $LongPaths = (&{If([int]$LPStatus -eq 0)
                       {'Disabled'} Else {'Enabled'}})
  }
  Catch {
    $LongPaths = "Error: Unable to retrieve registry entry!"
  }

  Add-Row -ARIN  'Long Paths' -ARIV $LongPaths

  $GMLArgs = @{TestObj = $CITable.Item[($FSTabStartCount)..(
                                        $CITable.Item.Count-1)]}
  $FSWidth = (Get-MaxLength @GMLArgs) + 3

  $fmtFileSys =
      @{Label='Item'   ;Width=$FSWidth;Expression={$_.Item}},
      @{Label='Setting';Width=(89-$FSWidth);
         Expression={$_.Value}}

  $FSItems = $CITable.Rows[$FSTabStartCount..(
             $CITable.Rows.Count-1)] |
             Format-Table -Property $fmtFileSys |
             Out-String   -Width $OStrWidth

  $FSITitle = "Selected File System Items:" | Out-String

  Return, $($FSITitle + $FSItems)

} #-------------- End FileSystem Tab ---------------------

Function GenerateForm {

#------------------------- Main Form --------------------------

#--- Form Drawing References ---

  $Form_Width       =  950
  $Form_Height      =  550
  $Btn_Height       =   30
  $tabWidth         = $Form_Width  - 10
  $tabHeight        = $Form_Height - 80
  $tboxWidth        = $tabWidth    - 10
  $tboxHeight       = $tabHeight   - 60
  $tabL2Width       = $tabWidth    - 10
  $tabL2Height      = $tabHeight   - 35
  $tboxL2Width      = $tboxWidth   - 15
  $tboxL2Height     = $tboxHeight  - 10
  $WindowTitle =
   "Computer Mentor's Local Computer Information " +
   "W10 Only Version {0:00.00}" -f $PgmVersion
  $WindowTitle +=
   " Running with {0} privileges." -f $(Get-AdminStatus)
  $WindowTitle += "  Run Time: {0:mm}:{0:ss}" -f $RunTime

[xml]$xaml = @"
<Window
    xmlns="http://schemas.microsoft.com/winfx/2006/xaml/presentation"
    xmlns:x="http://schemas.microsoft.com/winfx/2006/xaml"
    x:Name="Window" Title="$WindowTitle"
                    WindowStartupLocation="CenterScreen"
                    Width = "$Form_Width" Height = "$Form_Height"
                    FontSize="18" ShowInTaskbar = "True">

    <Window.Resources>

      <Style x:Key="TBoxStyle" TargetType="TextBox">
       <Setter Property="Height"     Value="$tBoxHeight"    />
       <Setter Property="Width"      Value="$tBoxWidth"     />
       <Setter Property="Margin"     Value="5"              />
       <Setter Property="Background" Value="Blue"           />
       <Setter Property="Foreground" Value="White"          />
       <Setter Property="FontFamily" Value="Courier New"    />
       <Setter Property="IsReadOnly" Value="$True"          />
       <Setter Property="VerticalScrollBarVisibility"
                                                Value="Auto"/>
      </Style>

    </Window.Resources>

    <Canvas>
        <TabControl Name="PCInfoTab"
                    Canvas.Top="0" Canvas.Left="0"
                    Width="$tabWidth" Height="$tabHeight"
                    Background="Blue" Foreground="White">

          <TabItem Header = "Hardware">
             <Canvas>
              <TabControl Name   = "HardwareTabs"
                          Width  = "$tabL2Width"
                          Height = "$tabL2Height"
                          Background = "Green">
                <TabItem Header="General Hardware">
                  <TextBox x:Name="tboxHardware"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Processor (CPU)">
                  <TextBox x:Name="tboxProcessor"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Memory">
                  <TextBox x:Name="tboxMemory"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>

            <TabItem Header="Video">
             <Canvas>
              <TabControl Name   = "VideoTabs"
                          Width  = "$tabL2Width"
                          Height = "$tabL2Height"
                          Background = "Green">
                <TabItem Header="Video Card">
                  <TextBox x:Name="tboxVideo"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Monitor(s)">
                  <TextBox x:Name="tboxMonitor"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>

            <TabItem Header="Audio">
                <TextBox x:Name="tboxAudio"
                    Style="{StaticResource TBoxStyle}" />
            </TabItem>

            <TabItem Header = "Storage">
             <Canvas>
              <TabControl Name   = "PhysicalDiskTabs"
                          Width  = "$tabL2Width"
                          Height = "$tabL2Height"
                          Background = "Green">
                <TabItem Header="Physical Disks">
                  <TextBox x:Name="tboxPhysicalDisk"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Partition Info">
                  <TextBox x:Name="tboxPartition"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Logical Disks">
                  <TextBox x:Name="tboxLogicalDisk"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="CD-DVD">
                  <TextBox x:Name="tboxCDDVD"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="File System">
                  <TextBox x:Name="tboxFileSystem"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Maps and Shares">
                  <TextBox x:Name="tboxMapsAndShares"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Encryption Info">
                  <TextBox x:Name="tboxEncryption"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>

            <TabItem Header = "Network">
             <Canvas>
              <TabControl Name   = "NetworkTabs"
                           Width  = "$tabL2Width"
                           Height = "$tabL2Height"
                           Background = "Green">
                <TabItem Header="Internet">
                  <TextBox x:Name="tboxInternet"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Active Net Adapter">
                  <TextBox x:Name="tboxANAdapter"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="WiFi Info">
                  <TextBox x:Name="tboxWiFi"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Local Network">
                  <TextBox x:Name="tboxLocalNetwork"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Network Adapters">
                  <TextBox x:Name="tboxNetAdapters"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>

            <TabItem Header="Printers">
                <TextBox x:Name="tboxPrinters"
                    Style="{StaticResource TBoxStyle}" />
            </TabItem>

            <TabItem Header = "Windows 10">
             <Canvas>
              <TabControl Name   = "WinTabs"
                          Width  = "$tabL2Width"
                          Height = "$tabL2Height"
                          Background = "Green">
                <TabItem Header="General">
                  <TextBox x:Name="tboxWindows"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Users">
                  <TextBox x:Name="tboxUsers"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Security">
                  <TextBox x:Name="tboxSecurity"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="PowerShell (.Net)">
                  <TextBox x:Name="tboxPowerShell"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Windows Update">
                  <TextBox x:Name="tboxWindowsUpdate"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Recent Updates">
                  <TextBox x:Name="tboxUpdateHistory"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Upgrade History">
                  <TextBox x:Name="tboxUpgradeHistory"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Environment">
                  <TextBox x:Name="tboxEnvironment"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header    ="Windows Defender"
                         x:Name    ="tabWindowsDefender"
                         Visibility="Hidden" >
                  <TextBox x:Name="tboxWindowsDefender"
                     Style  = "{StaticResource TBoxStyle}"
                     Width  = "$tboxL2Width"
                     Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>

            <TabItem Header="Drivers">
                <TextBox x:Name="tboxDrivers"
                    Style="{StaticResource TBoxStyle}" />
            </TabItem>
            <TabItem Header="Programs">
                <TextBox x:Name="tboxPrograms"
                    Style="{StaticResource TBoxStyle}" />
            </TabItem>
            <TabItem Header="Registry/Services">
             <Canvas>
              <TabControl Name   = "RegSvcs"
                          Width  = "$tabL2Width"
                          Height = "$tabL2Height"
                          Background = "Green">
                <TabItem Header="General Reg/Svc">
                  <TextBox x:Name="tboxGenRegSvc"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
                <TabItem Header="Security Reg/Svc">
                  <TextBox x:Name="tboxSecRegSvc"
                      Style  ="{StaticResource TBoxStyle}"
                      Width  = "$tboxL2Width"
                      Height = "$tboxL2Height" />
                </TabItem>
              </TabControl>
             </Canvas>
            </TabItem>
            <TabItem Header="Battery"
                     x:Name="tabBattery">
                <TextBox x:Name="tboxBattery"
                    Style="{StaticResource TBoxStyle}" />
            </TabItem>
        </TabControl>

        <Button x:Name="WriteFileBtn"  Content='Write File'
                Height="$Btn_Height" Width="100"
                Canvas.Bottom="5" Canvas.Left="5"
                Background="Blue" Foreground="Yellow"
                FontSize="20" ToolTip = "Write to File"
                Margin="5" />
        <Button x:Name="ExitBtn" Content='Exit'
                Height="$Btn_Height" Width="80"
                Canvas.Bottom="5" Canvas.Right="5"
                Background="Green" Foreground="Yellow"
                FontSize="20" ToolTip = "Exit"
                Margin="5" />
        <Label x:Name="Status" Content="" FontSize="18"
                Width="800" Height="$Btn_Height"
                Canvas.Bottom="5" Canvas.Left="5"
                Background="Blue" Foreground="Yellow"
                Visibility="Hidden"
                Margin="5" />
    </Canvas>
</Window>
"@

$reader=(New-Object System.Xml.XmlNodeReader $xaml)
$Window=[Windows.Markup.XamlReader]::Load( $reader )
$Window.Topmost = $True
$Window.Cursor  = [System.Windows.Input.Cursors]::Hand

#Connect to Exit Button Control
$ExitBtn = $Window.FindName("ExitBtn")
$ExitBtn.Add_Click({$Window.Close()})

#Connect to WriteFile Button Control
$WriteFileBtn = $Window.FindName("WriteFileBtn")
$WriteFileBtn.Add_Click({Write-ComputerInfoFile
                         $WriteFileBtn.Visibility = 'Hidden'
     $Status.Visibility = 'Visible'
     $Status.Content = "Data written to: $($InfoFileName)"
})

#Connect to tab control text boxes

$tboxHardware = $Window.FindName("tboxHardware")
$tboxHardware.AddText($HardwareInfo)

$tboxProcessor = $Window.FindName("tboxProcessor")
$tboxProcessor.AddText($ProcessorInfo)

$tboxMemory = $Window.FindName("tboxMemory")
$tboxMemory.AddText($MemoryInfo)

$tboxVideo = $Window.FindName("tboxVideo")
$tboxVideo.AddText($VideoInfo)

$tboxMonitor = $Window.FindName("tboxMonitor")
$tboxMonitor.AddText($MonitorInfo)

$tboxAudio = $Window.FindName("tboxAudio")
$tboxAudio.AddText($AudioInfo)

$tboxPhysicalDisk = $Window.FindName("tboxPhysicalDisk")
$tboxPhysicalDisk.AddText($PhysicalDiskInfo)

$tboxPartition = $Window.FindName("tboxPartition")
$tboxPartition.AddText("$PartitionInfo")

$tboxLogicalDisk = $Window.FindName("tboxLogicalDisk")
$tboxLogicalDisk.AddText("$LogicalDiskInfo")

$tboxMapsAndShares = $Window.FindName("tboxMapsAndShares")
$tboxMapsAndShares.AddText($MapsAndShares)

$tboxCDDVD = $Window.FindName("tboxCDDVD")
$tboxCDDVD.AddText($CDDVDInfo)

$tboxFileSystem = $Window.FindName("tboxFileSystem")
$tboxFileSystem.AddText($FileSystemInfo)

$tboxEncryption = $Window.FindName("tboxEncryption")
$tboxEncryption.AddText($EncryptInfo)

$tboxInternet = $Window.FindName("tboxInternet")
$tboxInternet.AddText($InternetInfo)

$tboxANAdapter = $Window.FindName("tboxANAdapter")
$tboxANAdapter.AddText($ActiveAdapterInfo)

$tboxWiFi = $Window.FindName("tboxWiFi")
$tboxWiFi.AddText("$WiFiInfo")

$tboxLocalNetwork = $Window.FindName("tboxLocalNetwork")
$tboxLocalNetwork.AddText($LocalNetworkInfo)

$tboxNetAdapters = $Window.FindName("tboxNetAdapters")
$tboxNetAdapters.AddText($NetAdaptersInfo)

$tboxPrinters = $Window.FindName("tboxPrinters")
$tboxPrinters.AddText($PrinterInfo)

$tboxUsers = $Window.FindName("tboxUsers")
$tboxUsers.AddText($UserInfo)

$tboxWindows = $Window.Findname("tboxWindows")
$tboxWindows.AddText($WindowsInfo)

$tboxSecurity = $Window.FindName("tboxSecurity")
$tboxSecurity.AddText($SecurityInfo)

$tboxPowerShell = $Window.FindName("tboxPowerShell")
$tboxPowerShell.AddText($PowerShellInfo)

$tboxWindowsUpdate = $Window.Findname("tboxWindowsUpdate")
$tboxWindowsUpdate.AddText($WindowsUpd)

If ($Null -ne $WinDefInfo) {
  $tabWindowsDefender = $Window.FindName("tabWindowsDefender")
  $tabWindowsDefender.Visibility = 'Visible'
  $tboxWindowsDefender = $Window.FindName("tboxWindowsDefender")
  $tboxWindowsDefender.AddText($WinDefInfo)
}

$tboxUpdateHistory = $Window.FindName("tboxUpdateHistory")
$tboxUpdateHistory.AddText($(Get-W10UpdateHistory))

$tboxUpgradeHistory = $Window.FindName("tboxUpgradeHistory")
$tboxUpgradeHistory.AddText($(Get-W10UpgradeHistory))

$tboxEnvironment = $Window.Findname("tboxEnvironment")
$tboxEnvironment.AddText($EnvironInfo)

$tboxDrivers = $Window.Findname("tboxDrivers")
$tboxDrivers.AddText($DriversInfo)

$tboxPrograms = $Window.Findname("tboxPrograms")
$tboxPrograms.AddText($ProgramsInfo)

$tboxGenRegSvc = $Window.Findname("tboxGenRegSvc")
$tboxGenRegSvc.AddText($GenRegSvcInfo)

$tboxSecRegSvc = $Window.Findname("tboxSecRegSvc")
$tboxSecRegSvc.AddText($SecRegSvcInfo)

If ($Null -ne $BatteryInfo) {
  $tboxBattery = $Window.FindName("tboxBattery")
  $tboxBattery.AddText($BatteryInfo)
}
Else {
  #*** No battery hide the tab! ***
  $tabBattery = $Window.FindName("tabBattery")
  $tabBattery.Visibility = 'Hidden'
}

$Status = $Window.FindName("Status")
$Status.Visibility = 'Hidden'

$Null = $Window.ShowDialog()

} #------------------ End Function GenerateForm ---------------

Function Get-AdminStatus {

  $identity =
     [Security.Principal.WindowsIdentity]::GetCurrent()
  $principal =
     New-Object Security.Principal.WindowsPrincipal $identity

  If (-NOT ($principal.IsInRole(
      [Security.Principal.WindowsBuiltinRole]::Administrator)))
    {"User"}
  Else
    {"Administrator"}

} # -----------  End Function Get-AdminStatus -----------------

Function Get-AntiVirual{

<#+---------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddaway |
  +---------------------------------------------------+
#>
  [CmdletBinding()]
  [OutputType('System.Object[]')]

  param (
    [parameter(ValueFromPipeline=$true,
      ValueFromPipelineByPropertyName=$true)]
     [string]$CompName="$env:COMPUTERNAME"
  )

  PROCESS{

    $os = Get-CimInstance -ClassName 'Win32_OperatingSystem'

   if ([int]$os.BuildNumber -ge 6001 ) {
     $GCIArgs = @{Namespace  = 'ROOT\SecurityCenter2'
                  Class      = 'AntiVirusProduct'
                 }
     $av = Get-CimInstance  @GCIArgs
   }
   else {
     $GCIArgs = @{Namespace  = 'ROOT\SecurityCenter'
                  Class      = 'AntiVirusProduct'}
     $av = Get-CimInstance  @GCIArgs
   }

   Return ,$av

  } #End Process

} #--------------- End Function Get-AntiVirual  ----------------

Function Get-AVStatusText {

  param (
    [parameter(Mandatory=$True)]
     [Int]$AVStatusCd
  )

switch ($AVStatusCd) {
    "262144" {$UpdateStatus             = "Up to date"
              $RealTimeProtectionStatus = "Disabled"}  #0x40000
    "262160" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Disabled"}  #0x40010
    "266240" {$UpdateStatus = "Up to date"
              $RealTimeProtectionStatus = "Enabled"}   #0x41000
    "266256" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Enabled"}   #0x41010
    "393216" {$UpdateStatus = "Up to date"
              $RealTimeProtectionStatus = "Disabled"}  #0x60000
    "393232" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Disabled"}  #0x60010
    "393472" {$UpdateStatus = "Up to date"
              $RealTimeProtectionStatus = "Disabled"}  #0x60100
    "393488" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Disabled"}  #0x60110
    "397312" {$UpdateStatus = "Up to date"
              $RealTimeProtectionStatus = "Enabled"}   #0x61000
    "397328" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Enabled"}   #0x61010
    "397568" {$UpdateStatus = "Up to date"
              $RealTimeProtectionStatus = "Enabled"}   #0x61100
    "397584" {$UpdateStatus = "Out of date"
              $RealTimeProtectionStatus = "Enabled"}   #0x61110
 default {$UpdateStatus = "Unknown"
          $RealTimeProtectionStatus = "Unknown"}
 }

 $RetVal = "Real Time Protection: $RealTimeProtectionStatus`n" +
           "Update Status       : $UpdateStatus"

 Return ,$RetVal

} #End Function Get-AVStateText

Function Get-BiosType {

<#
.Synopsis
   Determines underlying firmware (BIOS) type and returns an
   integer indicating UEFI, Legacy BIOS or Unknown.
   Supported on Windows 8/Server 2012 or later
.DESCRIPTION
   This function uses a compiled Win32 API call to determine
   the underlying system firmware type.
.EXAMPLE
   If (Get-BiosType -eq 1) { # System is running UEFI firmware... }
.EXAMPLE
    Switch (Get-BiosType) {
        1       {"Legacy BIOS"}
        2       {"UEFI"}
        Default {"Unknown"}
    }
.OUTPUTS
   Integer indicating firmware type:
   (1 = Legacy BIOS, 2 = UEFI, Other = Unknown)
.FUNCTIONALITY
   Determines underlying system firmware type
#>

 [OutputType([UInt32])]
 Param(
    [Parameter(Mandatory=$True)]
      [String] $CompName)

   $ATArgs = @{Language       = 'CSharp'
               ErrorAction    = 'SilentlyContinue'
               TypeDefinition = @'

    using System;
    using System.Runtime.InteropServices;

    public class FirmwareType
    {
     [DllImport("kernel32.dll")]
     static extern bool GetFirmwareType(ref uint FirmwareType);

     public static uint GetFirmwareType()
     {
       uint firmwaretype = 0;
       if (GetFirmwareType(ref firmwaretype))
         return firmwaretype;
       else
         return 0; // API call failed, just return 'unknown'
     }
    }
'@
    } #End $ATArgs

    $ErrorActionPreference = 'Stop'
    Try {
      If (-not
        [System.Management.Automation.PSTypeName][FirmwareType]) {
      }
    }
    Catch {  #Type does not Exist Create it!
      Add-Type @ATArgs
    }
    $ErrorActionPreference = 'SilentlyContinue'

   Return ,[FirmwareType]::GetFirmwareType()

}  #End  ---------------Function Get-BIOSTYPE ----------------

Function Get-Drive {

#	Courtesy of: http://powershell.com/cs/media/p/7924.aspx
#   Will not work with Get-CimInstance!

<#+-----------------------------------------------------------+
  |Note: The following Get-WMIObject can NOT be replaced by a |
  |      Get-CIMInstance because the resulting object does    |
  |      NOT have a .GetRelated method!                       |
  +-----------------------------------------------------------+
#>

  Try {

[Diagnostics.CodeAnalysis.SuppressMessageAttribute(
   "PSAvoidUsingWMICmdlet", "",Scope="Function",
                               Target="*")]
  $WMIArgs = @{Computer = $CompName
               Class    = 'Win32_DiskPartition'}
  Get-WMIObject @WMIArgs |
	ForEach-Object {
		$partition = $_
		$logicaldisk =
           $partition.psbase.GetRelated('Win32_LogicalDisk')
#Note: next statement does NOT work if $Null is coded first!
		if ($logicaldisk -ne $Null) {
			Merge-Object $logicaldisk $partition
		}
	} |  Select-Object -Property Name, VolumeName, FileSystem,
                                 DriveType, Size, Compressed,
                                 FreeSpace,  DiskIndex,
                                 Index, Bootable
<#+-----------------------------------------------------------+
  |Note: This returns the Select-Object output to the caller! |
  |      The Select-Object could be dropped but unneeded data |
  |      would be returned to caller...inefficient!           |
  +-----------------------------------------------------------+
#>
  } #---- End Try ---

  Catch { #For PS-7!

         $DiskTypeHash = @{
            2 = "Removable"
            3 = "Fixed"
            4 = "Network"
            5 = "Compact"}

     $fmtLogDsk =
       @{Expression={$_.DeviceID};Label="Drive`nLetter";
                   Align='Center';Width=6},
       @{Expression={$_.VolumeName};
           Label="Volume`nName";Width=15},
       @{Expression={$_.filesystem};
           Label="File`nSystem";Width=6},
       @{Expression={$DiskTypeHash.item([int]$_.DriveType)};
           Label="Drive`nType";Width=9},
       @{Expression={$_.compressed};
           Label="Compr`nessed";Width=5;Align='left'},
       @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
           Label="Disk Size`n / GB";Width=9;align='right'},
       @{Expression={ '{0:#,000.00}' -f ($_.FreeSpace/1gb)};
           Label="Free Space`n / GB";Width=10;align='right'}

          Get-CimInstance -Class Win32_LogicalDisk |
              Where-Object { $_.DriveType -eq 3 }  |
              Select-Object DeviceID, VolumeName,
                   Filesystem, DriveType, Compressed,
                   Size, FreeSpace |
          Format-Table -Property $fmtLogDsk
  }

} #--------------    End Function Get-Drive  -----------------

Function Get-EncryptStatus {

  Function Add-ETRow {

    param (
           [Parameter(Mandatory=$True,Position=0)]
           [alias("ARVol")]
           [string] $EITVolName,
           [Parameter(Mandatory=$True,Position=1)]
           [alias("ARDL")]
           [string] $EITDrvLtr,
           [Parameter(Mandatory=$True,Position=2)]
           [alias("AREM")]
           [string] $EITEncMethod
          )

    $EITRow           = $EITable.NewRow() #New Row Variable
    $EITRow.Name      = $EITVolName     #Assign to row Var.
    $EITRow.DrvLtr    = $EITDrvLtr
    $EITRow.EncMethod = $EITEncMethod
    $EITable.Rows.Add($EITRow) #Add Table Row using Row Var.

  }  #------------------ End Function Add-ETRow -------------

  # *** Check for UEFI required for BitLocker ***
  Try   {$Null = Confirm-SecureBootUEFI -EA Stop}
  Catch { $Message = "`nMachine does NOT support UEFI and " +
          "therefore`nCan NOT support BitLocker!`n`n" |
           Out-String
          Return ,$Message

        }

  # Create Encryption Information Table (EIT)
  $EITable =
    New-Object -TypeName System.Data.DataTable "Encryption Info"
  #Create 3 Columns for table
  $EITVolName  =
    New-Object -TypeName System.Data.DataColumn Name,([string])
  $EITDrvLtr =
    New-Object -TypeName System.Data.DataColumn DrvLtr,([string])
  $EITEncMethod =
    New-Object -TypeName System.Data.DataColumn EncMethod,([string])
  #Add Columns to table
  $EITable.columns.add($EITVolName)
  $EITable.columns.add($EITDrvLtr)
  $EITable.Columns.add($EITEncMethod)

  # Create Output format string
  $fmtEITable =
    @{Expression={$_.DrvLtr};
        Label="Drive`nLetter";Align='center';Width=6},
    @{Expression={$_.Name};
        Label="Volume`nName";Align='Left';Width=15},
    @{Expression={$_.EncMethod};
        Label="Encryption`nMethod";Align='Left'}

  $Vol     = 0
  $Method  = 0
  $Encrypt = manage-bde.exe -status
  $Cnt     = $Encrypt.Count

  # *** Loop through String Array ***

  For ($Cntr = 0; $Cntr -lt $Cnt; $Cntr++) {

     If ($($Encrypt[$($Cntr)]).Contains("Volume ")) {
       $Vol = $Cntr
     }
     If ($($Encrypt[$($Cntr)]).Contains("Method:")) {
       $Method = $Cntr
     }

   If ( ($Vol -gt 0) -and ($Method -gt 0)) {

     $VolParts = $($Encrypt[$($Vol)]).Split(" ")
     $MetParts = $($Encrypt[$($Method)]).Split(":")
     Add-ETRow $VolParts[2] $VolParts[1]  $MetParts[1].Trim()
     $Vol = 0
     $Method = 0
   }

  } #End For

  $EncryptTitle  = "Disk Encryption Information:" |Out-String

  # * Save Sorted Output to String for Dialog Box Display *
  $EncryptMsg = $EITable | Sort-Object -Property DrvLtr |
                           Format-Table $fmtEITable     |
                           Out-String

  Return , $($EncryptTitle + $EncryptMsg)

} #----------  End Function Get-EncryptStatus  ----------------

Function Get-HyperVStatus {

<#
  .Synopsis
   Returns the status of either the Virtuialization
   Hardware or Software

  .Description
   Returns the status of either the Virtuialization
   Hardware or Software

  .Parameter VirtualHW
   Optional switch, if present return the status of the
   virtuialization Hardware, if not present return the
   status of the virtuialization Software.

  .Outputs
   Returns either Enabled or Disabled

  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 11 Nov 2019
     Last Updated :
     Current Vers : 1.0

  Dell Inspiron 1564 Windows 10 Insider Ed 18908 +
    SystemInfo incorrectly reports Hardware Virtualization as
    being disabled when it is enabled!

  On Dell Inspiron 1564 Win 10 Home Insider Ed.
     Dell XPS 8700      Win 10 Pro  1909, 1903
     Dell XPS 8920      Win 10 Pro  1909, 1903, 1809
    PowerShell via:
      $CPU_Object =
        Get-CimInstance -ClassName  'Win32_Processor'
      $CPU_Object.VirtualizationFirmwareEnabled
    Reports incorrect value.

  .Example
   Get-HyperVStatus -VirtualHW

   Returns the status of the machines Virtuialization Hardware.

  .Example
   Get-HyperVStatus

   Returns the staus of the machines Virtuialization Software.

#>


# Two blank linea above required for Comment-Based Help

  Param (
    [Parameter(Mandatory=$False)]
     [Switch] $VirtualHW
  )

  $HyperV   = ""
  $HVStatus = ""

  $CurOS = Get-CimInstance -ClassName 'Win32_OperatingSystem'
  $WinHome = $CurOS.OperatingSystemSKU -eq 101

  $x = Systeminfo  #$x will contain an array of strings!

  For ($cntr = 0; $cntr -lt $x.count ; $cntr++) {

    If ($x[$cntr].contains('Hyper-V Requirements:')) {
      $HVStart = $cntr
    } # End If

  }   # End For

  If ($WinHome) {

    For ($cntr = $HVStart; $cntr -lt $x.count ; $cntr++) {

        If ($x[$cntr].contains(
            'A hypervisor has been detected.')) {
          $HyperV = 'Enabled'

        }

        If ($x[$cntr].contains(
            'Virtualization Enabled In Firmware: Yes')) {
          $HyperV = 'Enabled'
        }

        If ($x[$cntr].contains(
            'Virtualization Enabled In Firmware: No')) {
          $HyperV = 'Disabled'
        }

    } #End For

    $FeatureName = 'HypervisorPlatform'

  } #End If

  Else { #Win 10 Pro or above...

    For ($cntr = $HVStart; $cntr -lt $x.count ; $cntr++) {

        If ($x[$cntr].contains(
            'A hypervisor has been detected.')) {
          $HyperV = 'Enabled'
        }
        Else { $HyperV = 'Disabled' }
    } #End For

    $FeatureName = 'Microsoft-Hyper-V'

  } #End Else

  $GWOFArgs = @{Online      = $True
                FeatureName = $FeatureName
                ErrorAction = "Stop"}

  Try {
       $HVStatus = (Get-WindowsOptionalFeature @GWOFArgs).State

       If ($VirtualHW.IsPresent) { Return ,"$HyperV" }
       Else { Return ,"$HVStatus" }
  }
  Catch {Return ,"Requires Administative Privileges"}

} #------------- End FunctionGet-HyperVStatus  ----------------

Function Get-InternetStatus {

  $ProgressPreference           = "SilentlyContinue"
  $TNCArgs = @{ComputerName     = "google.com"
               TraceRoute       = $True
               Hops             = 3
               WarningAction    = "SilentlyContinue"}

  $INetStatus = Test-NetConnection @TNCArgs

  If ($INetStatus.Pingsucceeded) {
    $Script:ActiveConnectionIndex = $INetStatus.InterfaceIndex

    $InterNetStatus =
      "Internet connection is active: "   +
      "$($InetStatus.TraceRoute.Count -eq 3)`n" +
      "                           On: "   +
      "$($INetStatus.InterfaceAlias) ("   +
      "No: $($INetStatus.InterfaceIndex)" +
      ")"                                 | Out-String
  } #End If ($INetS...

  Else {$Script:ActiveConnectionIndex = 0
        $InterNetStatus =
         "There is NO Internet Connection available!" |
         Out-String
       }

  Return , "$InterNetStatus"

} #----------   End Function Get-InternetStatus  --------------

Function Get-IPParams {

  Param (
     [Parameter(Mandatory=$True)]
       [Int] $Index
  )

  If ($Index -eq 0) {
    $IPInfo = "Configuration information not available`n" +
              "No Internet Connection!"
  }

  Else {
    $GCIMArgs = @{Class  = "Win32_NetworkAdapterConfiguration"
                  Filter = "IPEnabled=TRUE"
                  ComputerName = "."
                 }
    $x = Get-CIMInstance @GCIMArgs |
         Where-Object {$_.InterFaceIndex -eq $Index} |
      Select-Object -Property [a-z]* -ExcludeProperty IPX*,WINS*

    $y = Get-NetIPConfiguration -InterfaceIndex $Index

    $DNSAddr = ""

    If ($y.DNSServer.ServerAddresses -is [System.array]) {

      ForEach ($addr in $y.DNSServer.ServerAddresses) {
        $DNSAddr += "$addr`n              "
      } #End ForEach

      $DNSAddr = $DNSAddr.Trim()

    }   #End If

    Else {
      $DNSAddr = $y.DNSServer.ServerAddresses
    }

    $IPInfo = "Active Network Adapter Configuration:`n"   +
              "-------------------------------------`n"   +
              "No          : $($x.InterfaceIndex)`n"      +
              "IP Address  : $($x.IPAddress[0])`n"        +
              "Sub-Net Mask: $($x.IPSubnet[0])`n"         +
              "Gateway     : $($x.DefaultIPGateway[0])`n" +
              "DNS  Domain : $($x.DNSDomain)`n"           +
              "     Server : $($DNSAddr)`n"               +
              "DHCP Enabled: $($x.DHCPEnabled)`n"         +
              "DHCP Server : $($x.DHCPServer)`n"          +
              "MAC  Address: $($x.MacAddress)`n"          |
              Out-String

  } #End Else ($Index -eq 0...

  Return ,$IPInfo

} #----------------  End Function Get-IPParms  ----------------

Function Get-NetworkComputer {

#Note: Requires SMBv1 to be Enabled as of Win 1909

# Check to see if SMBv1 is Enabled!

  $GWOFArgs = @{Online      = $True
                FeatureName = "SMB1Protocol"
                ErrorAction = "Stop"}
  Try {
    If ((Get-WindowsOptionalFeature @GWOFArgs).State -ne
          "Enabled") {
      $WgComps =
          "Windows Optional Feature SMBv1 is NOT enabled!`n" +
          "To list Network Computers please enable SMBV1,`n" +
          "reboot, and rerun this progam."
    }    #End If...
    Else {
      $wgrp = (
       [adsi]"WinNT://$(
         (Get-CimInstance Win32_ComputerSystem).Domain)"
              ).Children.path

      $WGComps = $Null

      ForEach ($Item in $wgrp) {

         If ($Null -ne $Item) {
           $temp = $Item.ToString() -Split('/')
           [Array]$WGComps += "$($temp[$($Temp.Count)-1])"
         } #If ($Item...

      }    #End ForEach

    } #End Else

  } #End Try...
  Catch {$WgComps = "`n`tRequires Administative Privileges"}

  Return ,$WGComps

} #-------------- End Function Get-NetworkComputer -----------

Function Get-Networks {

  $NetStatus = ("Network Connectivity:`n" +
                "---------------------" | Out-String)
  $fmtNetCon =
     @{Label="`nName";Width=15;
           Expression={$_.Name                           }},
     @{Label="`nNo";Width=2;
           Expression={$_.InterfaceIndex                 }},
     @{Label="Network`nCategory";   Width=8;Align="Left";
           Expression={$_.NetworkCategory                }},
     @{Label="`nIPv4";                Width=8;Align="Left";
           Expression={$_.IPv4Connectivity               }},
     @{Label="`nIPv6";                Width=8;Align="Left";
           Expression={$_.IPv6Connectivity               }}

  $NetWorks = Get-NetConnectionProfile

  If ($Null -eq $Networks) {
    $NetStatus += ("No Network Connections`n" | Out-String)
  }
  Else {
    $NetStatus += ($Networks | Format-Table $fmtNetCon |
                               Out-String)
  }

  Return ,$NetStatus

} #-----------------  End Function Get-Networks --------------

Function Get-MaxLength {

<#
.SYNOPSIS
   Finds the length of the longest item in collection.

.DESCRIPTION
   Use this Function to get the length of the longest item in a
   collection for use in format strings or other places where
   needed.

.PARAMETER TestObj
    The qualified object to be tested. See example!

.Parameter MinLen
    The minimum length of the item (if using for formatting) which
    should be the Label (title) length. Note if the object item
    being tested does not have a Length property you MUST specify
    the label length!

.OUTPUTS
    Returns a numerical value

.EXAMPLE
   $NameLen = Get-MaxLength -TestObj $DotNet.PSChildName
   $VerLen  = Get-MaxLength -TestObj $DotNet.Version
   $RNLen   = Get-MaxLength -TestObj $DotNet.Release -MinLen 11

     #--- .Net Information ---

 $fmtDotNet =
  @{Expression={$_.PSChildName};Label=".Net Type";Width=$NameLen},
  @{Expression={$_.Version};Label="Version No:";Width=$VerLen},
  @{Expression={$_.Release};Label="Release No:";Width=$RNLen}

  $Dotnet | Format-Table $fmtDotNet
#>

  Param(
    [Parameter(Mandatory=$True)]
     [object] $TestObj,
    [Parameter(Mandatory=$False)]
     [int] $MinLen = 0,
    [Parameter(Mandatory=$False)]
     [int] $MaxLen = 0
  )

   $ErrorActionPreference = "SilentlyContinue"

   foreach ($x in $TestObj) {
     If ($x.Trim().length -gt $MinLen) {
       $MinLen = $x.Trim().length
     }
   }

   If ($MaxLen -ne 0) {
     If ($MinLen -gt $MaxLen) {
       $MinLen = $MaxLen
     }
   }

   $ErrorActionPreference = "Continue"

   Return ,$MinLen

}  #-----------  End Function Get-MaxLength  ------------------

Function Get-MSDefCon {

 <#
  .Synopsis
   Retrieve MS-DEFCON setting from AskWoody.com
   and display in dialog box.

  .Description
   Retrieve MS-DEFCON setting from AskWoody.com
   and display in dialog box.

  .Outputs
   Dialog box with the current MS-DEFCON setting

  .Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 11 Mar 2019
     Last Updated : 02 Apr 2020
     Current Vers : 3.1 Adjusted from stand-alone to function.

  .Example
   $MSDefCon = Get-MSDefCon

#>


# Two blank linea above required for Comment-Based Help

  $DEFCONText = @{
    1 = "Current Microsoft patches are causing havoc.`n"     +
        "Don’t patch."
    2 = "Patch reliability is unclear.`n"                    +
        "Unless you have an immediate,`npressing need to "   +
        "install a specific patch,`n`n Don’t do it!"
    3 = "Patch reliability is unclear, "                     +
       "but widespread attacks make patching prudent.`n"     +
       "Go ahead and patch, but watch out for "              +
       "potential problems."
    4 = "There are isolated problems with current patches, " +
        "but they are well-known and documented here.`n"     +
        "Check this site to see if you’re affected and if "  +
        "things look OK... `n`n Go ahead and patch."
    5 = "All’s clear. Patch while it’s safe."
   }

  $LocalDTFmt = (Get-Culture).DateTimeFormat.ShortDatePattern
  $LocalTMFmt = (Get-Culture).DateTimeFormat.ShortTimePattern

  $LocalDateTime = Get-Date

  $DispDT = (Get-Date $LocalDateTime -Format $LocalDTFmt)

  If ($DispDT.Length -lt 10) { $DispDt = " " + $DispDT }

  $DispTM = (Get-Date $LocalDateTime -Format $LocalTMFmt)

  If ($DispTM.Length -lt  8) { $DispTM = " " + $DispTM}

  $DispDT += " " + $DispTM

  $ProgressPreference = "SilentlyContinue"

  $IWRArgs = @{URI = "https://www.askwoody.com/"
               UseBasicParsing = $True
               ErrorAction     = "Stop"
              }
  Try {

     $WebResponse = Invoke-WebRequest @IWRArgs

       If (($WebResponse).Images.Count -gt 0 ) {
  
       ForEach ($Image in $WebResponse.Images) {
  
         $x = $Image.OuterHTML.ToString()
  
         If ($x -like "*MS-DEFCON-*") {
  
           $y = $x.split('/')
           $z = $y[8].split('.')
           $DEFCONNo = [Int]$z[0].SubString(10,1)
  
           $Msg =
           @"
+-------------------------------------------------------+
| AskWoody MS-DEFCON Setting as of: $DispDT |
|                                                       |
| MS-DEFCON: $DEFCONNo                                          |
+-------------------------------------------------------+
$($DEFCONText[$DEFCONNo])
"@
           Return , ($Msg | Out-String)
  
         } #End If ($x...
  
       }   #End ForEach ($Image...

     }  #End If (($WebResponse).Images.Count...
     Else {
       "A bug in PS7 prevents retrieval of MS-DefCon!" | 
        Out-String
     }

  } #End Try

  Catch { "Can not Retrieve MS-DEFCON, No Internet Conection" |
          Out-String }

} #------------ End Function Get-MSDefCon

Function Get-PhysicalScreenSize {

<#
  .Synopsis
    Queries WMI to find basic monitor size information.
    It then performs some math on these attributes to come up with the
    Physical size of all monitors attached to a local or remote device.

  .Description

  .Parameter Computername

  .Outputs
   Custom PSObject - $MonPhySize
   TypeName: System.Management.Automation.PSCustomObject

   Name        MemberType   Definition
   ----        ----------   ----------
   Equals      Method       bool Equals(System.Object obj)
   GetHashCode Method       int GetHashCode()
   GetType     Method       type GetType()
   ToString    Method       string ToString()
   Instance    NoteProperty Object[] Instance=System.Object[]
   MonitorSize NoteProperty Object[] MonitorSize=System.Object[]

  .Notes
     Original PGMR: Adam Bertram
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      :
     Last Updated : 10 Mar 2017
     Current Vers : 2.0

     03/10/17 Modified code to include the InstanceName
              to be used to matchup with the DeviceID
              in the $mons object. Note: this class adds
              an additional _0 to the ent of the
              InstanceName so code was required to strip
              it off so the matchup could be made!

  .Example
    Get-PhysicalScreenSize -ComputerName "MyComputer"
#>


# Two blank linea above required for Comment-Based Help

    param(
        [Parameter(Mandatory=$False)]
        $ComputerName = $env:COMPUTERNAME
    )

    $MonPhySize = [PSCustomObject]@{Instance='';
                            ;MonitorSize=''}

    $CIMArgs = @{ComputerName = $env:COMPUTERNAME
                 Namespace = 'root\wmi'
                 Query = "SELECT MaxHorizontalImageSize," +
                         "MaxVerticalImageSize," +
                         "InstanceName FROM " +
                         "WmiMonitorBasicDisplayParams"
                }
    $MonObj = Get-CIMInstance @CIMArgs
    $sizes    = @();
    $Instance = @();
    if ($MonObj -is [System.Array]) {
      foreach ($i in $MonObj) {
              $x = [System.Math]::Pow(
                    $i.MaxHorizontalImageSize/2.54,2)
              $y = [System.Math]::Pow(
                    $i.MaxVerticalImageSize/2.54,2)
              $sizes += [System.Math]::Round(
                        [System.Math]::Sqrt($x + $y),0)
              $Instance += ($i.InstanceName -split '_')[0]
      }##endforeach
    }
    Else {
              $x = [System.Math]::Pow(
                    $MonObj.MaxHorizontalImageSize/2.54,2)
              $y = [System.Math]::Pow(
                    $MonObj.MaxVerticalImageSize/2.54,2)
              $sizes += [System.Math]::Round(
                        [System.Math]::Sqrt($x + $y),0)
              $Instance += ($MonObj.InstanceName -split '_')[0]
    } #endif

      $MonPhySize.Instance    = $Instance
      $MonPhySize.MonitorSize = $sizes

 #Note address as: $MonPhySize.Instance[0]
 #                 $MonPhySize.MonitorSize
      $MonPhySize
}

Function Get-ScreenResolution {

#Note: Results affected by Windows Customize your
#      display settings

  $Screens = [system.windows.forms.screen]::AllScreens

  foreach ($Screen in $Screens) {
   $DeviceName = $Screen.DeviceName
   $Width      = $Screen.Bounds.Width
   $Height     = $Screen.Bounds.Height
   $IsPrimary  = $Screen.Primary

   $OutputObj = New-Object -TypeName PSobject

   $AMArgs = @{MemberType = 'NoteProperty'
               Name       = 'DeviceName'
               Value      = $DeviceName}
   $OutputObj | Add-Member @AMArgs

   $AMArgs = @{MemberType = 'NoteProperty'
               Name       = 'Width'
               Value      = $Width}
   $OutputObj | Add-Member @AMArgs

   $AMArgs = @{MemberType = 'NoteProperty'
               Name       = 'Height'
               Value      = $Height}
   $OutputObj | Add-Member @AMArgs

   $AMArgs = @{MemberType = 'NoteProperty'
               Name       = 'IsPrimaryMonitor'
               Value      = $IsPrimary}
   $OutputObj | Add-Member @AMArgs

   $OutputObj

  } #End ForEach

}   #-----------  End Function Get-ScreenResolution  ----------

Function Get-UserAccount{

<#+---------------------------------------------------+
  | From CH 14 PowerShell and WMI by Richard Siddaway |
  +---------------------------------------------------+
#>
  [CmdletBinding()]
  param (
   [parameter(ValueFromPipeline=$true,
     ValueFromPipelineByPropertyName=$true)]
    [string]$ComputerName = "$env:COMPUTERNAME"
  )

  PROCESS{
   Get-CimInstance -ClassName 'Win32_UserAccount' |
    Select-Object -Property AccountType, Description, Disabled,
        Domain, FullName,InstallDate, LocalAccount, Lockout,
        Name, SID, SIDType, PasswordChangeable,PasswordExpires,
        PasswordRequired
  } #End Process

}  #-------------  End Function Get-UserAccount  -------------

Function Get-WiFi {

  Param (
     [Parameter(Mandatory=$False)]
       [String] $SSID
  )

  If ($SSID -eq "") {
  # *** Retrieve SSID ***

  <#+---------------------------------------------------------+
    | Note: this construct works because the All User Profile |
    |       happens to be the last line in the output!        |
    +---------------------------------------------------------+#>

  $WiFi = netsh wlan show profiles

    If ($WiFi -Like "*no wireless interface*") {
     $WiFiSettings = "Wi-Fi Information`n"   +
                     "-----------------`n"   +
                     "$Wifi`n"               | Out-String
     Return ,$WiFiSettings
    }
    Else {
    $SSArgs = @{InputObject = $WiFi
                Pattern     = 'All User Profile'}
    $SSID  = ((Select-String @SSArgs) -split ":")[-1].Trim()
    }

  } #End If ($SSID -eq "")

  #*** Retrieve WiFi Information ***

  <#+--------------------------------------------------------+
    | Note: Since the next two items do NOT happen to be the |
    |       the last line in the output it is necessary to   |
    |       pass it down the pipe to get it serialized so    |
    |       the items will be selected properly!             |
    +--------------------------------------------------------+#>

  $WifiSettings = netsh wlan show profile name="$SSID" key=clear

  If ( $WiFiSettings -like "*is not found*") {
  #SSID looked up so this is not necessary!
  } #End If

  Else {
    $Mode = $WiFiSettings |
            Select-String -pattern "Connection mode"
    $Mode = ($Mode -split ":")[-1].Trim()

    $Switch = $WiFiSettings |
            Select-String -pattern "AutoSwitch"
    $Switch = ($Switch -split ":")[-1].Trim()

    $MacRnd = $WiFiSettings |
            Select-String -pattern "MAC Randomization"
    $MacRnd = ($MacRnd -split ":")[-1].Trim()

    $Security = $WiFiSettings |
                Select-String -Pattern 'Authentication'
    $Security = ($Security -Split ":")[-1].Trim()

    $PW = $WiFiSettings |
           Select-String -Pattern "Key Content"
    $PW = ($PW -Split ":")[-1].Trim()

    $WiFiSettings =
       "Wi-Fi Information`n"             +
       "-----------------`n"             +
       "SSID              : $SSID`n"     +
       "Connection Mode   : $mode`n"     +
       "Auto Switch       : $Switch`n"   +
       "MAC Randomization : $MacRnd`n"   +
       "Security          : $Security`n" +
       "Password          : $PW`n"       | Out-String
  } #End Else

  Return ,$WifiSettings

} #--------------- End Function Get-WiFi ----------------------

Function Get-WindowsProductKey {

  param (
          [Parameter(Mandatory=$True)]
            [string]$CompName
        )

  $Reg = [WMIClass] ("\\" + $CompName +
                     "\root\default:StdRegProv")
  $values = [byte[]]($reg.getbinaryvalue(2147483650,
            "SOFTWARE\Microsoft\Windows NT\CurrentVersion",
            "DigitalProductId").uvalue)
  $lookup = [char[]]("B","C","D","F","G","H","J","K","M",
                     "P","Q","R","T","V","W","X","Y",
                     "2","3","4","6","7","8","9")
  $keyStartIndex      = [int]52;
  $keyEndIndex        = [int]($keyStartIndex + 15);
  $decodeLength       = [int]29
  $decodeStringLength = [int]15
  $decodedChars       = New-Object char[] $decodeLength
  $hexPid             = New-Object System.Collections.ArrayList

  for ($i = $keyStartIndex; $i -le $keyEndIndex; $i++) {
    [void]$hexPid.Add($values[$i])
  }

  for ( $i = $decodeLength - 1; $i -ge 0; $i--) {

    if (($i + 1) % 6 -eq 0) {
      $decodedChars[$i] = '-'
    }
    else
    {
      $digitMapIndex = [int]0
      for ($j = $decodeStringLength - 1; $j -ge 0; $j--)
      {
       $byteValue = [int](($digitMapIndex * [int]256) -bor
                           [byte]$hexPid[$j])
       $hexPid[$j] = [byte] ([math]::Floor($byteValue / 24))
       $digitMapIndex = $byteValue % 24
       $decodedChars[$i] = $lookup[$digitMapIndex]
      }
    }
  }   #End For ($i...)

  $WindowsKey = ''
  $decodedChars | ForEach-Object { $WindowsKey += $_ }

  Return ,$WindowsKey

}  #----------  End Function Get-WindowsProductKey ------------

Function Get-WUsetting {

  $GIArgs = @{path = 'HKLM:\SOFTWARE\Microsoft\Windows\' +
                     'CurrentVersion\WindowsUpdate\Auto Update'}
  $WUSettings  =  Get-Item @GIArgs

  Try {
  $GIArgs = @{Path = 'HKLM:\SOFTWARE\Policies\Microsoft\' +
                 'Windows\WindowsUpdate\AU'
              ErrorAction = "Stop"
             }
  $WUPolicies  =  Get-Item @GiArgs
  }
  Catch {$WUPolicies = $Null}

  $NotifyLevel =
     Switch ($WUSettings.GetValue('AUOptions')){
           1 {'Never Check for updates'}
           2 {'Check for updates but let me choose ' +
                'whether to download and install them'}
           3 {'Download updates but let me choose ' +
               'whether to install them'}
           4 {'Install updates automatically'}
           default {
              'Updates status could not be determined'}
     } #End Switch

  If ($Null -eq $WUPolicies) {
    $WUDays = 'N/A'
    $SchInstTime = 'N/A'
  }
  Else {
    $WUDays =
       Switch ($WUPolicies.GetValue('ScheduledInstallDay')) {
             0 {'Every Day'}
             1 {'Sunday'}
             2 {'Monday'}
             3 {'Tuesday'}
             4 {'Wednesday'}
             5 {'Thursday'}
             6 {'Friday'}
             7 {'Saturday'}
       } #End Switch

     $SchInstTime =
       $WUPolicies.GetValue('ScheduledInstallTime')
  } #End If ($WUPolicies...

  $RecUpdates =
     (&{If ($WUSettings.GetValue('IncludeRecommendedUpdates')) {
          "Yes"} Else {"No"}})

  Return ,$("$NotifyLevel" + ":" + "$WUDays" + ":" +
            "$SchInstTime" + ":" + "$RecUpdates")

}  #----------------- End Function Get-WUsetting ------------

Function Get-W10UpdateHistory {

<#
  .Synopsis
   Retrieve the Windows (minor) update history.

  .Description
   Retrieve the Windows (minor) update history.

  .Outputs
   String containing a Formatted and sorted listing the
   Minor Update history with dates/time suitable for display
   in a TextBox control.

  .Notes
     Programmer   : RetiredGeek (@askWoody.com)
                    aka: ComputerMentor
     Created      : 20 Jan 2020
     Last Updated :
     Current Vers : 1.0

  .Example
   $Variable = Get-Win10UpdateHistory.ps1


#>


# Two blank linea above required for Comment-Based Help

 $HFFmt = @{Expression={
             ($_.InstalledOn).ToString("yyyy/MM/dd")};
            Label="Installed On";     Width=12},
          @{Expression={$_.HotFixID};
            Label="Hot Fix ID";       Width=10},
          @{Expression={$_.CSName};
            Label="Computer";         Width=14},
          @{Expression={$_.Description};
            Label="Description";      Width=35}

  $x = Get-HotFix

  $RetVal = $x |
    Sort-Object -Descending -Property InstalledOn |
    Format-Table $HFFmt | Out-String

  Return ,"$RetVal"

} #End ------- Function Get-W10UpdateHistory  -----------------

Function Get-W10UpgradeHistory {

<#
  .Synopsis
   Retrieve the Windows (major) update history.

  .Description
   Retrieve the Windows (major) update history.

  .Outputs
   String containing a Formatted and sorted listing the
   Major Update history with dates/time suitable for display
   in a TextBox control.

  .Notes
     Programmer   : RetiredGeek (@askWoody.com)
                    aka: ComputerMentor
     Created      : 20 Jan 2020
     Last Updated :
     Current Vers : 1.0
  +-----------------------------------------------------------+
  |Note: Update Started shows the date/time that an update    |
  |      was initiated to the next version, the Vers is the   |
  |      currently installed version when the update was      |
  |      initiated, Vers. Installed shows the date/time when  |
  |      the Vers was installed.                              |
  +-----------------------------------------------------------+

  .Example
   $Variable = Get-Win10UpgradeHistory.ps1

#>


# Two blank linea above required for Comment-Based Help

$RegEx = "Updated\son\s(\d{1,2}\/\d{1,2}" +
         "\/\d{4}\s\d{2}:\d{2}:\d{2})\)$"

$OSUpgradeHistory = $(Get-ChildItem "HKLM:\System\Setup" |
  Where-Object {$_.Name -match "\\Source\s"})  |
  ForEach-Object { $_   |
     Select-Object @{Name="UpdateTime";
         Expression={if ($_.Name -match "$RegEx") {
           [dateTime]::Parse($Matches[1],
           ([Globalization.CultureInfo]::CreateSpecificCulture(
          'en-US')))}}},
       @{Name="Release"    ;
         Expression={$_.GetValue("ReleaseID")   }},
       @{Name="Branch"     ;
         Expression={$_.GetValue("BuildBranch") }},
       @{Name="Build"      ;
         Expression={$_.GetValue("CurrentBuild")}},
       @{Name="ProductName";
         Expression={$_.GetValue("ProductName") }},
       @{Name="InstallTime";
         Expression={[datetime]::FromFileTime(
                            $_.GetValue("InstallTime"))}}

 } #End ForEach-Object...

 $UPGRFmt = @{Expression={$_.Release};
              Label="`nVers";           Width=4},
            @{Expression={$_.Branch};
              Label="`nBranch";         Width=23},
            @{Expression={$_.Build};
              Label="`nBuild";          Width=5},
            @{Expression={$_.ProductName};
              Label="`nProduct Name";   Width=14},
            @{Expression={
              Get-Date($_.InstallTime) -f "yyyy/MM/dd"};
              Label="Vers. Installed";  Width=10},
            @{Expression={Get-Date($_.InstallTime) -f "HH:mm"};
              Label="`nTime";           Width=5}

  $RetVal = $OSupgradeHistory  |
    Sort-Object -Descending -Property UpdateTime |
    Format-Table $UPGRFmt | Out-String

  Return ,"$RetVal"

} #End ---------  Function Get-W10UpgradeHistory  -------------

Function HardwareTab {

  $BTArgs = @{Text    = "Investigating Computer Setup"
              Title   = "I'm working on $CompName"
              Icon    = 'Info'
              Timeout = [Int32]500}
  Show-BalloonTip  @BTArgs

  $enclosureNames = @{
     1 = "Other"
     2 = "Unknown"
     3 = "Desktop"
     4 = "Low Profile Desktop"
     5 = "Pizza Box"
     6 = "Mini Tower"
     7 = "Tower"
     8 = "Portable"
     9 = "Laptop"
    10 = "Notebook"
    11 = "Hand Held"
    12 = "Docking Station"
    13 = "All-in-One"
    14 = "Sub Notebook"
    15 = "Space Saving"
    16 = "Lunch Box"
    17 = "Main System Chassis"
    18 = "Expansion Chassis"
    19 = "Sub-Chassis"
    20 = "Bus Expansion Chassis"
    21 = "Peripheral Chassis"
    22 = "Storage Chassis"
    23 = "Rack Mount Chassis"
    24 = "Sealed-Case PC"
  }

  Add-Row -ARIN 'Manufacturer' -ARIV $CompSysObj.Manufacturer
  Add-Row -ARIN 'Model'        -ARIV $CompSysObj.Model
  Add-Row -ARIN 'System Type'  -ARIV $CompSysObj.SystemType

  $enclosure =
    Get-CimInstance -ClassName 'Win32_SystemEnclosure'

  $ChassisNo = $enclosure.ChassisTypes[0]
  $ChassisName = if ($ChassisNo -gt $enclosureNames.count) {
                        "Currently Unassigned"
                 }
                 else {
                        $enclosureNames[[int]$ChassisNo]
                 }

  Add-Row  -ARIN 'Enclosure' -ARIV $ChassisName
  Add-Row  -ARIN ''

  $GCIArgs = @{Class      = 'Win32_BIOS'
               Namespace  = 'Root\CIMV2'
              }
  $BIOSObject =  Get-CimInstance @GCIArgs

  Add-Row -ARIN 'BIOS Serial No' -ARIV $BIOSObject.SerialNumber
  Add-Row -ARIN '     Name'      -ARIV $BIOSObject.Name
  $ARArgs = @{ARIN = '     Version'
              ARIV = $BIOSObject.SMBIOSBIOSVersion }
  Add-Row @ARArgs

  $ARArgs =
   @{ARIN = '     Date'
     ARIV = (Get-Date $BIOSObject.ReleaseDate -Format $LocalDTFmt) +
             " " +
            (Get-Date $BIOSObject.ReleaseDate -Format $LocalTMFmt)
    }
  Add-Row @ARArgs

  $ARArgs = @{ARIN = '     Manufacturer'
              ARIV = $BIOSObject.Manufacturer}
  Add-Row @ARArgs

  If ($BIOSObject.BIOSVersion.count -eq 3) {
    $Separator = "-"
    $BIOSManuf =
       $BIOSObject.BIOSVersion[2].split($Separator)
         #BIOS Manufacturer
    Add-Row -ARIN '     Creator'  -ARIV $BIOSManuf[0]
  }

    Add-Row -ARIN ''

#--- PS InLine Switch construct! ---
    $FT= (&{
    Switch (Get-BiosType -CompName "$CompName") {
          1       {"Legacy BIOS"}
          2       {"UEFI"}
          Default {"Unknown"}
    }})
    Add-Row -ARIN 'Firmware Type' -ARIV "$FT"

    If ($FT -eq 'UEFI') {
      Try {
        $GIArgs = @{Path = "HKLM:\SYSTEM\ControlSet001\" +
                           "Control\SecureBoot\state"
                    ErrorAction = 'Stop'}
        $UEFISB = get-item @GIArgs
        $UEFIStatus =
          (&{If($UEFISB.GetValue('UEFISecureBootEnabled') -eq 1)
                                 {"Enabled"} Else {"Disabled"}})
      }
      Catch { $UEFIStatus = "Not Suported" }

      Add-Row -ARIN 'Secure Boot UEFI'  -ARIV $UEFIStatus

    } #End If ($FT -eq 'UEFI')

  $TPM = Get-Tpm

  If ($TPM -Like "*Administrator*") {
      $TPMStatus = $TPM
  }
  Else {
  $TPMStatus = (&{If($($TPM).TpmPresent) {"Present"}
                  Else {"Not Present"}})
  }

  Add-Row -ARIN 'Trusted Platform Module' -ARIV "$TPMStatus"

  $HWWidth = Get-MaxLength -TestObj $CITable.Item -MinLen 20

  $fmtHardware =
    @{Expression={$_.Item};Label="Item  ";Width=$HWWidth},
    @{Expression={$_.Value};Label="Value";Width=78-$HWWidth}

  $HWInfo = $CITable | Format-Table -Property $fmtHardware |
                       Out-String

  Return , $($HWInfo.Substring(0,($HWInfo.Length-2)))


} # ------------------   End HardwareTab  ---------------------

Function Install-Program {

  #--- Create Scheduled Task to runas Administrator   ---
  $TaskSet = New-ScheduledTaskSettingsSet -Compatibility Win8
  $TaskSet.DisallowStartIfOnBatteries = $False
  $TaskSet.StopIfGoingOnBatteries     = $False
  $TaskSet.IdleSettings.StopOnIdleEnd = $False

  $PSVer = $PsVersionTable.PSVersion.Major
  If ($PSVer -gt 5) {
    $TaskCommand = "C:\Program Files\PowerShell\7\pwsh.exe"
  }
  Else {$TaskCommand = "c:\windows\system32\windowspowershell" +
                       "\v1.0\powershell.exe"
  }
  $TaskName    = "CMsLocalPCInfo as Administrative Rights PSv" +
                 "$PSVer"
  $TaskArgs    = "-WindowStyle Minimized -Command " + '"' +
                 $PSScriptRoot + "\" + "$ScriptName"
  $TaskDesc    = "Run CMsLocalPCInfo.ps1 as Administrator"
  $TAArgs = @{Execute  = '"' + $TaskCommand + '"'
              Argument =   $TaskArgs
             }

  $TaskAction = New-ScheduledTaskAction @TAArgs

  $RSTArgs = @{Action      = $TaskAction
               TaskName    = "$TaskName"
               User        = "$env:USERDOMAIN\$env:username"
               RunLevel    = "Highest"
               Description = "$TaskDesc"
               Settings    = $TaskSet}

  Register-ScheduledTask @RSTargs -Force


  #--- Create Desktop Shortcut to call Scheduled Task ---

  $ShortcutName    = "$TaskName"
  $DestinationPath = "C:\Users\$env:Username\Desktop"

  $WshShell = New-Object -comObject WScript.Shell
  $Shortcut = $WshShell.CreateShortcut(
              "$DestinationPath\$ShortcutName.lnk")
  $Shortcut.TargetPath = "C:\Windows\System32\schtasks.exe"
  $Shortcut.Arguments = "`/run `/TN " + '"' + 
                        $ShortcutName + '"'
  $Shortcut.WindowStyle = 7
  $Shortcut.IconLocation = $TaskCommand

  $Shortcut.Save()

} #------------  End Install-Program   -------------------------

Function InternetTab {

  $Internet  = Get-InternetStatus

  $fmtNetIP =
     @{Label="`nInterface";  Width=9;
                        Expression={$_.InterfaceAlias }},
     @{Label="IP`nVers";  Width= 4;Align='Center';
                        Expression={$_.AddressFamily  }},
     @{Label="`nIP Address"; Width=50;
                        Expression={$_.IPAddress      }}


  $IPArgs = @{ErrorAction    = 'SilentlyContinue'
              InterfaceAlias = "Ethernet","Wi-Fi"
              InterfaceIndex = $Script:ActiveConnectionIndex}

  $IPInfo = Get-NetIPAddress @IPArgs  |
     Sort-Object  -Property InterfaceAlias, AddressFamily,
                            IPAddress |
     Format-Table -Property $fmtNetIP |
     Out-String   -Width    $OStrWidth

  Return ,$($Internet + $IPInfo )

} #--------------------- End Internet Tab ---------------------

Function LocalNetworkTab {

  $NetStatus = Get-Networks

  $NetworkList = 'Network Computers'    | Out-String
  $NetworkList += ('-----------------'  | Out-String)
  $NetworkList += (Get-NetworkComputer  |
                           Format-Table | Out-String)
  $NetworkList += " "                   | Out-String

  Try {
       $GIArgs = @{Path = "WSMan:\localhost\Client\TrustedHosts"
                   ErrorAction = 'Stop'}
       $THosts = Get-Item @GIArgs
       $HostsList = $THosts.value.Split(',')
  }
  Catch   {$HostsList = 'N/A'}
  Finally {
           $TrustedHosts =  'Trusted Hosts'  | Out-String
           $TrustedHosts += '-------------'  | Out-String
           $TrustedHosts += $HostsList       |
                             Format-Table    | Out-String
           $TrustedHosts += " "              | Out-String
  }

  $GWOFArgs = @{Online      = $True
                FeatureName = "SMB1Protocol"
                ErrorAction = "Stop"}
  Try {
       $SMBv1 = (Get-WindowsOptionalFeature @GWOFArgs).State
  }
  Catch   {$SMBv1 = "`n`tRequires Administative Privileges"}
  Finally {
      $SMBOut = ("SMBv1 Status`n-----------" | Out-String) +
                ("$SMBv1`n"                  | Out-String)
  }

  Return ,$($NetStatus + $NetworkList + $TrustedHosts + $SMBOut)

} #------------- End LocalNetwork Tab ------------------------

Function LogicalDiskTab {

  $DiskTypeHash = @{
   2 = "Removable"
   3 = "Fixed"
   4 = "Network"
   5 = "Compact"}

  $fmtDisk =
     @{Expression={$_.Name};Label="Drive`nLetter";
                   Align='Center';Width=6},
     @{Expression={$_.VolumeName};
         Label="Volume`nName";Width=15},
     @{Expression={$_.filesystem};
         Label="File`nSystem";Width=6},
     @{Expression={$DiskTypeHash.item([int]$_.DriveType)};
         Label="Drive`nType";Width=9},
     @{Expression={$_.compressed};
         Label="Compr`nessed";Width=5;Align='left'},
     @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
         Label="Disk Size`n / GB";Width=9;align='right'},
     @{Expression={ '{0:#,000.00}' -f ($_.FreeSpace/1gb)};
         Label="Free Space`n / GB";Width=10;align='right'},
     @{Expression={ '{0:N0}' -f $_.DiskIndex};
         Label="Drv`nNo";Width=4;Align='right'},
     @{Expression={
           (&{IF ($_.Bootable) {"Yes"} else {""}})};
         Label="`nBoot";Width=5;Align='left'}

  $DiskInfo = Get-Drive |
              Sort-Object  -Property Name     |
              Format-Table -Property $fmtDisk |
              Out-String   -Width $OStrWidth

  $DiskInfoTitle = "Logical Disk Information:"    | Out-String

  Return ,$($DiskInfoTitle + $DiskInfo)

} #-------------------  End LogicalDiskTab  -------------------

Function MapsAndSharesTab {

  $MappedDrives =
    Get-CimInstance -ClassName 'Win32_MappedLogicalDisk' |
       Where-Object {$_.size -ge 0 -and
                     $Null -ne $_.ProviderName}

  If ($Null -eq $MappedDrives) {

    $MDriveInfo = If ($AdminPriv) {@"

    --No Mapped Drives--

    If drives were mapped using standard user privileges,
    even from an administrator account, they will not show
    up if this script is run as Administrator!

"@

    } #End If ($AdminPriv)

    Else {
          @"

          --No Mapped Drives--

"@
    } #End $MDriveInfo = If...

    $MDriveInfo = $MDriveInfo | Out-String

  } #End If ($MappedDrives...)

  Else {
        $fmtMDR =
         @{Expression={$_.Name};Label="Drive`nLetter";Width=6},
         @{Expression={$_.Compressed};Label="Compressed";
                                      Width=5},
         @{Expression={$_.ProviderName};Label="Provider";
                                        Width=50},
         @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
               Label="Disk Size`n / GB";Width=9;align='right';}

        $MDriveInfo = $MappedDrives |
                      Format-Table -Property $fmtMDR |
                      Out-String   -Width $OStrWidth

  } #End Else ($MappedDrives...)

#Shares Info

  $fmtShare   =
    @{Expression={$_.Name};Label="`nShare Name";Width=30},
    @{Expression={'{0:### }' -f $($_.MaximumAllowed)};
      Label="Max`nUsers"; Width=5;align='Right'},
    @{Expression={$_.Path};Label="`nPath";Width=41}

  $GCIArgs = @{ClassName   = "Win32_Share"
               ErrorAction = "Stop"}

  Try {
  $ShareInfo  = Get-CimInstance @GCIArgs     |
                Where-Object {$_.Type -eq 0} |
                Sort-Object  -Property Name  |
                Format-Table -Property $fmtShare |
                Out-String   -Width $OStrWidth
  }
  Catch {$ShareInfo =
          "No Shares defined on this machine." | Out-String }

  $MDriveTitle   = "Mapped Drive Information:" | Out-String
  $ShareTitle    = "Share Information:"        | Out-String

  Return $($MDriveTitle + $MDriveInfo +
           $ShareTitle  + $ShareInfo)

}  #---------------  End MapsAndSharesTab ---------------------

Function MemoryTab {

  $MemoryInfo =
    Get-CimInstance -ClassName 'Win32_PhysicalMemory'  |
    Sort-Object  -Property {$_.DeviceLocator}

  $ManufLen =
     Get-MaxLength -TestObj $MemoryInfo.Manufacturer -MinLen 12

  $fmtMEM =
    @{Expression={$_.DeviceLocator.Trim('DIMM')};
                   Label="`nMB Slot";align='center';Width=15},
    @{Expression={$_.BankLabel.Trim('BANK ')};
                   Label="`nBank";Align='center';Width=4},
    @{Expression={$_.InterleavePosition};
                   Label="Bank`nPos";align='center';Width=4},
    @{Expression={$_.Speed};Label="`nSpeed";Width=5},
    @{Expression={$_.DataWidth};Label="Data`nWidth";Width=5},
    @{Expression={'{0:#.00}' -f ($_.Capacity/1gb)};
                   Label="Size`n/ GB";Width=5;align='right'},
    @{Expression={$_.Manufacturer};Label="`nManufacturer";
                                           Width=$ManufLen},
    @{Expression={$_.SerialNumber};Label="`nSerial No.";
                                   Width=10}

  $MemoryInfo = $MemoryInfo | Format-Table -Property $fmtMEM |
                              Out-String   -Width $OStrWidth

  $MemTitle     =  "Memory Information:" | Out-String

  Return ,$($MemTitle + $MemoryInfo)

} #-------------------   End MemoryTab    ---------------------

Function Merge-Object {

  param(
	 [Parameter(Mandatory=$True)] $object1,
	 [Parameter(Mandatory=$True)] $object2
	)

#	 Courtesy of: http://powershell.com/cs/media/p/7924.aspx

	$propertylistObj1 =
      @($object1 | Get-Member -ea Stop -memberType *Property |
                   Select-Object -ExpandProperty Name)
	$propertylistObj2 =
      @($object2 | Get-Member -memberType *Property |
                   Select-Object -ExpandProperty Name |
                   Where-Object { $_ -notlike '__*'})

	$propertylistObj2 | ForEach-Object {
		if ($propertyListObj1 -contains $_) {
			$name = '_{0}' -f $_
		}
        else {
			$name = $_
		}

      $AMArgs = @{NotePropertyName = "$name"
                  NotePropertyValue = ($object2.$_)}

	  $object1 = $object1 |
         Add-Member @AMArgs -PassThru
	}

	Return ,$object1

} #----------------- End Function Merge-Object --------------

Function MonitorTab {

  $MonPhysSize = Get-PhysicalScreenSize

  $GCIArgs = @{Query =
    'select * from Win32_PnPEntity where service="monitor"'}
  $Mons = Get-CimInstance @GCIArgs |
          Select-Object -Property Name, Manufacturer,
                                  HardwareID, DeviceID

  $AMArgs = @{Name       = 'SerialNo'
              MemberType = 'NoteProperty'
              Value      = $null}
  $Mons | Add-Member @AMArgs

  $AMArgs = @{Name       = 'Size'
              MemberType = 'NoteProperty'
              Value      = $null}
  $Mons | Add-Member @AMArgs

  if ($MonPhysSize.MonitorSize -is [System.Array]) {
    ForEach ($xMon in $Mons) {
      For ($Cntr = 0; $Cntr -lt $MonPhysSize.MonitorSize.Count;
         $Cntr++ ) {
         If ($MonPhysSize.Instance[$Cntr] -eq
               $xMon.DeviceID) {
           $xMon.Size =
            [String]$($MonPhysSize.MonitorSize[$Cntr]) + '"'
         }
      }  #End For ($Cntr...
    }  #End ForEach ($xMon...
  }
  Else {
      $Mons.Size = [String]$($MonPhysSize.MonitorSize) + '"'
  }

  ForEach ($xx in $Mons) {

    $ErrorActionPreference = 'SilentlyContinue'

     $xx.HardwareID = $xx.HardwareId.split("\")[1]

     $keytype=[Microsoft.Win32.RegistryHive]::LocalMachine
     $reg=[Microsoft.Win32.RegistryKey]::OpenRemoteBaseKey(
                                                $keytype,"")
     $regKey = $reg.OpenSubKey(
         "SYSTEM\\CurrentControlSet\\Enum\DISPLAY" )
     $regKey = $reg.OpenSubKey(
         "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\$(
                                      $xx.HardwareId)")
     $DID = $regkey.GetSubKeyNames()

     ForEach($DID_KEY_NAME in $DID) {

           $regKey= $reg.OpenSubKey(
             "SYSTEM\\CurrentControlSet\\Enum\\DISPLAY\\" +
             "$($xx.HardwareId)\\$($DID_KEY_NAME)" +
             "\\Device Parameters")
           $EDID = $regKey.GetValue("EDID")

           ForEach($int in $EDID){
             $EDID_String = $EDID_String+([char]$int)
           }

           #Serial Number
           $checkstring = [char]0x00 + [char]0x00 + [char]0x00 +
                          [char]0xFF + [char]0x00
           $matchfound =  $EDID_String -match "$checkstring(\S+)"
           if($matchfound){
             $xx.SerialNo = [String]$matches[1]}
           else {
                 $xx.SerialNo = 'N/A'
           }

           $EDID_String = ''

     }   # End - foreach($DID_KEY_NAME in $DID)

  }   #  End ---- ForEach $xx ----

  $ErrorActionPreference = 'Continue'

  $MLArgs = @{TestObj = $Mons.Name
              MinLen  = 4
              MaxLen  = 25
             }
  $MNLen  = Get-MaxLength @MLArgs

  $MLArgs = @{TestObj = $Mons.Manufacturer
              MinLen  = 12
              MaxLen  = 25
             }
  $MFLen  = Get-MaxLength @MLArgs

  $MLArgs = @{TestObj = $Mons.SerialNo
              MinLen  = 6
              MaxLen  = 20
             }

  $ErrorActionPreference = "Stop"
  Try {
  $MSNo  = Get-MaxLength @MLArgs
  }
  Catch { $MSNo = 6 }
  $ErrorActionPreference = "Continue"

  $MLArgs = @{TestObj = $Mons.HardWareID
              MinLen  = 8
              MaxLen  = 14
             }
  $MRid  = Get-MaxLength @MLArgs

  $monfmt  = @{Expression={$_.Name};
                 Label="Monitor`nName";Width=$MNLen},
             @{Expression={$_.Manufacturer};
                 Label="`nManufacturer";Width=$MFLen},
             @{Expression={$_.SerialNo};
                 Label="Serial`nNumber";Width=$MSNo},
             @{Expression={$_.HardWareID};
                 Label="Registry`nID";Width=$MRid},
             @{Expression={$_.Size};
                 Label="Size";Width=4}

  $MonTitle   = "Monitor Information:" | Out-String

  $MonInfo = $mons | Format-Table -Property $monfmt |
                     Out-String

  $fmtMonRes = @{Expression={$_.DeviceName};
                 Label="Device Name";Width=12},
               @{Expression={$_.Width};
                 Label="Effective`nWidth";Width=9},
               @{Expression={$_.Height};
                 Label="Effective`nHeight";Width=9},
               @{Expression={$_.IsPrimaryMonitor};
                 Label="Primary`nMonitor";Width=7;
                 Align='Left'}

  $ResMsg = @"
    Resolution reported below is the effective resolution,
    i.e., it takes into consideration the scaling factor,
    e.g., if your monitor's actual resolution is:
    1280 x 1020 and scaled at 125% then the reported
    resolution will be 1536 x 864!

"@

  $MonRes = Get-ScreenResolution |
            Format-Table -Property $FmtMonRes | Out-String

  Return ,$($MonTitle + $MonInfo + $ResMsg + $MonRes)

} #--------------- End Function MonitorTab ---------------------

Function NetworkAdaptersTab {

  $NetInfo = Get-NetAdapter |
             Where-Object {$Null -ne $_.Name} |
             Sort-Object -Property @{
               Expression = "InterfaceOperationalStatus";
                             Descending = $False},
             @{Expression = "InterfaceIndex"            ;
                             Descending = $False}

  $GMArgs = @{TestObj = $NetInfo.Name
              MinLen  = 10
              MaxLen  = 21}
  $NameLen = Get-MaxLength @GMArgs

  $IFDLen = $OStrWidth - ($NameLen + 40)

  $fmtNet =
     @{Label="Connection`nName";Width=$NameLen;
           Expression={$_.Name                           }},
     @{Label="`nNo";Width=2;
           Expression={$_.InterfaceIndex                 }},
     @{Label="`nStatus";   Width=6;
           Expression={(&{IF($_.InterfaceOperationalStatus -eq 1) {
               'Up'} Else {'Down'}}) }},
     @{Label="Speed`n / MB";  Width=6;Align='right';
           Expression={ '{0:#,000}' -f ($_.Speed/1000000)}},
     @{Label="`nMAC Address";Width=17;
           Expression={$_.MacAddress                     }},
     @{Label="`nAdapter Name";Width=$IFDLen;
           Expression={$_.InterfaceDescription           }}

  $NetInfo = $NetInfo |
         Format-Table -Property $fmtNet -Wrap |
         Out-String   -Width $OStrWidth

  Return ,$NetInfo

} #----------------- End NetworkAdapters Tab ------------------

Function PhysicalDiskTab {

  #Physical Drive Info

  $PhyDiskInfo = Get-Disk  | Where-Object {$_.size -gt 0 } |
                 Sort-Object -Property DiskNumber

  $SSD = $False

  $AMArgs = @{Type  = 'NoteProperty'
              Name  = 'SSD'
              Value = 'No'}
  $PhyDiskInfo | Add-Member @AMArgs

  $AMArgs = @{Type  = 'NoteProperty'
              Name  = 'Speed'
              Value = '0'}
  $PhyDiskInfo | Add-Member @AMArgs

  $GCIArgs = @{
     NameSpace    = "root\Microsoft\Windows\Storage"
     Class        = "MSFT_PhysicalDisk"
     ErrorAction  = "SilentlyContinue"
  }
  $RotateSpeed = Get-CimInstance @GCIArgs |
    Select-Object -Property  DeviceID,@{Name="Speed/RPMs";
      Expression={(&{If($_.MediaType -eq 0) {[Int]0}
            Else {$_.SpindleSpeed/600000 -f "#,###"}})}} |
      Sort-Object DeviceID

ForEach ($x in $phydiskinfo) {

  ForEach ($Device in $RotateSpeed) {
      If ($x.number -eq $Device.DeviceID) {
        If ($Device.'Speed/RPMs' -eq 0) {
		    $SSD   = $True
          $x.SSD = "Yes"
        }  #End If
        Else {
         $x.Speed = $([Int]$Device.'SPeed/RPMs') -f "#,###"
        }
      }  #End If ($x.number...
  }      #End ForEach ($Device

} #End ForEach $x

$DNLen =
  Get-MaxLength -TestObj $PhyDiskInfo.Model        -MinLen 4
$SNLen =
  Get-MaxLength -TestObj $PhyDiskInfo.SerialNumber -MinLen 13

$fmtPhyDisk1 =
   @{Expression={ '{0:N0}' -f $_.Number};
       Label="Drive`n No.";Width=5;Align='Center'},
   @{Expression={$_.Model};Label="`nName";Width=$DNLen},
   @{Expression={$_.SSD};Label="`nSSD";Width=3;Align='left'},
   @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
       Label="Disk Size`n / GB";Width=9;align='right'},
   @{Expression={$_.NumberOfPartitions};
       Label="Parti`ntions";Width=5},
   @{Expression={$_.PartitionStyle};Label="GPT`nMBR";Width=3},
   @{Expression={(&{If ($_.IsBoot) {"Yes"} else {""}})};
       Label="`nBoot";Width=5;Align="Left"},
   @{Expression={(&{If ($_.BusType.GetType().Name -eq 'UInt16'){
       (& {Switch ($_.BusType) {
            0 {"Unknown"}
            1 {"SCSI"}
            2 {"ATAPI"}
            3 {"ATA"}
            4 {"1394"}
            5 {"SSA"}
            6 {"Fibre Channel"}
            7 {"USB"}
            8 {"RAID"}
            9 {"iSCSI"}
           10 {"SAS"}
           11 {"SATA"}
           12 {"SD"}
           13 {"MMC"}
           14 {"MAX"}
           15 {"File Backed Virtual"}
           16 {"Storage Spaces"}
           17 {"NVMe"}
           18 {"MS Reserved"}
           Default {"Unknown"}}})}
         Else{$_.BusType}})};Label="`nData Bus";Width=20}

  $fmtPhyDisk2 =
     @{Expression={ '{0:N0}' -f $_.Number};
         Label="Drive`n No.";Width=5;Align='Center'},
     @{Expression={$_.Model};Label="`nName";Width=$DNLen},
     @{Expression={$_.SerialNumber.Trim()};
         Label="`nSerial Number";Width=$SNLen;Align='left'},
     @{Expression={
        (&{If ($_.HealthStatus.GetType().Name -eq 'UInt16'){
         (& {Switch ($_.HealthStatus) {
              0      {"Healthy"}
              1      {"Warning"}
              2      {"Unhealthy"}
             Default {"Unknown"}}})}
           Else{$_.HealthStatus}})};Label="`nStatus";Width=7},
     @{Expression={$_.Speed};
         Label="Rotation`n  RPMs  ";Width=8;Align='Right'}

  $PhyDiskInfo1 = $PhyDiskInfo |
           Format-Table -Property $fmtPhyDisk1 -Wrap |
           Out-String   -Width $OStrWidth

  $PhyDiskInfo2 = $PhyDiskInfo |
           Format-Table -Property $fmtPhyDisk2 -Wrap |
           Out-String   -Width $OStrWidth

  $PhyDiskTitle  = "Physical Disk Information:" | Out-String

  Return ,$($PhyDiskTitle  + $PhyDiskInfo1 + $PhyDiskInfo2)

} #---------------- End Function Storage Tab ------------------

Function ProcessorTab {

  $CodeNames = @{2 = "Sandy Bridge"
                 3 = "Ivy Bridge"
                 4 = "Haswell"
                 5 = "Broadwell"
                 6 = "Sky Lake"
                 7 = "Kaby Lake"
                 8 = "Coffee Lake"
                 9 = "Ice Lake"
                10 = "Comet Lake"}

  $CPUTabStartCnt = $CITable.Rows.Count

  $CPU_Object =
     Get-CimInstance -ClassName  'Win32_Processor'

  Add-Row -ARIN 'Processor Name'  -ARIV $CPU_Object.Name

  $ErrorActionPreference = "SilentlyContinue"
    $Parts     = $CPU_Object.Name.Split("-")
    $Parts2    = $Parts[1].Split(" ")

  # $Parts2[0] = 10700   #Test Gen 10+
  # Strip letter designations U,K,L, etc.
    $Parts2[0] = $Parts2[0] -replace "[A-Z]",""

    If ($parts2[0].length -eq 4) {
      $Gen = $parts2[0].Substring(0,1)
    }
    Else {$Gen = $Parts2[0].Substring(0,2)}

  $ErrorActionPreference = "Continue"

  $ARArgs = @{
     ARIN = '  Code Name'
     ARIV = & {If($Parts.Count -lt 2){"Pre-Sandy Bridge"}
               Else {$CodeNames[[Int]$($Gen)]}
              }
             }
  Add-Row @ARArgs

  Add-Row -ARIN '  Info'          -ARIV $CPU_Object.Caption
  Add-Row -ARIN '  Maker'         -ARIV $CPU_Object.Manufacturer
  Add-Row -ARIN '  ID'            -ARIV $CPU_Object.ProcessorID

  $ARAgs = @{ARIN = '  Max CPU Speed'
             ARIV =  "$(($CPU_Object.MaxClockSpeed)/1000) GHz"}
  Add-Row @ARAgs

  $ARArgs= @{ARIN = '  Physical CPUs'
             ARIV = $CompSysObj.NumberofProcessors}
  Add-Row @ARArgs

  $ARArgs= @{ARIN = '  Physical Cores'
             ARIV = $CPU_Object.NumberofCores}
  Add-Row @ARArgs

  $ARArgs= @{ARIN = '  Logical  Cores'
             ARIV = $CPU_Object.NumberofLogicalProcessors}
  Add-Row @ARArgs

  $ARArgs= @{ARIN = '  Address Width'
             ARIV = $CPU_Object.AddressWidth}
  Add-Row @ARArgs

  $HyperThreadingStatus =
        if ($CPU_Object.NumberOfCores -le
            $CPU_Object.NumberofLogicalProcessors)
                      {'Enabled'} Else {'Disabled'}
  $ARArgs= @{ARIN = '  HyperThreading'
             ARIV = $HyperThreadingStatus}
  Add-Row @ARArgs

<#Note: Not reported correctly in 1809, 1903 or 1909 *********
  $VMFirmEnabled = $CPU_Object.VirtualizationFirmwareEnabled
#>

  $ARArgs= @{ARIN = '  VM Firmware'
             ARIV = Get-HyperVStatus -VirtualHW }
  Add-Row @ARArgs

  $ARArgs= @{ARIN = '  Socket'
             ARIV = "$($CPU_Object.SocketDesignation)"}
  Add-Row @ARArgs

  $CPUWidth = Get-MaxLength -TestObj $CITable.Item -MinLen 20

  $fmtCPU =
    @{Expression={$_.Item};Label="Item  ";Width=$CPUWidth},
    @{Expression={$_.Value};Label="Value";Width=78-$CPUWidth}

  $CPUInfo = $CITable.rows[$CPUTabStartCnt..(
                           $CITable.Rows.Count - 1)] |
            Format-Table -Property $fmtCPU | Out-String

  Return ,$CPUInfo

} # ------------------   End ProcessorTab  ---------------------

Function PartitionTab {

    $fmtPart =
       @{Expression={ '{0:N0}' -f ($_.Index)};
           Label="Part`nNo";Width=4;Align='right'},
       @{Expression={ '{0:#,000.00}' -f ($_.Size/1gb)};
           Label="Partition`nSize/GB";Width=9; Align='right'},
       @{Expression={$_.Alignment};
           Label="Part`nAlignment";Width=9; Align='left'},
       @{Expression={
            (&{IF ($_.PrimaryPartition) {"Yes"} else {""}})};
           Label="`nPrimary";Width=7; Align='Center'},
       @{Expression={
            (&{IF ($_.Bootable) {"Yes"} else {""}})};
           Label="`nBootable";Width=8; Align='Center'},
       @{Expression={
            (&{IF ($_.BootPartition) {"Yes"} else {""}})};
           Label="Boot`nPartition";Width=9; Align='Center'}

    $Partitions =
       Get-CimInstance -ClassName 'Win32_DiskPartition' |
       Sort-Object DiskIndex, Index

    $AMArgs = @{NotePropertyName  = 'Alignment'
                NotePropertyValue = ""}
    $Partitions | Add-Member @AMArgs

    ForEach ($part in $Partitions) {
        $Part.Alignment = (
          &{IF ($Part.StartingOffset % 4096 -eq 0)
             {"Aligned"} Else {"UnAligned"}})
    } #End ForEach ($part...

    $FTArgs = @{InputObject  = $Partitions
                Property     = $fmtPart
                GroupBy      = 'Diskindex'}

    $Partitions = Format-Table @FTArgs |
                  Out-String -Width $OStrWidth

  $PartInfoTitle = "Disk Partition Information:"  | Out-String
  $PartitionNote = @"

  Note: The Partition count shown in the Physical Disk Section Will
        be one more than shown below for GPT partitioned boot Drives!
"@ | Out-String

  Return ,$($PartInfoTitle + $PartitionNote + $Partitions)

}  #-----------------  End PartitionTab -----------------------

Function PowerShellDotNetTab {

  $PSTabStartCNt = $CITable.Rows.Count

  #--- PowerShell Information ---

  $PSVersInfo = $PSVersionTable
  $ExecPolicy = Get-ExecutionPolicy

  $ARArgs = @{ARIN = 'PowerShell Version'
              ARIV =  $PSVersInfo.PSVersion}
  Add-Row @ARArgs

  Try     { $Result = $PSVersInfo.PSEdition}
  Catch   { $Result =  "N/A" }
  Finally { $ARArgs = @{ARIN = 'PowerShell Edition'
                        ARIV = $Result}
           Add-Row @ARArgs
          }
  If ($Result -ne "Core") {
    $ARArgs = @{ARIN = 'Runtime Lang. Version'
                ARIV = "$($PSVersInfo.clrversion.Major)." +
                       "$($PSVersInfo.clrversion.Minor)"}
    Add-Row @ARArgs
  }

  $ARArgs = @{ARIN = 'WS-Man Stack  Version'
              ARIV = $PSVersInfo.WsManStackVersion}
  Add-Row @ARArgs
    $ARArgs = @{ARIN = 'PS Remoting   Version'
                ARIV =
       "$($PSVersInfo.PSRemotingProtocolVersion.Major)." +
       "$($PSVersInfo.PSRemotingProtocolVersion.Minor)"}
  Add-Row @ARArgs

  $ARArgs = @{ARIN = 'Execution Policy'
              ARIV = $ExecPolicy}
  Add-Row @ARArgs

 $GMArgs = @{TestObj =
      $CITable.Item[($PSTabStartCnt)..($CITable.Item.Count-1)]}

  $PSWidth = (Get-MaxLength @GMArgs ) + 3

  $fmtPS =
    @{Label="Item  ";Width=$PSWidth    ;Expression={$_.Item}},
    @{Label="Value" ;Width=100-$PSWidth;Expression={$_.Value}}

  $PSInfo =
    $citable.rows[($PSTabStartCnt)..($CITable.Rows.Count - 1)] |
    Format-Table -Property $fmtPS -Wrap |
    Out-String   -Width $OStrWidth

  $PSInfo = $($PSInfo.Substring(0,($PSInfo.Length-2)))

#-------------- .Net Information ----------------------

  $DotNetTitle = "Installed .Net Versions:" |
                 Out-String -Width $OStrWidth

  $DNArgs = @{Path =
        'HKLM:\SOFTWARE\Microsoft\NET Framework Setup\NDP'}
  $DotNet = Get-ChildItem @DNArgs -recurse |
            Get-ItemProperty -name Version,Release -EA 0        |
            Where-Object { $_.PSChildName -match '^(?!S)\p{L}'} |
            Sort-Object Version                                 |
            Select-Object PSChildName, Version, Release

  $NameLen =
     Get-MaxLength -TestObj $DotNet.PSChildName -MinLen 9
  $VerLen  =
     Get-MaxLength -TestObj $DotNet.Version     -MinLen 11
  $RNLen   =
     Get-MaxLength -TestObj $DotNet.Release     -MinLen 11

  $fmtDotNet =
      @{Label=".Net Type"  ;Width=$NameLen;
                                  Expression={$_.PSChildName}},
      @{Label="Version No:";Width=$VerLen;
                                  Expression={$_.Version}},
      @{Label="Release No:";Width=$RNLen;
                                  Expression={$_.Release}}

  $DotNet = $DotNet | Sort-Object Version,PSChildName |
            Format-Table $fmtDotNet | Out-String
  $DotNet = $($DotNet.Substring(0,($DotNet.Length-2)))

  Return ,$($PSInfo + $DotNetTitle + $DotNet)

} #--------------------- End PowerShellDotNetTab --------------

Function PrinterTab {

  $fmtPrt =
      @{Label="Printer";Width=35; Expression={$_.DeviceID  }},
      @{Label="Driver"; Width=25; Expression={$_.DriverName}},
      @{Label="Port";   Width=25; Expression={$_.PortName  }}

  $Printers =
     Get-CimInstance -ClassName 'Win32_Printer' |
     Select-Object   -Property DeviceID,DriverName, PortName |
     Sort-Object     -Property DeviceId |
     Format-Table    -Property $fmtPrt -Wrap |
     Out-String      -Width $OStrWidth

  Return ,$Printers

} #--------------------- End Printers Tab ---------------------

Function ProgramsTab {

  $fmtPgms = @{Label="Program Name";Width=40;
                               Expression={$_.DisplayName}},
              @{Label="Installed";Width=9;
                               Expression={$_.InstallDate}},
              @{Label="Install Path";Width=(100-51);
                               Expression={$_.InstallLocation}}

  If (!([Diagnostics.Process]::
         GetCurrentProcess().Path -match '\\syswow64\\'))
  {
    $unistallPath =
      "\SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall\"
    $unistallWow6432Path =
      "\SOFTWARE\Wow6432Node\Microsoft\Windows\" +
      "CurrentVersion\Uninstall\"

     $PgmsInfo =
      @(
        if (Test-Path   -Path "HKLM:$unistallWow6432Path") {
          Get-ChildItem -Path "HKLM:$unistallWow6432Path"  }
        if (Test-Path   -Path "HKLM:$unistallPath") {
          Get-ChildItem -Path "HKLM:$unistallPath"  }
        if (Test-Path   -Path "HKCU:$unistallWow6432Path") {
          Get-ChildItem -Path "HKCU:$unistallWow6432Path"  }
        if (Test-Path   -Path "HKCU:$unistallPath") {
          Get-ChildItem -Path "HKCU:$unistallPath"  }
       ) |
       Where-Object {
         $_.GetValue('DisplayName')     -and
        ($_.GetValue('UninstallString') -or
         $_.GetValue('NoRemove'))       -and
        !$_.GetValue('SystemComponent') -and
        !$_.GetValue('ReleaseType')     -and
        !$_.GetValue('ParentKeyName')
      } | Get-ItemProperty

  } #End If (!([Diagnostics.Process]...
  Else
  {
    $PgmsInfo =
      "You are running 32-bit Powershell " +
      "on 64-bit system.`n" +
      "Please run 64-bit Powershell instead."
  }

  #Add Member to Object
  $PgmsInfo |
    Add-Member -NotePropertyName BasePath -NotePropertyValue ""

  $ErrorActionPreference = "SilentlyContinue"

  $PgmsInfo | ForEach-Object {
     $Parts = $_.InstallLocation.Trim('"').Split('\')
     $_.BasePath = "$($Parts[0])\$($Parts[1])"
     $Temp = "\"
     For ($i = 2 ; $i -lt $Parts.Count ; $i++) {
        $Temp = $Temp + $Parts[$i] + "\"
     }
     $_.InstallLocation = $Temp.Replace('\\','\')
     $_.InstallDate = $_.InstallDate.Replace('/','')
  }

  $ErrorActionPreference = "Continue"

 $PgmsInfo = $PgmsInfo |
   Sort-Object  -Property BasePath, DisplayName |
   Format-Table -Property $fmtPgms -Wrap -GroupBy BasePath |
   Out-String   -Width $OStrWidth

  Return ,$PgmsInfo

} #-------------------   End Programs Tab    ------------------

Function GenRegSvcTab {

#--- formats declared as Script scope for reuse in SecRegSvcTab

  $Script:fmtRegistry =
      @{Expression={$_.Item}; Label="Registry Item";Width=40},
      @{Expression={$_.Value};Label="Setting"      ;Width=20}

  $Script:fmtSvcs = 
              @{Label="Service Name";Width=40;
                               Expression={$_.DisplayName}},
              @{Label="Status";Width=9;Align="Left";
                               Expression={$_.Status}},
              @{Label="Start Type";Width=(10);Align="Left";
                               Expression={$_.StartType}}

  $GeneralTabStartCnt = $CITable.Rows.Count

  $PropArgs =
    @{Path= 'HKLM:\SYSTEM\CurrentControlSet' +
            '\Control\Session Manager\Power'
     Name = 'HiberbootEnabled'
     ErrorAction = 'SilentlyContinue'}

    $Prop = Get-ItemProperty @PropArgs

  If ($Null -ne $Prop) {
    $ARArgs=
      @{ARIN = 'HiberbootEnabled'
        ARIV = ([Bool]$($Prop.HiberbootEnabled.Equals(1)))}
    Add-Row @ARArgs
  }

  $PropArgs =
      @{Path='HKLM:\SYSTEM\CurrentControlSet\Control\Power'
        Name = 'HibernateEnabled'
        ErrorAction = 'SilentlyCOntinue'}

  $Prop = Get-ItemProperty @PropArgs

  If ($Null -ne $Prop) {
    $ARArgs=
      @{ARIN = 'HibernateEnabled'
        ARIV = ([Bool]$($Prop.HibernateEnabled.Equals(1)))}
    Add-Row @ARArgs
  }

  $ServList = @("TeamViewer*","USBDLM*","Macrium*","Remote*",
                 "MsKey*","VSS","Wsearch","DiagTrack","SysMain")

  $GSArgs = @{Name        = $ServList
             ErrorAction = "SilentlyContinue"}

  $General1 = $CITable.Rows[$GeneralTabStartCnt..$(
                            $CITable.Rows.Count - 1)] |
              Format-Table  -Property $fmtRegistry    |
              Out-String -Width $OStrWidth

  $General2 = Get-Service @GSArgs               |
              Sort-Object -Property DisplayName |
              Format-Table $fmtSvcs             |
              Out-String -Width $OStrWidth
  
  Return, $($($General1.Substring(0,($General1.Length-3)))  + 
            $($General2.Substring(0,($General2.Length-2))))

}  #----------- End GenRecSvcTab -----------------------------

Function SecRegSvcTab {

  $SecurityTabStartCnt = $CITable.Rows.Count

  $UACArgs = @{Name = 'EnableLUA'
               Path = 'HKLM:\SOFTWARE\Microsoft\Windows\' +
                      'CurrentVersion\Policies\System'
                               }
  $UAC = Get-ItemProperty @UACArgs

  $ARArgs= @{ARIN = 'UAC Enabled'
             ARIV = ($UAC.EnableLUA.Equals(1))}
  Add-Row @ARArgs

  $ServList = @("MBAM*","WinDefend*","MpsSvc","wuauserv",
                "BDESVC","fhsvc")

  $GSArgs = @{Name        = $ServList
             ErrorAction = "SilentlyContinue"}

  $Security1 = $CITable.Rows[$SecurityTabStartCnt..$(
                             $CITable.Rows.Count - 1)] |
               Format-Table  -Property $fmtRegistry    |
               Out-String -Width $OStrWidth

  $Security2 = Get-Service @GSArgs               |
               Sort-Object -Property DisplayName |
               Format-Table $fmtSvcs             |
               Out-String -Width $OStrWidth

  Return, $($($Security1.Substring(0,($Security1.Length-3)))  + 
            $($Security2.Substring(0,($Security2.Length-3))))

} #----------------------- End SecRegSvc Tab ------------------

Function SecurityTab {

  $SecTabStartCNt = $CITable.Rows.Count

  Add-Row -ARIN 'Security Software'

  $FWKey = "HKLM:\System\CurrentControlSet\Services\" +
           "SharedAccess\Parameters\FirewallPolicy\"  +
           "StandardProfile"
  $FWStatus = Get-Item $FWKey
  $FWStatus = $FWStatus.GetValue("EnableFireWall")
  If ($Null -eq $FWStatus) {
    $FireWallStatus = "Not Using Windows Firewall"
  }
  Else {
     $FireWallStatus =
       (&{If ([Int]$FwStatus -eq 1) {"On"}
          Else {"Off"}})
  } #End $FWStatus -eq $Null

  Add-Row -ARIN "  Firewall"   -ARIV $FireWallStatus

  $AV = Get-AntiVirual -CompName $CompName

  If ($AV -is [System.Array]) {

    For ($AVCnt =0 ; $AVCnt -lt $AV.Count; $AVCnt++) {

       $ARArgs= @{ARIN = "  #$($($AVCnt)+1) AV Program"
                  ARIV = $($AV[$($AVCnt)].displayName)}
       Add-Row @ARArgs

       $ARArgs= @{ARIN = "  #$($($AVCnt)+1) AV State"
                  ARIV = $(Get-AVStatusText (
                           $AV[$($AVCnt)].ProductState))}
       Add-Row @ARArgs

    } # For...

  } #End if [system.array]

  Else {

       $ARArgs= @{ARIN = "  AV Program"
                  ARIV = $AV.displayName}
       Add-Row @ARArgs

       $ARArgs= @{ARIN = "  AV State"
                  ARIV = $(Get-AVStatusText $AV.ProductState)}
       Add-Row @ARArgs

  } #Else

  IF ($AV.DisplayName -eq "Windows Defender") {

    $ARArgs= @{ARIN = "  Defender Sandboxed"
               ARIV = $False}

    Try {
         $SandBoxed =  Get-Process -Name MsMpEngCP -EA Stop
    }
    Catch { $SandBoxed = $Null }
    Finally {
             If ($Null -ne $SandBoxed) {
               $ARArgs.ARIV = $True
             }
             Add-Row @ARArgs
    }

  } # End If ($AV...

 $GMArgs = @{TestObj =
      $CITable.Item[($SecTabStartCnt)..($CITable.Item.Count-1)]}

  $SecWidth = (Get-MaxLength @GMArgs ) + 3

  $fmtSec =
    @{Label="Item  ";Width=$SecWidth    ;Expression={$_.Item}},
    @{Label="Value" ;Width=100-$SecWidth;Expression={$_.Value}}

  $SecurityInfo =
   $citable.rows[($SecTabStartCnt)..($CITable.Rows.Count - 1)] |
     Format-Table -Property $fmtSec -Wrap                      |
     Out-String   -Width $OStrWidth

  Return ,$SecurityInfo

} #---------------------- End Security Tab ---------------------

Function Show-BalloonTip  {

  param
  (
    [Parameter(Mandatory=$true)]
       [String] $Text,
    [Parameter(Mandatory=$True)]
       [String] $Title,
    [Parameter(Mandatory=$True)]
    [ValidateSet('None', 'Info', 'Warning', 'Error')]
       [String] $Icon = 'Info',
    [Parameter(Mandatory=$False)]
       [Int32] $Timeout = [Int32]2000
  )

  Add-Type -AssemblyName System.Windows.Forms

  $GVArgs = @{Name        = "Balloon"
              ErrorAction = "SilentlyContinue"}
  If (-not (Get-Variable @GVArgs))
  {
   $script:balloon = New-Object System.Windows.Forms.NotifyIcon
  }

  $path                    = Get-Process -id $pid |
                             Select-Object -ExpandProperty Path
  $balloon.Icon            =
       [System.Drawing.Icon]::ExtractAssociatedIcon($path)
  $balloon.BalloonTipIcon  = $Icon
  $balloon.BalloonTipText  = $Text
  $balloon.BalloonTipTitle = $Title
  $balloon.Visible         = $true

  $balloon.ShowBalloonTip($Timeout)

} #----------- End Function Show-BalloonTip -------------------

Function Test-BatteryHealth {

  $CIMArgs = @{Class        = "BatteryFullChargedCapacity"
               Namespace    = 'ROOT\WMI'}
  $fullchargecapacity =
      (Get-CIMInstance @CIMArgs).FullChargedCapacity

  $CIMArgs = @{ClassName = "CIM_Battery"}
#--- Test for Array and adjust ----
  $designcapacity = (Get-CIMInstance @CIMArgs)[0].DesignCapacity

  if ($fullchargecapacity -eq $designcapacity) {
      $Batteryhealth=-1
  }
  Else {
        If ($Null -eq $designcapacity -or
            $Null -eq $fullchargecapacity -le 0) {$batteryhealth = 0}
        Else {
            $batteryhealth =
                ($fullchargecapacity / $designcapacity) * 100
            if ($batteryhealth -gt 100) {$batteryhealth = 100}
        }
  }

  return ,[decimal]::round($batteryhealth)

} #----------------  End Test-BatteryHealth -------------------

Function UsersTab {

  $CurUser =  "`nCurrently Logged On User: " +
                ($($CompSysObj.username).Split("\"))[1] +
              "`nCurrent User Profile    : " +
                ((Get-ChildItem -Path Env:UserProfile).Value) |
              Out-String



  #--- User Account Information ---

  $uai    = Get-UserAccount -ComputerName $CompName

  $UIDLen = Get-MaxLength -TestObj $uai.Name   -MinLen 6
  $WGLen  = Get-MaxLength -TestObj $uai.Domain -MinLen 9

  $fmtAcct =
     @{Label="`nUserID";Width=$UIDLen;Align='left';
               Expression={$_.Name}},
     @{Label="`nDomain";Width=$WGLen;
               Expression={$_.Domain}},
     @{Label="`nStatus";Width=8;Align='left';
               Expression={If ($_.Disabled) {'Disabled'}
                                       Else {'Enabled'}}},
     @{Label="|----P A`nRequired";Width=8;Align='left';
               Expression={If ($_.PasswordRequired) {'   Yes'}
                           Else {'   No'}}},
     @{Label="S S W O`nExpires";    Align='left';Width=7;
               Expression={If ($_.PasswordExpires) {'  Yes'}
                           Else {'  No'}}},
     @{Label="R D S---|`nChangable";Width=9;Align='left';
               Expression={If ($_.PasswordChangeable) {'   Yes'}
                           Else {'   No'}}}

  $UATitle = "User Account Info:" | Out-String
  $uai     = $uai | Format-Table -Property $fmtAcct |
                    Out-String   -Width $OStrWidth

#------- Show location of File Explorer Shell Folders -------

  $ShellFoldLoc = [Ordered]@{}

  $Folders = @("MyDocuments","MyMusic","MyPictures","MyVideo")

  ForEach ($Folder in $Folders) {

     $ShellFoldLoc +=
       @{$Folder = $([Environment]::getfolderpath($Folder))}

  } #End ForEach ($Folder...



  $fmtShellFolds =
      @{Expression={$_.Name};
         Label="      Shell`nName";Width=11},
      @{Expression={$_.Value};
         Label="Folders`nLocation";Width=55}

  $ShellFolds = $ShellFoldLoc                         |
                Format-Table -Property $fmtShellFolds |
                Out-String

  Return ,$($UATitle + $CurUser + $uai + $ShellFolds)

} #-------------  End Function UsersTab  ----------------------

Function ValidateWinVer {

  Param (
         [Parameter (Mandatory=$True)]
           [Int] $MajorVer,
         [Parameter (Mandatory=$False)]
           [Int] $MinorVer = 0
  )

  $ClrVers = @{"10.0" = "10";
                "6.3" = "8.1";
                "6.2" = "8.0";
                "6.1" = "7";
                "6.0" = "Vista"}

  $CurOS   = Get-CimInstance -Class Win32_OperatingSystem
  $VersionItems = $CurOS.Version.split('.')

  #Set Global ThisPCWinVer Variable for later reference
  $SVArgs = @{Name = "ThisPCWinVer"
              Scope = "Global"
              Value =
   [Single]$ClrVers["$($VersionItems[0]).$($VersionItems[1])"]}
  Set-Variable @SVArgs

  If (([int]$VersionItems[0] -lt $MajorVer) -or
     (([int]$VersionItems[0] -eq $MajorVer) -and
      ([int]$VersionItems[1] -lt $MinorVer))) {
    $Message = "You are running Windows " +
      $ClrVers["$($VersionItems[0]).$($VersionItems[1])"] +
      "`nthe program requires at least Windows " +
      $CLrVers["$MajorVer.$MinorVer"] +" to run."

     [Windows.Forms.MessageBox]::Show($Message,
                                      "Program Terminated:",
     [Windows.Forms.MessageBoxButtons]::OK ,
     [Windows.Forms.MessageBoxIcon]::Information)

    Return ,$False

  } #End If

  Else {
    Return "OK",$True   #Mimic MsgBox OK button!
  }

} #------------------- End ValidateWinVer ---------------------

Function VideoTab {

<#
  Note: If you have not run installation software for your
        monitor or if you have dual monitors and can't install
        the second one you can manually edit some of the
        information using a System Level Registry Editor
        session at the key:
        HKLM:\System\CurrentControlSet\Enum\Display\
        I successfully edited DeviceDesc and Mfg values.
#>

  $VideoArchitecture = @{
     1 = 'Other'
     2 = 'Unknown'
     3 = 'CGA'
     4 = 'EGA'
     5 = 'VGA'
     6 = 'SVGA'
     7 = 'MDA'
     8 = 'HGC'
     9 = 'MCGA'
    10 = '8514A'
    11 = 'XGA'
    12 = 'Linear Frame Buffer'
    13 = 'PC-98'
   }
  $VideoMemoryType = @{
     1= 'Other'
     2 = 'Unknown'
     3 = 'VRAM'
     4 = 'DRAM'
     5 = 'SRAM'
     6 = 'WRAM'
     7 = 'EDO RAM'
     8 = 'Burst Synchronous DRAM'
     9 = 'Pipelined Burst SRAM'
    10 = 'CDRAM'
    11 = '3DRAM'
    12 = 'SDRAM'
    13 = 'SGRAM'
  }

  $CurrentScanMode = @{
     1 = 'Other'
     2 = 'Unknown'
     3 = 'Interlaced'
     4 = 'Non Interlaced'
    }

  $GCIArgs = @{Class      = 'Win32_VideoController'
               Namespace  = "root\CIMV2"}
  $Video = Get-CimInstance @GCIArgs

  $VideoTabStartCnt = $CITable.Rows.Count

  ForEach ($VidAdp in $Video) {

    Add-Row -ARIN 'Video Name' -ARIV $VidAdp.Name
    Add-Row -ARIN '  RAM (Mb)' -ARIV (
                     ($VidAdp.AdapterRAM/1mb))

    $ARArgs= @{ARIN = '  RAM Type'
               ARIV = $($VideoMemoryType[
                         [int]$VidAdp.VideoMemoryType])}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Architecture'
               ARIV = $($VideoArchitecture[
                         [int]$VidAdp.VideoArchitecture])}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Current Horiz. Resolution'
               ARIV = $VidAdp.CurrentHorizontalResolution}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Current Vert.  Resolution'
               ARIV = $VidAdp.CurrentVerticalResolution}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Current Bits Per Pixel'
               ARIV = $VidAdp.CurrentBitsPerPixel}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Refresh Rate'
               ARIV = $VidAdp.CurrentRefreshRate}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Current Scan Mode'
               ARIV = $CurrentScanMode[
                       [int]$VidAdp.CurrentScanMode]}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Installed Drivers'
               ARIV = $VidAdp.InstalledDisplayDrivers}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Driver Date'
               ARIV = $vidAdp.DriverDate.ToShortDateString()}
    Add-Row @ARArgs

    $ARArgs= @{ARIN = '  Driver Version'
               ARIV = $VidAdp.DriverVersion}
    Add-Row @ARArgs

  } # End ForEach ($Vid...


  $TOArg = @{TestObj =
    $CITable.Item[$VideoTabStartCNt..($CITable.Item.Count-1)]}
  $VidWidth = (Get-MaxLength @TOArg) + 3 #Allow for indent

  $fmtVid = @{Expression={$_.Item};
                Label="Item  ";Width=$VidWidth},
            @{Expression={$_.Value};
                Label="Value";Width=($OStrWidth-($VidWidth+3))}

  $VideoHW    =
     $citable.rows[$VideoTabStartCnt..(
                           $CITable.Rows.Count - 1)] |
     Format-Table -Property $fmtVid -Wrap |
     Out-String   -Width $OStrWidth

  Return ,$VideoHW

} #----------------------- End Video Tab ----------------------

Function WindowsDefender {

  $DefenderTabStartCnt = $CITable.Rows.Count

  Try
  {
   $DefenderOptions = Get-MpComputerStatus -ErrorAction Stop

   If([string]::IsNullOrEmpty($DefenderOptions))
   {
       $DefenderActive = $False
    ARArgs = @{
       ARIN = "Windows Defender was not found running on:"
       ARIV =  $env:computername
    }
    Add-Row @ARArgs

   } #End If ([strin....

   Else {
      $DefenderActive = $True

      $ARArgs = @{
        ARIN = "Windows Defender was found on:"
        ARIV = $env:computername
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Defender:"
        ARIV = (& {IF($($DefenderOptions.AntivirusEnabled) -eq 1)
                    {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Defender Service:"
        ARIV = (& {IF($($DefenderOptions.AMServiceEnabled) -eq 1)
                    {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Antispyware:"
        ARIV = (& {
         IF($($DefenderOptions.AntispywareEnabled) -eq 1)
                    {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "On Access Protection:"
        ARIV = (&{
         IF($($DefenderOptions.OnAccessProtectionEnabled) -eq 1)
                    {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Real Time Protection:"
        ARIV = (&{
         IF($($DefenderOptions.RealTimeProtectionEnabled) -eq 1)
                   {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Virus Definition Last Update:"
        ARIV = $($DefenderOptions.AntiVirusSignatureLastUpdated)
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Virus Signature Version:"
        ARIV = $($DefenderOptions.AntiVirusSignatureVersion)
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Behavior Monitor Enabled:"
        ARIV = (&{
         IF($($DefenderOptions.BehaviorMonitorEnabled) -eq 1)
                   {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

   } #End Else

  If ($DefenderActive) {  $WinDefender = Get-MpPreference

      $ARArgs = @{
         ARIN = "PUA/PUP Protection:"
         ARIV =  (&{IF($($WinDefender.PUAProtection) -eq 1)
               {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
         ARIN = "Controlled Folder Access:"
         ARIV = (&{
          IF($($WinDefender.EnableControlledFolderAccess) -eq 1)
               {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Sample Submission Consent:"
        ARIV = (&{IF($($WinDefender.SubmitSamplesConsent) -eq 1)
               {"Enabled"} Else{"Disabled"}})
      }
      Add-Row @ARArgs

      $ARArgs = @{
        ARIN = "Exclusion Path:"
        ARIV =  (&{IF($Null -ne $($WinDefender.ExclusionPath))
                  {"$($WinDefender.ExclusionPath)"}
              Else{"No Exclusion Path Set."}})
      }
      Add-Row @ARArgs

    $TOArg = @{TestObj =
      $CITable.Item[$DefenderTabStartCNt..(
                    $CITable.Item.Count-1)]}
    $DefWidth = (Get-MaxLength @TOArg) + 3

    $fmtDefender =
      @{Expression={$_.Item};Label="Item  ";Width=$DefWidth},
      @{Expression={$_.Value};Label="Value";Width=78-$DefWidth}

    $DefenderInfo =
       $citable.rows[$DefenderTabStartCnt..(
                     $CITable.Rows.Count - 1)]   |
       Format-Table -Property $fmtDefender -Wrap |
       Out-String   -Width $OStrWidth

    } #End If ($DefenderActive)

  }     #End Try

  Catch {
       $DefenderInfo   = $Null
       $DefenderActive = $False
  }

 Return ,$DefenderInfo

} #-----------  End Function WindowsDefender  -----------------

Function WindowsTab {

  $DomainRoles = @{
    0="Standalone Workstation"
    1="Member Workstation"
    2="Standalone Server"
    3="Member Server"
    4="Backup Domain Controller"
    5="Primary Domain Controller"
  }

  $CurOS = Get-CimInstance -ClassName 'Win32_OperatingSystem'
  $OSKey = Get-WindowsProductKey $CompSysObj.name

  $CIMArgs = @{ClassName = 'SoftwareLicensingService'
               ErrorAction = 'Stop'}
  Try { $EmbWinKey =
         (Get-CimInstance @CIMArgs).OA3xOriginalProductKey
      }
  Catch {$EmbWinKey = "N/A"}

  $LocalDateTime  = $CurOS.LocalDateTime

  $GWEArgs =
       @{ProviderName = 'Microsoft-Windows-Kernel-Boot'
         MaxEvents    = 100
         ErrorAction  = 'Stop'}
  $FilterID = "27"

  Try {

    $Events = Get-WinEvent @GWEArgs
    $Events = $Events |
       Where-Object {$_.id -like "$FilterID"} |
       Select-Object -First 1

#If next stmt errors increase MaxEvents above!
    $LastBootupTime = $Events.TimeCreated

    $PreUpTime      =  $LocalDateTime - $LastBootupTime
    If ($PreUptime.Days -gt 0) {
      $UpTime = "{00:dd} Day(s) {00:hh}:{00:mm}" -f $PreUptime
    }
    Else {
      $UpTime = "{00:hh}:{00:mm}" -f $PreUptime
    }

    $LastBootStr =
     (Get-Date ($Events.TimeCreated) -format $LocalDTFmt) + " " +
     (Get-Date ($Events.TimeCreated) -format $LocalTMFmt)

  } #End Try
  Catch {
   $UpTime      = "Requires Administrative Access"
   $LastBootStr = "Requires Administrative Access"
  }

  $GIPArgs = @{Path = "Registry::HKLM\SOFTWARE\Microsoft\" +
                      "Windows NT\CurrentVersion"
              }
  $WinVer = (Get-Item @GIPArgs).GetValue("UBR")

  $OSTabStartCNt = $CITable.Rows.Count
  Add-Row -ARIN 'OS Name'         -ARIV $CurOS.Caption
  Add-Row -ARIN '   Version'      -ARIV $($CurOS.Version +
                                          "." + $winVer)

  $GIPArgs = @{Path = "HKLM:\SOFTWARE\Microsoft\Windows NT" +
                      "\CurrentVersion"
               Name = "ReleaseId"}
  $ARArgs  = @{ARIN = '   Release ID'
               ARIV = (Get-ItemProperty @GIPArgs).ReleaseId}
  Add-Row @ARArgs

  Add-Row -ARIN '   Bit-Width'    -ARIV $CurOS.OSArchitecture
  Add-Row -ARIN '   OEM/User Key' -ARIV $OSKey
  Add-Row -ARIN '   Embedded Key' -ARIV $EmbWinKey

  $GCIArgs = @{ClassName = "SoftwareLicensingProduct"
               Filter    = "PartialProductKey IS NOT NULL"}
  $Activation = Get-CimInstance @GCIArgs  |
    Where-Object -Property Name -Like "Windows*"
  $ARArgs = @{ARIN = '   Activated'
              ARIV = &{If ($Activation.LicenseStatus -eq 1) {
                       $True} else {$False}
                      }
             }
  Add-Row @ARArgs

  $ARArgs = @{ARIN = '   Channel'
              ARIV = $($Activation.ProductKeyChannel)
             }
  Add-Row @ARArgs

  $ARArgs = @{ARIN = '   Registered Owner'
              ARIV = $CurOs.RegisteredUser}
  Add-Row @ARArgs

  If ( ((Test-Path variable:CurOS.RegisteredOrganization) -eq
        $False) -or ($CurOS.Organization -eq "")) {

    $GIPArgs = @{Path =
        "HKLM:\Software\Microsoft\Windows NT\CurrentVersion\"
                 Name        = 'RegisteredOrganization'
                 ErrorAction = 'SilentlyContinue'}
    $RO = Get-ItemProperty @GIPArgs

    $RegOrg = $RO.RegisteredOrganization
  }
  Else { $RegOrg = $CurOS.RegisteredOrganization }

  $ARArgs = @{ARIN = '   Registered Org'
              ARIV = $RegOrg}
  Add-Row @ARArgs

  $ARArgs = @{ARIN = '   System Root'
              ARIV = (Get-ChildItem -Path Env:SystemRoot).Value}
  Add-Row @ARArgs

  $ARArgs =
    @{ARIN = '   Install Date'
      ARIV =
        (Get-Date ($CurOS.InstallDate) -format $LocalDTFmt) +
        " " +
        (Get-Date ($CurOS.InstallDate) -format $LocalTMFmt)
     }
  Add-Row @ARArgs

  $ARArgs =
    @{ARIN = '   Date/Time'
      ARIV =
       (Get-Date $LocalDateTime -format $LocalDTFmt) + " " +
       (Get-Date $LocalDateTime -format $LocalTMFmt)
     }
  Add-Row @ARArgs

  $ARArgs =
    @{ARIN = '   Time Zone'
      ARIV = "GMT $($CompSysObj.CurrentTimeZone / 60) " +
             (&{If ($CompSysObj.DaylightInEffect) {
                      'Daylight savings time.'} else {
                      'Standard time'}})
     }
  Add-Row @ARArgs

  If ($Events[0].ID -eq 27) {
    $BootType = @("Cold boot from full shutdown",
                "hybrid boot (fast startup)",
                "Resume from hibernation")
    $LastBootStr +=  " " +
      $($BootType[$Events[0].Message.Substring(20,1)])
  }

  $ARArgs = @{ARIN = '   Last Boot'
              ARIV = "$LastBootStr"}
  Add-Row @ARArgs

  $ARArgs = @{ARIN = '   Up Time'
              ARIV = $UpTime }
  Add-Row @ARArgs

  $OSCompacted = & { Compact.exe /CompactOS:Query }

  $OSCompacted = (& {
    If (($OSCompacted -like "*system is in the Compact state*"
        ).count -eq 1) {"True"}
    Else {"False"}})

  Add-Row -ARIN '   Compacted'  -ARIV $OSCompacted
  Add-Row -ARIN ' '
  Add-Row -ARIN 'Computer Name' -ARIV $CompSysObj.Name

  $ARArgs = @{ARIN = (&{If ($CompSysObj.PartOfDomain) {
                      'Domain Name'} else {
                      'Workgroup Name'}})
              ARIV = $CompSysObj.Domain}
  Add-Row @ARArgs

  If ($CompSysObj.PartOfDomain) {
    $ARArgs = @{ARIN = 'Domain Role'
                ARIV = $DomainRoles[$CompSysObj.DomainRole]}
    Add-Row @ARArgs
  }

  $GMArgs = @{TestObj =
      $CITable.Item[($OSTabStartCnt)..($CITable.Item.Count-1)]}

  $WinWidth = (Get-MaxLength @GMArgs ) + 3

  $fmtWin =
    @{Label="Item  ";Width=$WinWidth    ;Expression={$_.Item}},
    @{Label="Value" ;Width=100-$WinWidth;Expression={$_.Value}}

  $WinInfo =
    $citable.rows[($OSTabStartCnt)..($CITable.Rows.Count - 1)] |
    Format-Table -Property $fmtWin -Wrap |
    Out-String   -Width $OStrWidth

  Return ,$WinInfo

} #---------------------- End Windows Tab ---------------------

Function WinUpdSettings {

<#
.Synopsis
  Print the Microsoft auto-update settings for the local computer
  and list pending & hidden updates.

  Copyright (c) 2018 by MCB Systems. All rights reserved.
  Free for personal or commercial use.  May not be sold.
  No warranties.  Use at your own risk.

.Notes
    Name:       MCB.WindowsUpdate.ShowSettings.ps1
    Author:     Mark Berry, MCB Systems
    Created:    10/03/2014
    Last Edit:  04/01/2020 - RetiredGeek

    Adapted from:
     http://fixitscripts.com/problems/windows-update-settings.
    Also see:
     https://technet.microsoft.com/en-us/library/ee692834.aspx
     http://powershellautomation.blogspot.com/2013/05/
           list-installed-pending-updates.html

    Changes:

    10/02/2018 - Cleaned up output (prevent errors) and code to
                 make easier to read.
                 RetiredGeek@AskWoody.com

    04/01/2010 - Replaced Pending & Hidden Updates code using
                 Get-WindowsUpdate from PSWindowsUpdateModule.
                 Stripped Update totals & Parameters.
                 RetiredGeek@AskWoody.com

#>


  function ListRegKeyValues {
    param(
      [Parameter(Mandatory=$true)]
      [string]$path)

    $RegStr = "Hive: $path`n"

  	Try {
       $RegKey = (Get-ItemProperty $path -EA Stop)
       $RegKey.PSObject.Properties | ForEach-Object {
     	  $Name  = $_.Name
     	  $Value = $_.Value
     	  switch ($Name) {
     		  # Don't print the PowerShell properties that are
               # apparently present in every PSObject
     		    "PSPath"       { break }
     			"PSParentPath" { break }
     			"PSChildName"  { break }
     			"PSDrive"      { break }
     			"PSProvider"   { break }
     			default {$RegStr +=  "$Name : " +
                         (&{If ($Value -eq 1) {"True"}
                            Else {"False"}}) + "`n"}
         } #End Switch
        }  #ForEach-Object
     } #End Try

     Catch {$RegStr += " - Not Set -"}

    "$RegStr"

  }  #End Function ListRegKeyValues

  #---------------  Main Function WinUpdSettings -------------

  $TStr = @"
$(Get-MSDefCon)

Microsoft AutoUpdate settings (from the Microsoft.Update COM object)
--------------------------------------------------------------------
"@ | Out-String

    $objAutoUpdateSettings =
    (New-Object -ComObject "Microsoft.Update.AutoUpdate").Settings
    $objSysInfo =
      New-Object -ComObject "Microsoft.Update.SystemInfo"

  	# See https://stackoverflow.com/a/32253197/550712 for the
      # Out-String.Trim() trick to trim blank lines
     $TStr += ($objAutoUpdateSettings | Out-String).Trim()
     $TStr += "`nReboot required           : " +
              "$($objSysInfo.RebootRequired)" | Out-String

    # Static info on the meaning of various Settings.
  $TStr += @"

    Key to AutoUpdate Settings
    --------------------------
    NotIficationLevel:
    1 - Never check for updates
    2 - Check for updates but let me choose whether to download
        and install them
    3 - Download updates but let me choose whether to install them
    4 - Install updates automatically

    ScheduledInstallationDay:
      0 - Every day
    1-7 - Sunday through Saturday

"@ | Out-String

      $path1 = "HKLM:\Software\Policies\Microsoft\Windows" +
               "\WindowsUpdate"
      $Key1  =  ListRegKeyValues "$path1"
      $Key2  =  ListRegKeyValues "$path1\AU"

  $TStr += @"

Registry values (set by Group Policy or manually)
-------------------------------------------------
$key1
$Key2
"@ | Out-String

  <#
     WSUS server info, If available
     Reference: https://technet.microsoft.com/en-us/library/
                        dd939844(v=ws.10).aspx
  #>

    $Title1 = "WSUS Server       : "
    $Title2 ="`n`nWSUS Status Server: "

    Try {
      <#
        If Get-ItemProperty fails, value is not in regisTry.
        Do not fail entire script.  -ErrorAction Stop" forces it
        to Catch even a non-terminating error.
      #>

      $GIPArgs = @{Path        = "HKLM:\Software\Policies\" +
                                 "Microsoft\Windows\WindowsUpdate"
                   Name        = "WUServer"
                   ErrorAction = "Stop"}
      $WsusStr1 = (Get-ItemProperty @GIPArgs).WuServer
    }
    Catch {
      $WsusStr1 =
      "WSUS not configured.`n`t`t`tThe machine" +
      " must be contacting Windows Update directly."
    }


    Try {
      $GIPArgs = @{Path        = "HKLM:\Software\Policies\" +
                                 "Microsoft\Windows\WindowsUpdate"
                   Name        = "WUStatusServer"
                   ErrorAction = "Stop"}
      $WsusStr2 = (Get-ItemProperty @GIPArgs).WUStatusServer
    }
    Catch {
      $WsusStr2 = "WSUS not configured.`n`t`t`tThe machine " +
      "must be contacting Windows Update directly."
    }

    $TStr += $Title1 + $WsusStr1 +
             $Title2 + $WsusStr2 | Out-String

   $fmtUpds =
    @{Expression={&{
    Switch ($_.MsrcSeverity) {
          1 {"Critical"}
          2 {"Important"}
          3 {"Moderate"}
          4 {"Low"}
          5 {"Unspecified"}
          Default {"Other"}
    }}};Label="`nSeverity";Width=9},
    @{Expression={"{0:N1} MB" -f ($_.Size / 1MB)};
                  Label="`n    Size";
                  Width=8; Align="Right"},
    @{Expression={$_.BrowseOnly};Label="Browse`nOnly";
                  Width=6; Align="Left"},
    @{Expression={$_.IsDownLoaded};Label="Down`nloaded";
                  Width=6; Align="Left"},
    @{Expression={$_.RebootRequired};Label="Reboot`nReq.";
                  Width=6; Align="Left"},
    @{Expression={$_.Title};Label="`nDescription";Align="Left"}

  Try {
       $PU = get-windowsUpdate -ErrorAction Stop   |
             Where-Object {$_.IsHidden -eq $False} |
             Format-Table -Property $fmtUpds -Wrap |
             Out-String

       If ($PU -eq "") { $PU = "`tNo Pending Updates Found!" |
         Out-String}
  }
  Catch {$PU = "`n`tRequires Administrative privleges to report"}

  Try {
       $HU = get-windowsUpdate -ErrorAction Stop   |
             Where-Object {$_.IsHidden -eq $True}  |
             Format-Table -Property $fmtUpds -Wrap |
             Out-String

       If ($HU -eq "") { $HU = "`tNo Hidden Updates Found!`n" |
         Out-String}
  }
  Catch {$HU =
           "`n`tRequires Administrative privleges to report`n"}

  $TStr += ("`nPending Updates:" | Out-String) +
             ($PU | Out-String) +
           ("`nHidden Updates: " | Out-String) +
             ($HU | Out-String)

  Return ,$TStr

} #-----------  End WinUpdSettings  ----------------------

Function Write-ComputerInfoFile {

  $LocaleDate   = (Get-Date -f "$LocalDTFmt").Replace("/","-")

  $OSDT = "Computer Mentor's Local System Information"  +
          "`nVersion W10 Only " +
          "{0:00.00} as of: $LocaleDate" -f $PGMVersion +
      "`nRun Time to menu display: {0:mm}:{0:ss}" -f $RunTime |
                                             Out-String
   $OSHW = "`n`n---     Hardware      ---" | Out-String
   $OSVD = "---       Video       ---"     | Out-String
   $OSMO = "---     Monitors      ---"     | Out-String
   $OSAO = "---       Audio       ---"     | Out-String
   $OSST = "---      Storage      ---`n"   | Out-String
   $OSNW = "---      Network      ---`n"   | Out-String
   $OSPR = "---     Printers      ---"     | Out-String
   $OSUR = "---       Users       ---`n"   | Out-String
   $OSWI = "---      Windows      ---"     | Out-String
   $OSAV = "--- Windows Security  ---"     | Out-String
   $OSPS = "--- Powershell & .Net ---"     | Out-String
   $OSWU = "---   Windows Update  ---`n"   | Out-String
   $OSUD = "---   Recent Updates  ---"     | Out-String
   $OSUG = "---  Windows Upgrades ---"     | Out-String
   $OSWD = "---  Windows Defender ---"     | Out-String
   $OSEN = "---    Environment    ---"     | Out-String
   $OSDR = "---      Drivers      ---`n"   | Out-String
   $OSPI = "--- Installed Programs ---"    | Out-String
   $OSPI = $($OSPI.Substring(0,$OSPI.Length-3))
   $OSGI = "--- Registry/Services ---`n`n" +
           "`t`t[ General Items ]"         | Out-String
   $OSSI = "`t`t[ Security Items ]"        | Out-String

   $OSDT +
   $OSHW + $HardwareInfo        +
           $ProcessorInfo       +
           $MemoryInfo          +
   $OSVD + $VideoInfo           +
   $OSMo + $MonitorInfo         +
   $OSAO + $AudioInfo           +
   $OSST + $PhysicalDiskInfo    +
           $PartitionInfo       +
           $LogicalDiskInfo     +
           $CDDVDInfo           +
           $FileSystemInfo      +
           $MapsAndShares       +
           $EncryptInfo         +
   $OSNW + $InternetInfo        +
           $ActiveAdapterInfo   +
           $WiFiInfo            +
           $LocalNetworkInfo    +
           $NetAdaptersInfo     +
   $OSPR + $PrinterInfo         +
   $OSWI + $WindowsInfo         +
   $OSUR + $UserInfo            +
   $OSAV + $SecurityInfo        +
   $OSPS + $PowerShellInfo      +
   $OSWU + $WindowsUpd          +
   $OSUD + "$WinUpdateHistory"  +
   $OSUG + "$WinUpgradeHistory" +
   $OSEN + $EnvironInfo         +
   (&{If($Null -ne $WinDefInfo) {($OSWD + $WinDefInfo)}}) +
   $OSDR + $DriversInfo         +
   $OSPI + $ProgramsInfo        +
   $OSGI + $GenRegSvcInfo       +
   $OSSI + $SecRegSvcInfo       |
       Out-File "$InfoFileName" -Encoding "OEM"

  If ($Null -ne $BatteryInfo) {
    $OSBS = "--- Battery Status    ---"  | Out-String
    $OSBS + $BatteryInfo  |
       Out-File "$InfoFileName" -Encoding "OEM" -Append
  }
#>
} #End ---------------  Write-ComputerInfoFile  ---------------

<#
  +-----------------------------------------------------------+
  |                        Main Program                       |
  +-----------------------------------------------------------+
#>
#Requires -Version 4

$RunStartTime = Get-Date
$PGMVersion   = 5.12     #--- Remember to Update! ----
$OStrWidth    = 80       #--- Out-String Width    ----
$ScriptName   = $MyInvocation.MyCommand.Name

Clear-Host

If ($Setup.IsPresent) {
  Install-Program
  Exit
}

$CompName = $Env:Computername

  $BTArgs = @{Text    = "Checking Local Machine Setup"
              Title   = "I'm working on $Env:COMPUTERNAME"
              Icon    = 'Info'
              Timeout = [Int32]500}
  Show-BalloonTip  @BTArgs

Add-Type -AssemblyName "System.Windows.Forms"

$TermMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
                     "CMsLocalPCInfoW10 - Program Terminated:",
  [Windows.Forms.MessageBoxButtons]::OK ,
  [Windows.Forms.MessageBoxIcon]::Information)}

If ($host.Name -eq 'ConsoleHost' -or
    $host.Name -eq 'Visual Studio Code Host') {

  Try{  <#+------------------------------------------------+
          | Check that the proper assemblies are loaded    |
          | Required for PS Console and Visual Code, while |
          | only Systems.Windows.Forms needed for PSISE!   |
          +------------------------------------------------+
        #>
    $ATArgs = @{AssemblyName = "PresentationCore",
                               "PresentationFramework",
                               "WindowsBase"
                ErrorAction = 'Stop'}
    Add-Type @ATArgs
  }
  Catch {
    $Message =
        "Failed to load Windows Presentation Framework" +
        " and/or other assemblies required for this program!"
     $Null = & $TermMsg
    Exit
  }

} #End If ($host...

#---   Check Local (Client) Windows Version   ---
If (-not($(ValidateWinVer -MajorVer 6 -MinorVer 1)[1])) {
  $Message =
      "You are running Windows prior to version 7:`n`n" +
      "This program REQUIRES Windows 7 or later to run!"
  $Null = & $TermMsg
  Exit
}

If ( (Get-AdminStatus) -eq "User") {
  $AdminPriv = $False
}
Else {
  $AdminPriv = $True
}

If ($PSVersionTable.PSVersion.Major -lt 4) {

  $Message = "You are running PowerShell version: " +
             "$($PSVersionTable.PSVersion)`n`n" +
             "The program requires Version 4.0 to run.`n`n" +
             "Please upgrade your PowerShell."
  $Null = & $TermMsg
  Exit
} #End If ($PSVersionTable...

#Check if Windows Remote Management (WinRM) is Running.
$StopWinRM = $False

If (( Get-Service -Name WinRM).Status -ne "Running") {

  Try {
        $SSArgs = @{Name        = "WinRM"
                    Status      = "Running"
                    ErrorAction = "Stop"
                  }
        Set-Service @SSArgs
        $StopWinRM = $True
  }
  Catch {
    $Message = "Error: Windows Remote Management service " +
               "is not running`nand failed to start when " +
               "an attempt was made.`n`nPlease correct "   +
               "this problem and rerun the program."  
    If ( -not $AdminPriv ) {
      $Message += "`n`nTry running as Administrator!"
    }  
                        
    $Null = & $TermMsg
    Exit
  }  #End Catch

} #End If (( Get-Service -Name WinRM).Status

# Create Computer Information Table (CIT)
$CITable =
New-Object -TypeName System.Data.DataTable "Comp Info"
#Create Columns for table
$CITItem  =
 New-Object -TypeName System.Data.DataColumn Item,([string])
$CITValue =
 New-Object -TypeName System.Data.DataColumn Value,([string])
#Add Columns to table
$CITable.columns.add($CITItem)
$CITable.columns.add($CITValue)

#--------- Create FileSpec for the computer's info ------
$FormattedDate = (Get-Date -f "yyyyMMdd")
$InfoFileName =
  "$([environment]::getfolderpath("mydocuments"))\" +
  "$($ScriptName.split(".")[0]) v" + "$PGMVersion" +
  " Using PS v" + $(($PSVersionTable).PSVersion.Major) +
  " as of $FormattedDate "          +
  "for $CompName.txt"

$LocalDTFmt = (Get-Culture).DateTimeFormat.ShortDatePattern
#To get times in local format delete "HH:mm" # below!
$LocalTMFmt = "HH:mm" #(Get-Culture).DateTimeFormat.ShortTimePattern

$CompSysObj = Get-CimInstance -ClassName 'Win32_ComputerSystem'

$HardwareInfo      = HardwareTab
$ProcessorInfo     = ProcessorTab
$MemoryInfo        = MemoryTab
$VideoInfo         = VideoTab
$MonitorInfo       = MonitorTab
$AudioInfo         = AudioTab
$PhysicalDiskInfo  = PhysicalDiskTab
$PartitionInfo     = PartitionTab
$LogicalDiskInfo   = LogicalDiskTab
$CDDVDInfo         = CDDVDTab
$FileSystemInfo    = FileSystemTab
$MapsAndShares     = MapsAndSharesTab
$EncryptInfo       = Get-EncryptStatus
$InternetInfo      = InternetTab
$ActiveAdapterInfo = Get-IPParams -Index $ActiveConnectionIndex
$WiFiInfo          = Get-WiFi
$LocalNetworkInfo  = LocalNetworkTab
$NetAdaptersInfo   = NetworkAdaptersTab
$PrinterInfo       = PrinterTab
$UserInfo          = UsersTab
$WindowsInfo       = WindowsTab
$PowerShellInfo    = PowerShellDotNetTab
$SecurityInfo      = SecurityTab
$WindowsUpd        = WinUpdSettings
$WinUpdateHistory  = Get-W10UpdateHistory
$WinUpgradeHistory = Get-W10UpgradeHistory
$WinDefInfo        = WindowsDefender
$EnvironInfo       = EnvironmentTab
$DriversInfo       = DriversTab
$ProgramsInfo      = ProgramsTab
$GenRegSvcInfo     = GenRegSvcTab
$SecRegSvcInfo     = SecRegSvcTab
$BatteryInfo       = BatteryTab

$RunTime = ((Get-Date) - $RunStartTIme)

GenerateForm

#Cleanup
If ($StopWinRM -eq $True) {
  Try {
       $SSArgs = @{Name        = "WinRM"
                   Confirm     = $False
                   Force       = $True
                   ErrorAction = "Stop"}
       Stop-Service @SSArgs  }
  Catch {
    $Message = "Error: Windows Remote Management`n" +
               "was not running at start of program`n" +
               "at start of program. Service was Started`n" +
               "by the program.`nProgram was unable to " +
               "stop the process at completion."
    $Null = & $TermMsg
  }
} #End If ($StopWinRM...

#  ------------------ End Main Program ------------------------