﻿<#
  .Synopsis
   List the restore points on the system.

  .Description
   List the restore points on the system.


  .Parameter rpType
   Type of restore point you wish to have returned or * for all.
   If you use this parameter the interactive selection screen
   will not display.

  .Outputs
   Dialog box displaying the matching restore points on the
   computer.

  .Notes
     Programmer   : RetiredGeek (@askWoody.com) aka: ComputerMentor
     Created      : 29 Jul 2011
     Last Updated : 30 Apr 2020
     Current Vers : 2.1

     In the normal output for Get-ComputerRestorePoint:
       Type 0 shows as APPLICATION_INSTALL
       Type 1 shows as APPLICATION_UNINSTALL
     You MUST use the numbers for comparision!
     Using an UNTyped param allows the use of the *
     for the All option

  .Example

   [d:\path\]Get-RestorePoints.ps1

   This will display a dialog showing they types of restore
   points and allow the user to enter the number for the type
   they with to see in the output dialog.

  .Example

   [d:\path\]Get-RestorePoints.ps1 -rpType 12
                 -or-
   [d:\path\]Get-RestorePoints.ps1  12

   With this command the dialog will not be displayed and only
   Modify Settings restore points will be returned in the
   output dialog.

#>


# Two blank linea above required for Comment-Based Help

param( 
       [Parameter(Mandatory=$False,Position=0)]
         [String] $rpType = ""
     )

Function Show-Form1 {

#--- Form Drawing References ---

  $Form_Width       =  450 #950
  $Form_Height      =  440 #605
  $Btn_Height       =   35
  $Btn_Width        =   80
  $tbox_Height      =  300
  $tbox_Width       =  200

  $WindowTitle = "CM's Get-RestorePoints " +
                 " - Version: {0:00.00}" -f $PGMVersion

[xml]$xaml = @"
<Window
  xmlns =
    "http://schemas.microsoft.com/winfx/2006/xaml/presentation"
  xmlns:x  = "http://schemas.microsoft.com/winfx/2006/xaml"
    Name = "Window"
              Title                 = "$WindowTitle"
              WindowStartupLocation = "CenterScreen"
              Width                 = "$Form_Width"
              Height                = "$Form_Height"
              FontSize              = "18"
              Background            = "Blue"
              ShowInTaskbar         = "True">

    <Window.Resources>

      <Style x:Key="BtnStyle" TargetType="Button">
       <Setter Property="Height"        Value="$Btn_Height"/>
       <Setter Property="Width"         Value="$Btn_Width" />
       <Setter Property="Canvas.Bottom" Value="2"          />
       <Setter Property="Background"    Value="Blue"       />
       <Setter Property="Foreground"    Value="Yellow"     />
       <Setter Property="Margin"        Value="5"          />
       <Setter Property="FontSize"      Value="20"         />
      </Style>

      <Style x:Key="TBoxStyle" TargetType="TextBox">
       <Setter Property="Height"     Value="$tbox_Height"   />
       <Setter Property="Width"      Value="$tbox_Width"    />
       <Setter Property="Margin"     Value="5"              />
       <Setter Property="Background" Value="Green"          />
       <Setter Property="Foreground" Value="White"          />
       <Setter Property="FontFamily" Value="Courier New"    />
       <Setter Property="FontSize"   Value="20"             />
       <Setter Property="IsReadOnly" Value="$False"         />
       <Setter Property="BorderThickness" Value ="0"        />
       <Setter Property="Margin"     Value="10,10,0,0"      />
       <Setter Property="VerticalScrollBarVisibility"
                                                Value="Auto"/>
      </Style>

      <Style x:Key="TBlockStyle" TargetType="TextBlock">
       <Setter Property="Height"     Value="35"             />
       <Setter Property="Width"      Value="$tbox_Width"    />
       <Setter Property="VerticalAlignment"   Value="Top"   />
       <Setter Property="HorizontalAlignment" Value="Left"  />
       <Setter Property="Background"   Value="Blue"         />
       <Setter Property="Foreground"   Value="White"        />
       <Setter Property="FontFamily"   Value="Courier New"  />
       <Setter Property="FontSize"     Value="20"           />
       <Setter Property="TextWrapping" Value="Wrap"         />
       <Setter Property="Margin"     Value="10,10,0,0"      />
      </Style>

    </Window.Resources>

    <Canvas>

         <TextBlock x:Name   = "tblockPrompt"
                      Style  = "{StaticResource TBlockStyle}"
                      Width  = "400"
                      Height = "260"
                      HorizontalAlignment = "Left" />

         <TextBlock x:Name   = "tblockPrompt2"
                      Style  = "{StaticResource TBlockStyle}"
                      Text   = "$Prompt"
                      Width  = "360"
                      Height = "260"
                      HorizontalAlignment = "Left"
                      Margin              =  "10,260,0,0" />

         <TextBox   x:Name     = "tboxInput"
                      Style    = "{StaticResource TBoxStyle}"
                      Width    = "40"
                      Height   = "35"
                      FontSize = "30"
                      HorizontalAlignment = "Left"
                      Margin              =  "385,280,0,0" />

        <Button x:Name    = "RstPtsBtn"
                  Content = "List Restore Points"
                  Style   = "{StaticResource BtnStyle}"
                  Width         = "200"
                  Background    = "Green"
                  Canvas.Left   =  "5"
                  Canvas.Bottom = "10" />

        <Button x:Name    = "ExitBtn"
                  Content = "Exit"
                  Style   = "{StaticResource BtnStyle}"
                  Width         = "80"
                  Background    = "Red"
                  Canvas.Right  =  "5" 
                  Canvas.Bottom = "10" />

    </Canvas>
</Window>
"@

$NOArgs = @{TypeName     = 'System.Xml.XmlNodeReader'
            ArgumentList = $xaml}
$reader=(New-Object @NOArgs)

Try  {$Form=[Windows.Markup.XamlReader]::Load( $reader )}
Catch{ $ErrMsg = ("Unable to load Windows.Markup.XamlReader." +
                 "`nDouble-check syntax and ensure " +
                 ".net is installed.") | Out-String
       Write-Host "$ErrMsg"

     }

$Form.Topmost = $False
$Form.Cursor  = [System.Windows.Input.Cursors]::Hand

#Connect Checkboxes


$tblockPrompt = $Form.FindName("tblockPrompt")
$tblockPrompt.AddText("$DisplayTypes")

$tboxInput = $Form.FindName("tboxInput")

$ExitBtn = $Form.FindName("ExitBtn")
$ExitBtn.Add_Click({$Script:rpType = "-1"
                    $Form.Close()})

$RstPtsBtn = $Form.FindName("RstPtsBtn")
$RstPtsBtn.Add_Click({ $Script:RpType = $tboxInput.Text
                       $Form.Close()})

#Mandatory last line of every script to load form
$Null = $Form.ShowDialog()

} #End --------------------- Show-Form1 ---------------------

Function Show-Form2 {

#--- Form Drawing References ---

  $Window_Width = 900
  $WindowTitle = "CM's Get-RestorePoints " +
                 " - Version: {0:00.00}" -f $PGMVersion

[xml]$xaml = @"
<Window
  xmlns =
    "http://schemas.microsoft.com/winfx/2006/xaml/presentation"
  xmlns:x  = "http://schemas.microsoft.com/winfx/2006/xaml"
    Name = "Window"
              Title                 = "$WindowTitle"
              WindowStartupLocation = "CenterScreen"
              Width                 = "$($Window_Width)"
              Height                = "500"
              FontSize              = "18"
              Background            = "Blue"
              ShowInTaskbar         = "True">

    <Canvas>

         <TextBlock x:Name                = "tblockResults"
                      Width               = "$($Window_Width - 20)"
                      Height              = "300"
                      Background          = "Blue"
                      Foreground          = "White"
                      FontSize            = "20"
                      FontFamily          = "Courier New"
                      HorizontalAlignment = "Left"
                      Margin              = "10,0,0,0" />


        <Button x:Name          = "ExitBtn"
                  Content       = "Exit"
                  Width         = "80"
                  Height        = "35"
                  Background    = "Red"
                  Foreground    = "Yellow"
                  FontSize      = "20"
                  Canvas.Right  = "5"
                  Canvas.Bottom = "5" />

    </Canvas>
</Window>
"@

$NOArgs = @{TypeName     = 'System.Xml.XmlNodeReader'
            ArgumentList = $xaml}
$reader=(New-Object @NOArgs)

Try  {$Form=[Windows.Markup.XamlReader]::Load( $reader )}
Catch{ $ErrMsg = ("Unable to load Windows.Markup.XamlReader." +
                 "`nDouble-check syntax and ensure " +
                 ".net is installed.") | Out-String
       Write-Host "$ErrMsg"
     }

$Form.Topmost = $False
$Form.Cursor  = [System.Windows.Input.Cursors]::Hand

#Connect Checkboxes

$tblockResults = $Form.FindName("tblockResults")
$tblockResults.AddText("$Message")

$ExitBtn = $Form.FindName("ExitBtn")
$ExitBtn.Add_Click({$Form.Close()})

#Mandatory last line of every script to load form
$Null = $Form.ShowDialog()

} #End --------------------- Show-Form2 ---------------------

#------------------- Main Program ------------------------------

Clear-Host

$PGMVersion = 2.1

Add-Type -AssemblyName "System.Windows.Forms"

$TermMsg = {
  [Windows.Forms.MessageBox]::Show($Message,$Title,
  [Windows.Forms.MessageBoxButtons]::OK ,
  [Windows.Forms.MessageBoxIcon]::Information)}

If ($PSVersionTable.PSVersion.Major -gt 5) {

  $Message = "You are running PowerShell version: " +
             "$($PSVersionTable.PSVersion)`n`n" +
             "The program requires Version 5.1 or less to run."
  $Title = "Get-RestorePoints.ps1 Terminated:"
  $Null = & $TermMsg

  Exit

} #End If ($PSVersionTable...


If ($host.Name -eq 'ConsoleHost' -or
    $host.Name -eq 'Visual Studio Code Host') {

  Try{  <#+------------------------------------------------+
          | Check that the proper assemblies are loaded    |
          | Required for PS Console and Visual Code, while |
          | only Systems.Windows.Forms needed for PSISE!   |
          +------------------------------------------------+
        #>
    $ATArgs = @{AssemblyName = "PresentationCore",
                               "PresentationFramework",
                               "WindowsBase"
                ErrorAction = 'Stop'}
    Add-Type @ATArgs
  }
  Catch {
    $Title   = "CMs List Restore Points - Program Terminated:"
    $Message =
        "Failed to load Windows Presentation Framework" +
        " and/or other assemblies required for this program!"
     $Null = & $TermMsg
    Exit
  }

} #End If ($host...

$RstPtTypes =  [ordered]@{ '0' = "Installation"
                           '1' = "Uninstall"
                           '6' = "System Restore"
                          '10' = "Device Driver Inst"
                          '12' = "Modify Settings"
                          '13' = "Cancelled Operation"
                          '16' = "Manually Created"
                          '18' = "Windows Update"
                         }

If ($rpType -eq "") {

  $fmtHash = @{Expression={$_.Key};
                     Label="Type";align='right';Width=4},
             @{Expression={$_.Value};
                     Label="Restore Point Type";Width=20}


  $DisplayTypes = $RstPtTypes|
                  Format-Table -Property $fmtHash | Out-String
  $DisplayTypes = $DisplayTypes.Substring(2,270)

  $prompt = "Enter the Type number for the desired`n" +
            "Restore Point, leave blank for all."

  Show-Form1

  If ($rpType -eq -1) {Exit}
}

$fmtRstPts =
    @{Expression={$_.SequenceNumber};
                   Label="`n#";align='right';Width=2},
    @{Expression={$($_.ConvertToDateTime($_.CreationTime)
                       ).ToShortDateString()};
                   Label="Creation`nDate";Width=10},
    @{Expression={$($_.ConvertToDateTime($_.CreationTime)
                       ).ToShortTimeString()};
                   Label="Creation`nTime";Width=10},
    @{Expression={$RstPtTypes[[String]$($_.RestorePointType)]};
                   Label="`nRestore Point Type  ";Width=20},
    @{Expression={$_.Description};
                   Label="`nDescription";Width=26}

If ($rpType -eq "") {
  $RstPts = Get-ComputerRestorePoint
}
Else {
    $RstPts = Get-ComputerRestorePoint |
      Where-Object {$_.RestorePointType -eq [int] $rpType}
}

    $Title   = "CMs List Restore Points - Results:"
    $Message = $RstPts | Sort-Object SequenceNumber |
                         Format-Table -Property $FmtRstPts -wrap |
                         Out-String
Show-Form2
