﻿<#
.Synopsis
   Return the Windows 10 Version and Update information

.Description
   Displays a tabbed dialog box that contains the windows 10
   verion, update, and cumulative update information.
   With links to the Release and Build information.
   The data is also copied to the clipboard for posting
   into help sites like AskWoody.com

.Outputs
    Dialog Box
    Data copied to clipboard

.Notes
     Programmer   : RetiredGeek (WSL) aka: ComputerMentor
     Created      : 11 Mar 2019
     Last Updated : 28 May 2020 
     Current Vers : 4.5
     Credits      :
       *** Originally based on Auto Hot Key Script ***
       *** by Rick Corbett                         ***

.Example
   [d:\path\]Get-WinVer.ps1

#>


# Two blank linea above required for Comment-Based Help

Function Show-Dialog {

$Form_Width    = 970
$Form_Height   = 500
$Btn_Height    =  25
$Btn_Width     = 370
$StdSpacing    =  35
$tabWidth      = $Form_Width  - 10
$tabHeight     = $Form_Height - 80
$tbox_Width    = $tabWidth    - 25 #leave room for scroll bar
$tbox_Height   = $tabHeight   - 10

$WindowTitle =
   "Computer Mentor's Windows 10 Version Info: Ver. " +
   "{0:00.00} Running with {1} privledges" -f $PgmVers ,
                                              $(Get-AdminStatus)

[xml]$xaml = @"
<Window
  xmlns =
    "http://schemas.microsoft.com/winfx/2006/xaml/presentation"
  xmlns:x  = "http://schemas.microsoft.com/winfx/2006/xaml"
    Name = "Window"
              Title         = "$WindowTitle"
              WindowStartupLocation = "CenterScreen"
              Width         = "$Form_Width"
              Height        = "$Form_Height"
              FontSize      = "18"
              Background="Blue" Foreground="White"
              ShowInTaskbar = "True">

    <Window.Resources>

      <Style x:Key="BtnStyle" TargetType="Button">
       <Setter Property="Height"        Value="$Btn_Height"/>
       <Setter Property="Width"         Value="$Btn_Width" />
       <Setter Property="Canvas.Bottom" Value="2"          />
       <Setter Property="Background"    Value="Blue"       />
       <Setter Property="Foreground"    Value="Yellow"     />
       <Setter Property="Margin"        Value="5"          />
       <Setter Property="FontSize"      Value="20"         />
      </Style>

      <Style x:Key="TBoxStyle" TargetType="TextBox">
       <Setter Property="Height"     Value="$tbox_Height"   />
       <Setter Property="Width"      Value="$tbox_Width"    />
       <Setter Property="Margin"     Value="5"              />
       <Setter Property="Background" Value="Blue"           />
       <Setter Property="Foreground" Value="White"          />
       <Setter Property="FontFamily" Value="Courier New"    />
       <Setter Property="FontSize"   Value="20"             />
       <Setter Property="BorderThickness" Value ="0"        />
       <Setter Property="IsReadOnly" Value="$True"          />
       <Setter Property="Margin"     Value="10,10,5,0"      />
       <Setter Property="VerticalScrollBarVisibility"
                                                Value="Auto"/>
      </Style>

    </Window.Resources>

    <Canvas>
     <TabControl Name="W10Info"
                 Canvas.Top="0" Canvas.Left="0"
                 Width="$tabWidth" Height="$tabHeight"
                 BorderThickness="0"
                 Background="Blue" Foreground="White">

      <TabItem   Header="Version">
       <Grid>
         <TextBox   x:Name="tboxVerInfo"
                    Style="{StaticResource TBoxStyle}"
                    Width="400"
                    HorizontalAlignment="Left" />
         <Button x:Name="VersionInfoBtn"
                 Content='Click here for Version Info'
                 Height="$Btn_Height" Width="$Btn_Width"
                 Background="Green" Foreground="Yellow"
                 ToolTip = "Version Info"
                 HorizontalAlignment="Right"
                 Margin="10,-80,15,0" />
         <Button x:Name="BuildInfoBtn"
                 Content='Click here for Release Info'
                 Height="$Btn_Height" Width="$Btn_Width"
                 Background="Green" Foreground="Yellow"
                 ToolTip = "Release Info"
                 HorizontalAlignment="Right"
                 Margin="10,70,15,0" />
       </Grid>
      </TabItem>

      <TabItem   Header="Version Updates">
       <TextBox  x:Name="VersionUpdates"
                 Style="{StaticResource TBoxStyle}" />
      </TabItem>

      <TabItem   Header="Monthly Updates">
       <TextBox  x:Name="InterimUpdates"
                 Style="{StaticResource TBoxStyle}" />
      </TabItem>

     </TabControl>

     <Button x:Name="ExitBtn" Content='Exit'
             Height="$Btn_Height" Width="80"
             Canvas.Bottom="5" Canvas.Right="5"
             Background="Red" Foreground="Yellow"
             ToolTip = "Exit"
             Margin="5" />
    </Canvas>
</Window>
"@

$NOArgs = @{TypeName = 'System.Xml.XmlNodeReader'
            ArgumentList = $xaml}
$reader=(New-Object @NOArgs)

Try  {$Form=[Windows.Markup.XamlReader]::Load( $reader )}
Catch{Write-Host $("Unable to load Windows.Markup.XamlReader." +
                 "`nDouble-check syntax and ensure " +
                 ".net is installed.")
     }

$Form.Topmost = $False
$Form.Cursor = [System.Windows.Input.Cursors]::Hand

$tboxVerInfo = $Form.FindName("tboxVerInfo")
$tboxVerInfo.AddText($VersionMsg)

$VersionUpdates = $Form.FindName("VersionUpdates")
$VersionUpdates.AddText($W10UpdHist)

$InterimUpdates = $Form.FindName("InterimUpdates")
$InterimUpdates.AddText($MonthlyUpdates)

$VersionInfoBtn = $Form.FindName("VersionInfoBtn")
$VersionInfoBtn.Add_Click({
    [system.Diagnostics.Process]::start(
       "https://support.microsoft.com/en-us/help/$VerURL" +
       "/windows-10-update-history")
    $VersionInfoBtn.Foreground="Purple"})

$BuildInfoBtn = $Form.FindName("BuildInfoBtn")
$BuildInfoBtn.Add_Click({
    [system.Diagnostics.Process]::start(
       "https://support.microsoft.com/en-us/help/$BuildURL" +
       "/windows-10-update-kb$BuildURL")
    $BuildInfoBtn.Foreground="Purple"})

$ExitBtn = $Form.FindName("ExitBtn")
$ExitBtn.Add_Click({$Form.Close()})

#Mandatory last line of every script to load form
$Form.ShowDialog()

} #End Function Show-Dialog()

Function Get-VersionURL {

  $verURL = (& { Switch ($ReleaseID) {
                  '1511' { '4000824' }
                  '1607' { '4000825' }
                  '1703' { '4018124' }
                  '1709' { '4043454' }
                  '1803' { '4099479' }
                  '1809' { '4464619' }
                  '1903' { '4498140' }
                  '1909' { '4529964' }
                  '2004' { '4556803' }
                 }
                } )

  "$VerURL"

} #End Get-SupportURL

Function Get-AdminStatus {

  $identity =
     [Security.Principal.WindowsIdentity]::GetCurrent()
  $principal =
     New-Object Security.Principal.WindowsPrincipal $identity

  If (-NOT ($principal.IsInRole(
      [Security.Principal.WindowsBuiltinRole]::Administrator)))
    {"User"}
  Else
    {"Administrator"}

} # -----------  End Function Get-AdminStatus -----------------


Function Get-BuildURL {

#Update Info @: https://support.microsoft.com/en-us/help/4464619

  $Build = "$BuildNo.$ReleaseNo"

  $BuildURL = (& { Switch ($Build) {
                  '10586.42'   { '3163018' } #1511 9/30/19
                  '10586.104'  { '3135173' }
                  '10586.122'  { '3140743' }
                  '10586.154'  { '4093109' }
                  '10586.164'  { '3140768' }
                  '10586.218'  { '3147458' }
                  '10586.318'  { '3156421' }
                  '10586.494'  { '3172985' }
                  '10586.545'  { '3176493' }
                  '10586.589'  { '3185614' }
                  '10586.633'  { '3192441' }
                  '10586.679'  { '3198586' }
                  '10586.682'  { '3198586' }
                  '10586.713'  { '3205386' }
                  '10586.753'  { '3210721' }
                  '10586.839'  { '4013198' }
                  '10586.842'  { '4016636' }
                  '10586.873'  { '4015219' }
                  '10586.916'  { '4019473' }
                  '10586.962'  { '4022714' }
                  '10586.965'  { '4032693' }
                  '10586.1007' { '4025344' }
                  '10586.1045' { '4054660' }
                  '10586.1106' { '4038783' }
                  '10586.1176' { '4041689' }
                  '10586.1177' { '4052232' }
                  '10586.1232' { '4048952' }
                  '10586.1295' { '4053578' }
                  '10586.1356' { '4056888' }
                  '10586.1358' { '4075200' }
                  '10586.1417' { '4074591' }
                  '10586.1478' { '4088779' }
                  '10586.1540' { '4093109' } #No Longer Supported!
                  '14393.1'    { '3176929' } #1607 05/12/20
                  '14393.51'   { '3176495' }
                  '14393.82'   { '3176934' }
                  '14393.97'   { '4016635' }
                  '14393.105'  { '3176938' }
                  '14393.123'  { '4023680' }
                  '14393.148'  { '4025339' }
                  '14393.167'  { '4039396' }
                  '14393.177'  { '4041691' }
                  '14393.187'  { '3193494' }
                  '14393.189'  { '3193494' }
                  '14393.222'  { '3194496' }
                  '14393.243'  { '4343887' }
                  '14393.258'  { '4462928' }
                  '14393.321'  { '3194798' }
                  '14393.351'  { '3197954' }
                  '14393.447'  { '3200970' }
                  '14393.448'  { '3200970' }
                  '14393.479'  { '3201845' }
                  '14393.576'  { '3206632' }
                  '14393.693'  { '3213986' }
                  '14393.726'  { '3216755' }
                  '14393.729'  { '4010672' }
                  '14393.953'  { '4013429' }
                  '14393.969'  { '4015438' }
                  '14393.1066' { '4015217' }
                  '14393.1083' { '4015217' }
                  '14393.1198' { '4019472' }
                  '14393.1358' { '4022715' }
                  '14393.1378' { '4022723' }
                  '14393.1532' { '4025334' }
                  '14393.1537' { '4038220' }
                  '14393.1593' { '4034658' }
                  '14393.1613' { '4034661' }
                  '14393.1715' { '4038782' }
                  '14393.1737' { '4038801' }
                  '14393.1794' { '4041688' }
                  '14393.1797' { '4052231' }
                  '14393.1884' { '4048953' }
                  '14393.1914' { '4051033' }
                  '14393.1944' { '4053579' }
                  '14393.2007' { '4056890' }
                  '14393.2034' { '4057142' }
                  '14393.2068' { '4074590' }
                  '14393.2097' { '4077525' }
                  '14393.2125' { '4088787' }
                  '14393.2126' { '4088787' }
                  '14393.2155' { '4088889' }
                  '14393.2156' { '4096309' }
                  '14393.2189' { '4093119' }
                  '14393.2214' { '4093120' }
                  '14393.2248' { '4103723' }
                  '14393.2273' { '4103720' }
                  '14393.2458' { '4457131' }
                  '14393.2312' { '4284880' }
                  '14393.2339' { '4284833' }
                  '14393.2363' { '4338814' }
                  '14393.2368' { '4345418' }
                  '14393.2395' { '4338822' }
                  '14393.2396' { '4346877' }
                  '14393.2457' { '4343884' }
                  '14393.2458' { '4457131' }
                  '14393.2515' { '4457127' }
                  '14393.2551' { '4462917' }
                  '14393.2608' { '4467691' }
                  '14393.2639' { '4467684' }
                  '14393.2641' { '4478877' }
                  '14393.2665' { '4471321' }
                  '14393.2670' { '4483229' }
                  '14393.2724' { '4480961' }
                  '14393.2759' { '4480977' }
                  '14393.2791' { '4487026' }
                  '14393.2828' { '4487006' }
                  '14393.2848' { '4489882' }
                  '14393.2879' { '4489889' }
                  '14393.2906' { '4493470' }
                  '14393.2941' { '4493473' }
                  '14393.2969' { '4494440' }
                  '14393.2972' { '4505052' }
                  '14393.2999' { '4499177' }
                  '14393.3025' { '4503267' }
                  '14393.3053' { '4503294' }
                  '14393.3056' { '4509475' }
                  '14393.3085' { '4507460' }
                  '14393.3115' { '4507459' }
                  '14393.3144' { '4512517' }
                  '14393.3181' { '4512495' }
                  '14393.3204' { '4516044' }
                  '14393.3206' { '4522010' }
                  '14393.3242' { '4516061' }
                  '14393.3243' { '4524152' }
                  '14393.3274' { '4519998' }
                  '14393.3300' { '4519979' }
                  '14393.3326' { '4525236' }
                  '14393.3384' { '4530689' }
                  '14393.3443' { '4534271' }
                  '14393.3504' { '4537764' }
                  '14393.3542' { '4537806' }
                  '14393.3564' { '4540670' }
                  '14393.3595' { '4550929' }
                  '14393.3630' { '4550929' }
                  '14393.3659' { '4550947' }
                  '14393.3686' { '4556813' }
                  '15063.675'  { '4049370' }  #1703 05/12/20
                  '15063.726'  { '4048954' }
                  '15063.728'  { '4048954' }
                  '15063.729'  { '4055254' }
                  '15063.786'  { '4053580' }
                  '15063.85'   { '4056891' }
                  '15063.877'  { '4057144' }
                  '15063.909'  { '4074592' }
                  '15063.936'  { '4092077' }
                  '15063.966'  { '4088782' }
                  '15063.968'  { '4088782' }
                  '15063.994'  { '4088891' }
                  '15063.1029' { '4093107' }
                  '15063.1058' { '4093117' }
                  '15063.1088' { '4103731' }
                  '15063.1112' { '4103722' }
                  '15063.1155' { '4284874' }
                  '15063.1182' { '4284830' }
                  '15063.1206' { '4338826' }
                  '15063.1209' { '4345419' }
                  '15063.1235' { '4338827' }
                  '15063.1266' { '4343885' }
                  '15063.1292' { '4343889' }
                  '15063.1324' { '4457138' }
                  '15063.1358' { '4457141' }
                  '15063.1387' { '4462937' }
                  '15063.1418' { '4462939' }
                  '15063.1446' { '4467696' }
                  '15063.1478' { '4467699' }
                  '15063.1506' { '4471327' }
                  '15063.1508' { '4483230' }
                  '15063.1563' { '4480973' }
                  '15063.1596' { '4480959' }
                  '15063.1631' { '4487020' }
                  '15063.1659' { '4487011' }
                  '15063.1689' { '4489871' }
                  '15063.1716' { '4489888' }
                  '15063.1746' { '4493474' }
                  '15063.1784' { '4493436' }
                  '15063.1785' { '4502112' }
                  '15063.1805' { '4499181' }
                  '15063.1808' { '4505055' }
                  '15063.1839' { '4499162' }
                  '15063.1868' { '4503279' }
                  '15063.1897' { '4503289' }
                  '15063.1898' { '4509476' }
                  '15063.1928' { '4507450' }
                  '15063.1955' { '4507467' }
                  '15063.1988' { '4512507' }
                  '15063.2021' { '4512474' }
                  '15063.2045' { '4516068' }
                  '15063.2046' { '4522011' }
                  '15063.2078' { '4516059' }
                  '15063.2079' { '4524151' }
                  '15063.2108' { '4520010' }
                  '15063.2172' { '4525245' }
                  '15063.2224' { '4530711' }
                  '15063.2254' { '4534296' }
                  '15063.2284' { '4537765' }
                  '15063.2313' { '4540705' }
                  '15063.2346' { '4550939' }
                  '15063.2375' { '4556804' }
                  '16299.19'   { '4043961' } #1709 05/12/20
                  '16299.64'   { '4048955' }
                  '16299.98'   { '4051963' }
                  '16299.125'  { '4054517' }
                  '16299.192'  { '4056892' }
                  '16299.201'  { '4073291' }
                  '16299.214'  { '4058258' }
                  '16299.248'  { '4074588' }
                  '16299.251'  { '4090913' }
                  '16299.309'  { '4088776' }
                  '16299.334'  { '4089848' }
                  '16299.371'  { '4093112' }
                  '16299.402'  { '4093105' }
                  '16299.431'  { '4103727' }
                  '16299.461'  { '4103714' }
                  '16299.492'  { '4284819' }
                  '16299.522'  { '4284822' }
                  '16299.547'  { '4338825' }
                  '16299.551'  { '4345420' }
                  '16299.579'  { '4338817' }
                  '16299.611'  { '4343897' }
                  '16299.637'  { '4343893' }
                  '16299.665'  { '4457142' }
                  '16299.666'  { '4464217' }
                  '16299.699'  { '4457136' }
                  '16299.726'  { '4462918' }
                  '16299.755'  { '4462932' }
                  '16299.785'  { '4467686' }
                  '16299.820'  { '4467681' }
                  '16299.846'  { '4471329' }
                  '16299.847'  { '4483232' }
                  '16299.904'  { '4480978' }
                  '16299.936'  { '4480967' }
                  '16299.967'  { '4486996' }
                  '16299.1004' { '4487021' }
                  '16299.1029' { '4489886' }
                  '16229.1059' { '4489890' }
                  '16229.1087' { '4493441' }
                  '16229.1127' { '4493440' }
                  '16229.1146' { '4499179' }
                  '16229.1150' { '4505062' }
                  '16229.1182' { '4499147' }
                  '16229.1217' { '4503284' }
                  '16229.1237' { '4503281' }
                  '16229.1239' { '4509477' }
                  '16229.1268' { '4507455' }
                  '16229.1296' { '4507465' }
                  '16229.1331' { '4512516' }
                  '16229.1365' { '4512494' }
                  '16229.1387' { '4516066' }
                  '16229.1392' { '4522012' }
                  '16229.1420' { '4516071' }
                  '16299.1421' { '4524150' }
                  '16299.1451' { '4520004' }
                  '16299.1481' { '4520006' }
                  '16299.1508' { '4525241' }
                  '16299.1565' { '4530714' }
                  '16229.1625' { '4534276' }
                  '16229.1654' { '4534318' }
                  '16229.1686' { '4537789' }
                  '16229.1717' { '4537816' }
                  '16229.1747' { '4540681' }
                  '16229.1775' { '4541339' }
                  '16229.1776' { '4554342' }
                  '16229.1868' { '4556812' }
                  '17134.42'   { '4458469' } #1803 Updated 05/12/20
                  '17134.48'   { '4103721' }
                  '17134.81'   { '4100403' }
                  '17134.83'   { '4338548' }
                  '17134.112'  { '4284835' }
                  '17134.137'  { '4284848' }
                  '17134.165'  { '4338819' }
                  '17134.167'  { '4345421' }
                  '17134.191'  { '4340917' }
                  '17134.228'  { '4343909' }
                  '17134.254'  { '4346783' }
                  '17134.285'  { '4457128' }
                  '17134.286'  { '4464218' }
                  '17134.445'  { '4462919' }
                  '17134.476'  { '4462933' }
                  '17134.407'  { '4467702' }
				  '17134.441'  { '4467682' }
				  '17134.471'  { '4471324' }
				  '17134.472'  { '4483234' }
				  '17134.523'  { '4480966' }
				  '17134.556'  { '4480976' }
				  '17134.590'  { '4487017' }
				  '17134.619'  { '4487029' }
				  '17134.648'  { '4489868' }
				  '17134.677'  { '4489894' }
				  '17134.706'  { '4493437' }
				  '17134.753'  { '4493437' }
				  '17134.765'  { '4499167' }
				  '17134.766'  { '4505064' }
				  '17134.799'  { '4499183' }
				  '17134.829'  { '4503286' }
				  '17134.858'  { '4503288' }
				  '17134.860'  { '4509478' }
				  '17134.885'  { '4507435' }
				  '17134.915'  { '4507466' }
				  '17134.950'  { '4512501' }
				  '17134.984'  { '4512509' }
				  '17134.1006' { '4516058' }
				  '17134.1009' { '4522014' }
				  '17134.1039' { '4516045' }
                  '17134.1040' { '4524149' }
                  '17134.1069' { '4520008' }
                  '17134.1099' { '4519978' }
                  '17134.1130' { '4525237' }
                  '17134.1184' { '4530717' }
                  '17134.1246' { '4534293' }
                  '17134.1276' { '4534308' }
                  '17134.1304' { '4537762' }
                  '17134.1345' { '4537795' }
                  '17344.1365' { '4540689' }
                  '17134.1399' { '4541333' }
                  '17134.1401' { '4554349' }
                  '17134.1488' { '4556807' }
                  '17763.134'  { '4467708' } #1809 Updated 05/12/20
                  '17763.107'  { '4464455' }
                  '17763.55'   { '4464330' }
                  '17763.107'  { '4464455' }
                  '17763.134'  { '4467708' }
                  '17763.168'  { '4469342' }
                  '17763.194'  { '4471332' }
                  '17763.195'  { '4483235' }
                  '17763.253'  { '4480116' }
                  '17763.292'  { '4476976' }
                  '17763.316'  { '4487044' }
                  '17763.348'  { '4482887' }
                  '17763.379'  { '4489899' }
                  '17763.404'  { '4490481' }
                  '17763.437'  { '4493509' }
                  '17763.439'  { '4501835' }
                  '17763.475'  { '4495667' }
                  '17763.503'  { '4494441' }
                  '17763.504'  { '4505056' }
                  '17763.529'  { '4497934' }
                  '17763.557'  { '4503327' }
                  '17763.592'  { '4501371' }
                  '17763.593'  { '4509479' }
                  '17763.615'  { '4507469' }
                  '17763.652'  { '4505658' }
                  '17763.678'  { '4511553' }
                  '17763.720'  { '4512534' }
                  '17763.737'  { '4512578' }
                  '17763.740'  { '4522015' }
                  '17763.774'  { '4516077' }
                  '17763.775'  { '4524148' }
                  '17763.805'  { '4519338' }
                  '17763.832'  { '4520062' }
                  '17763.864'  { '4523205' }
                  '17763.914'  { '4530715' }
                  '17763.973'  { '4534273' }
                  '17763.1012' { '4534321' }
                  '17763.1075' { '4537818' }
                  '17763.1098' { '4538461' }
                  '17763.1158' { '4549949' }
                  '17763.1192' { '4550969' }
                  '17763.1217' { '4551853' }
                  '18362.145'  { '4497935' } #1903 Updated 05/12/20
                  '18362.175'  { '4503293' }
                  '18362.207'  { '4501375' }
                  '18362.239'  { '4507453' }
                  '18362.267'  { '4505903' }
                  '18362.295'  { '4512508' }
                  '18362.329'  { '4512941' }
                  '18362.356'  { '4515384' }
                  '18362.357'  { '4522016' }
                  '18362.387'  { '4517211' }
                  '18362.388'  { '4524147' }
                  '18362.481'  { '4517389' }
                  '18362.449'  { '4522355' }
                  '18362.476'  { '4524570' }
                  '18362.535'  { '4530684' }
                  '18362.592'  { '4528760' }
                  '18362.628'  { '4532695' }
                  '18362.693'  { '4535996' }
                  '18362.719'  { '4540673' }
                  '18362.720'  { '4551762' }
                  '18362.788'  { '4549951' }
                  '18362.815'  { '4550945' }
                  '18362.836'  { '4556799' }
                  '18363.476'  { '4524570' } #1909 Updated 05/12/20
                  '18363.535'  { '4530684' }
                  '18363.592'  { '4528760' }
                  '18363.628'  { '4532695' }
                  '18363.657'  { '4532693' }
                  '18363.693'  { '4535996' }
                  '18363.719'  { '4540673' }
                  '18363.720'  { '4551762' }
                  '18363.778'  { '4549951' }
                  '18363.815'  { '4550945' }
                  '18363.836'  { '4556799' }
                  '19041.264'  { '4536803' } #2004 Updated 05/27/20

                  Default {"$Build is not supported`n" +
                           "by this program!"}
                  }
                } )

  "$BuildURL"

} #End Get-BuildURL

Function Get-W10UpdateHistory {

<#
  .Synopsis
   Retrieve the Windows (minor) update history.

  .Description
   Retrieve the Windows (minor) update history.

  .Outputs
   String containing a Formatted and sorted listing the 
   Minor Update history with dates/time suitable for display
   in a TextBox control.

  .Notes
     Programmer   : RetiredGeek (@askWoody.com)
                    aka: ComputerMentor
     Created      : 20 Jan 2020
     Last Updated :
     Current Vers : 1.0

  .Example
   $Variable = Get-Win10UpdateHistory.ps1


#>


# Two blank linea above required for Comment-Based Help

 $HFFmt = @{Expression={
             ($_.InstalledOn).ToString("yyyy/MM/dd")};
            Label="Installed On";    Width=12},
          @{Expression={$_.HotFixID};
            Label="Hot Fix ID";      Width=10},
        @{Expression={$_.CSName};
           Label="Computer";         Width=14},
        @{Expression={$_.Description};
           Label="Description";      Width=35}

  $x = Get-HotFix

  $RetVal = $x |
    Sort-Object -Descending -Property InstalledOn |
    Format-Table $HFFmt | Out-String
  "$RetVal"

} #End Function Get-W10UpdateHistory

Function Get-W10UpgradeHistory {

<#
  .Synopsis
   Retrieve the Windows (major) update history.

  .Description
   Retrieve the Windows (major) update history.

  .Outputs
   String containing a Formatted and sorted listing the 
   Major Update history with dates/time suitable for display
   in a TextBox control.

  .Notes
     Programmer   : RetiredGeek (@askWoody.com)
                    aka: ComputerMentor
     Created      : 20 Jan 2020
     Last Updated :
     Current Vers : 1.0
  +-----------------------------------------------------------+
  |Note: Update Started shows the date/time that an update    |
  |      was initiated to the next version, the Vers is the   |
  |      currently installed version when the update was      |
  |      initiated, Vers. Installed shows the date/time when  |
  |      the Vers was installed.                              |
  +-----------------------------------------------------------+

  .Example
   $Variable = Get-Win10UpgradeHistory.ps1


#>


# Two blank linea above required for Comment-Based Help

$RegEx = "Updated\son\s(\d{1,2}\/\d{1,2}" +
         "\/\d{4}\s\d{2}:\d{2}:\d{2})\)$"

$OSUpgradeHistory = $(Get-ChildItem "HKLM:\System\Setup" |
  Where-Object {$_.Name -match "\\Source\s"})  |
  ForEach-Object { $_   |
     Select @{Name="UpdateTime";
         Expression={if ($_.Name -match "$RegEx") {
           [dateTime]::Parse($Matches[1],
           ([Globalization.CultureInfo]::CreateSpecificCulture(
          'en-US')))}}},
       @{Name="Release"    ;
         Expression={$_.GetValue("ReleaseID")   }},
       @{Name="Branch"     ;
         Expression={$_.GetValue("BuildBranch") }},
       @{Name="Build"      ;
         Expression={$_.GetValue("CurrentBuild")}},
       @{Name="ProductName";
         Expression={$_.GetValue("ProductName") }},
       @{Name="InstallTime";
         Expression={[datetime]::FromFileTime(
                            $_.GetValue("InstallTime"))}}

 } #End ForEach-Object...

 $UPGRFmt = @{Expression={
              '{0:yyyy/MM/dd hh:mm}' -f $_.UpdateTime};
              Label="`nUpdated";        Width=16},
            @{Expression={$_.Release};
              Label="From`nVers";       Width=4},
            @{Expression={$_.Branch};
              Label="`nBranch";         Width=22},
            @{Expression={$_.Build};
              Label="`nBuild";          Width=5},
            @{Expression={$_.ProductName};
              Label="`nProduct Name";   Width=14},
            @{Expression={
             Get-Date($_.InstallTime) -f "yyyy/MM/dd"};
              Label="Vers. Installed";  Width=10}
<#              ,
            @{Expression={Get-Date($_.InstallTime) -f "HH:mm"};
              Label="`nTime";           Width=5}
#>

  $RetVal = $OSupgradeHistory  |
    Sort-Object -Descending -Property UpdateTime |
    Format-Table $UPGRFmt | Out-String

  "$RetVal"

} #End Function Get-W10UpgradeHistory

Function Get-W10Version {

  $GIPArgs = @{Path = "Registry::HKLM\SOFTWARE\Microsoft\" +
                      "Windows NT\CurrentVersion"
              }

  $WinVer = Get-Item @GIPArgs

  $ProdName         = $WinVer.getvalue('ProductName')
  $Script:ReleaseID = $Winver.getvalue('ReleaseId')
  $Script:ReleaseNo = $winVer.getvalue('UBR')
  $Script:BuildNo   = $WinVer.getvalue('CurrentBuildNumber')

  $CurOS     =
    Get-CimInstance -ClassName 'Win32_OperatingSystem'
  $Bitness   = $CurOS.OSArchitecture
  $LocalDTFmt = (Get-Culture).DateTimeFormat.ShortDatePattern
  #To get times in local format delete "HH:mm" # below!
  $LocalTMFmt =
    "HH:mm" #(Get-Culture).DateTimeFormat.ShortTimePattern

  $Installed =
      (Get-Date ($CurOS.InstallDate) -format $LocalDTFmt) +
      " " +
      (Get-Date ($CurOS.InstallDate) -format $LocalTMFmt)

  $Message = "Edition:   $ProdName $Bitness`n`n" +
             "Version:   $ReleaseId`n`n" +
             "Build:     $BuildNo.$ReleaseNo`n`n" +
             "Installed: $Installed"

  $VersionMsg = $Message +
      "`n`n     ( The info above has been`n" +
      "       copied to Clipboard )" | Out-String

  #Copy Data to the Clipboard
  Set-Clipboard -Value  $Message

  "$VersionMsg"

} #End Function Get-W10Version

#--------------   Main Program ----------------

Clear-Host

$PgmVers = 4.5

#Load required libraries
$TermMsg = {
  [Windows.Forms.MessageBox]::Show($Message,
                                   "Program Terminated:",
  [Windows.Forms.MessageBoxButtons]::OK ,
  [Windows.Forms.MessageBoxIcon]::Information) | Out-Null}

Add-Type -AssemblyName "System.Windows.Forms"

If ($host.Name -eq 'ConsoleHost' -or
    $host.Name -eq 'Visual Studio Code Host') {

  try{  <#+------------------------------------------------+
          | Check that the proper assemblies are loaded    |
          | Required for PS Console and Visual Code, while |
          | only Systems.Windows.Forms needed for PSISE!   |
          +------------------------------------------------+
        #>
    $ATArgs = @{AssemblyName = "PresentationCore",
                               "PresentationFramework",
                               "WindowsBase"
                ErrorAction = 'Stop'}
    Add-Type @ATArgs
  }
  catch {
    $Message =
        "Failed to load Windows Presentation Framework" +
        " and/or other assemblies required for this program!"
     & $TermMsg
    Exit
  }

} #End If ($host...

$VersionMsg     = Get-W10Version
$VerURL         = Get-VersionURL
$BuildURL       = Get-BuildURL
$W10UpdHist     = Get-W10UpgradeHistory
$MonthlyUpdates = Get-W10UpdateHistory

$Null = Show-Dialog