﻿# Ensure script is running as Admin
if (-not ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator")) {
    Write-Host "Please run the Windows PowerShell ISE using 'Run as administrator'!" -ForegroundColor Red
    exit
}

# Define time range - current example is last 3 days; feel free to change
$startTime = (Get-Date).AddDays(-3)

# Explicitly define key logs to search; feel free to change 
$logNames = @("System", "Application", "Security", "Setup")

# Get events from specified logs (Levels: Critical, Error, Warning - feel free to change)
$events = @()
foreach ($log in $logNames) {
    try {
        $logEvents = Get-WinEvent -FilterHashtable @{
            LogName = $log
            Level = @(1, 2, 3)  # Critical (1), Error (2), Warning (3), Information (4), Verbose (5)
            StartTime = $startTime
        } -ErrorAction SilentlyContinue | Select-Object TimeCreated, LogName, Id, Level, ProviderName, Message

        if ($logEvents) {
            $events += $logEvents
        }
    } catch {
        Write-Host "Error accessing log: $log - $_" -ForegroundColor Yellow
    }
}

# Sort events by TimeCreated
$events = $events | Sort-Object TimeCreated

# Check if any events were found
if ($events.Count -eq 0) {
    Write-Host "No Critical, Error, or Warning events found in the last 3 days." -ForegroundColor Yellow
    exit
}

# Prompt for output choice
$choice = Read-Host "Please choose an output format: [1] Screen, [2] Grid, [3] Text File, [4] CSV File"

# Define file paths
$desktopPath = [System.Environment]::GetFolderPath("Desktop")
$txtFile = "$desktopPath\Problem_Finder_Report.txt"
$csvFile = "$desktopPath\Problem_Finder_Report.csv"

# Handle output choice
switch ($choice) {
    "1" {  # Display on screen
        $events | Format-Table -AutoSize
    }
    "2" {  # Show in grid view
        $events | Out-GridView -Title "Event Log Problem Finder Report"
    }
    "3" {  # Save as TXT
        $events | Format-Table -AutoSize | Out-File -FilePath $txtFile
        #New-BurntToastNotification -AppLogo C:\Scipts\check-mark_256x256x16.png -Text "Problem Finder", 'Your TXT file has been saved to your desktop.'
        Write-Host "Report saved as TXT file: $txtFile" -ForegroundColor Green
    }
    "4" {  # Save as CSV
        $events | Export-Csv -Path $csvFile -NoTypeInformation
        #New-BurntToastNotification -AppLogo C:\Scripts\tick-mark.png -Text "Problem Finder", 'Your CSV file has been saved to your desktop.'
        Write-Host "Report saved as CSV file: $csvFile" -ForegroundColor Green
    }
    default {
        #New-BurntToastNotification -AppLogo C:\Scripts\cross-mark.png -Text "Problem Finder", 'Invalid choice! Please re-run the script and choose a valid option.'
        Write-Host "Invalid choice! Please re-run the script and choose a valid option." -ForegroundColor Red
    }
}
