﻿# Define time range (last 3 days)
$startTime = (Get-Date).AddDays(-3)

# Define event log and source
$logName = "Application"
$providerName = "MsiInstaller"

# Try to retrieve the events
try {
    $events = Get-WinEvent -FilterHashtable @{
        LogName = $logName
        ProviderName = $providerName
        StartTime = $startTime
    } -ErrorAction SilentlyContinue | 
    Select-Object TimeCreated, LogName, Id, Level, ProviderName, Message |
    Sort-Object TimeCreated
} catch {
    Write-Host "Error retrieving event logs: $_" -ForegroundColor Red
    exit
}

# Check if any events were found
if ($events.Count -eq 0) {
    Write-Host "No MsiInstaller events found in the last 3 days." -ForegroundColor Yellow
    exit
}

# Prompt user for output choice
$choice = Read-Host "Choose output format: [1] Screen, [2] Grid, [3] Text File, [4] CSV File"

# Define file paths
$desktopPath = [System.Environment]::GetFolderPath("Desktop")
$txtFile = "$desktopPath\MsiInstallerEvents.txt"
$csvFile = "$desktopPath\MsiInstallerEvents.csv"

# Handle output choice
switch ($choice) {
    "1" {  # Display on screen
        $events | Format-Table -AutoSize
    }
    "2" {  # Show in grid view
        $events | Out-GridView -Title "MsiInstaller Event Log Report"
    }
    "3" {  # Save as TXT
        $events | Format-Table -AutoSize | Out-File -FilePath $txtFile
        Write-Host "Report saved as TXT file: $txtFile" -ForegroundColor Green
    }
    "4" {  # Save as CSV
        $events | Export-Csv -Path $csvFile -NoTypeInformation
        Write-Host "Report saved as CSV file: $csvFile" -ForegroundColor Green
    }
    default {
        Write-Host "Invalid choice! Please run the script again and choose a valid option." -ForegroundColor Red
    }
}
